/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

/**
 *  This file contains a set of macros, routines and classes to deal with
 *  system checking and logging facility.
 */

#ifndef TEPDIAGNOSTIC_HPP
  #define TEPDIAGNOSTIC_HPP

  #include <stdlib.h>
  #include <stdio.h>
  #include <sstream>
  #include <string>
  #include <iostream>

  #ifdef USETERRALIBLOG

    #include <TeException.h>
    #include <TeErrorLog.h>

    /**
     * Logs a message to stdout
     *
     * @param message Message to be logged.
    */
    #define PDIAGN_LOGMSG(message) \
      TeErroLog::instance().insert( LOG_MESSAGE, std::string( __DATE__ ) + \
      " - " + std::string( __TIME__ ) + " - " + std::string( __FILE__ ) + \
      ":" + std::string( __LINE__ ) + " : " + \
      TePDIAgnostic::to_string( message ) );

    /**
     * Logs a message to stderr
     *
     * @param message Message to be logged.
    */
    #define PDIAGN_LOGERR(message) \
      TeErroLog::instance().insert( UNKNOWN_ERROR_TYPE, \
      std::string( __DATE__ ) + \
      " - " + std::string( __TIME__ ) + " - " + std::string( __FILE__ ) + \
      ":" + std::string( __LINE__ ) + " : " + \
      TePDIAgnostic::to_string( message ) );

  #else // USETERRALIBLOG

    /**
    * Logs a message to stdout
    *
    * @param message Message to be logged.
    */
    #define PDIAGN_LOGMSG(message) \
      std::cout << std::endl << "Message :" \
        << " " << __DATE__ \
        << " " << __TIME__ \
        << " - " << __FILE__ \
        << ":" << __LINE__ \
        << " - " << TePDIAgnostic::to_string( message ) \
        << std::endl;

    /**
    * Logs a error message to stderr
    *
    * @param message Message to be logged.
    */
    #define PDIAGN_LOGERR(message) \
      std::cerr << std::endl << "Error :" \
        << " " << __DATE__ \
        << " " << __TIME__ \
        << " - " << __FILE__ \
        << ":" << __LINE__ \
        << " - " << TePDIAgnostic::to_string( message ) \
        << std::endl;

  #endif // USETERRALIBLOG

  /**
   * Logs a message to stderr and throws.
   *
   * @param variable Variable to be logged.
   */
  #define PDIAGN_WATCH( variable ) \
    { \
      PDIAGN_LOGMSG( "WATCH - " + TePDIAgnostic::to_string( #variable ) + \
      "=" + TePDIAgnostic::to_string( variable ) ); \
    }

  /**
   * Logs a message to stderr and throws.
   *
   * @param message Message to be logged.
   */
  #define PDIAGN_LOG_AND_THROW( message ) \
    { \
      PDIAGN_LOGERR( message ); \
      throw; \
    }

  /**
   *  Checks if value is true and throws an exception if not.
   *
   * @param value Value to be checked.
   * @param message Message to be logged.
   */
  #define PDIAGN_TRUE_OR_THROW(value,message) \
    if( value == 0 ) { \
      PDIAGN_LOG_AND_THROW( message ); \
    };

  /**
   *  Checks if value is true and logs an error message if not.
   *
   * @param value Value to be checked.
   * @param message Message to be logged.
   */
  #define PDIAGN_TRUE_OR_LOG(value,message) \
    if( value == 0 ) { \
      PDIAGN_LOGERR( message ); \
    };

  /**
   * Checks if value is true. if not a message will be logget and
   * a return of context with false value will be done.
   *
   * @param value Value to be checked.
   * @param message Message to be logged.
   */
  #define PDIAGN_TRUE_OR_RETURN(value,message) \
    if( value == 0 ) { \
      PDIAGN_LOGERR( message ); \
      return false; \
    };

  /**
   *  Checks if value is false and logs an error message if not.
   *
   * @param value Value to be checked.
   * @param message Message to be logged.
   */
  #define PDIAGN_FALSE_OR_LOG(value,message) \
    if( value != 0 ) { \
      PDIAGN_LOGERR( message ); \
    };

  /**
   *  Checks if two values are equal and throws an exception if not.
   *
   * @param value1 Value to be checked.
   * @param value2 Value to be checked.
   * @param message Message to be logged.
   */
  #define PDIAGN_CHECK_EQUAL(value1,value2,message) \
    PDIAGN_TRUE_OR_THROW( ( value1 == value2 ), \
    message + std::string( " [" ) + \
    TePDIAgnostic::to_string( value1 ) + "!=" + \
    TePDIAgnostic::to_string( value2 ) + "]" );

  /**
   *  Checks if two values are diferent and throws an exception if not.
   *
   * @param value1 Value to be checked.
   * @param value2 Value to be checked.
   * @param message Message to be logged.
   */
  #define PDIAGN_CHECK_NOTEQUAL(value1,value2,message) \
    PDIAGN_TRUE_OR_THROW( ( value1 != value2 ), \
    message + std::string( " [Values are equal]" ) );

  /**
   * Throws an exception for not implemented source.
   */
  #define PDIAGN_NOT_IMPLEMENTED \
    PDIAGN_LOG_AND_THROW( "Not Implemented." );

  /**
   *  Checks if Debug mode is enabled and throws an exception if not.
   */
  #ifdef NDEBUG
    #define PDIAGN_DEBUG_MODE_CHECK \
      PDIAGN_LOG_AND_THROW( "Debug Mode Disabled." );
  #else
    #define PDIAGN_DEBUG_MODE_CHECK {};
  #endif

  /**
   * Checks if value is true and throws an exception if not.
   * This macro will be disabled for non debug mode.
   *
   * @param value Value to be checked.
   * @param message Message to be logged.
   */
  #ifndef NDEBUG
    #define PDIAGN_PRE_CONDITION(value,message) \
      PDIAGN_TRUE_OR_THROW( value, message );
  #else
    #define PDIAGN_PRE_CONDITION(value,message) {};
  #endif

  /**
   * Checks if value is true and throws an exception if not.
   * This macro will be disabled for non debug mode.
   *
   * @param value Value to be checked.
   * @param message Message to be logged.
   */
  #define PDIAGN_POS_CONDITION(value,message) \
    PDIAGN_PRE_CONDITION(value,message);

  namespace TePDIAgnostic{

    /**
     * Data convertion to string.
     *
     * @param data Data to be converted.
     * @return The converted string.
     */
     template< class T >
     std::string to_string( const T& data )
     {
        std::stringstream temp_ss;
        temp_ss << data;
        return temp_ss.str();
     }

    /**
     * Redirects stdout and stderr to disk files.
     *
     * @param log_file_name stdout output file name.
     * @param error_log_file_name stderr output file name.
     */
     void redirect_log( char* log_file_name, char* error_log_file_name );

  };

#endif //TEPDIAGNOSTIC_HPP
