#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include "netconf.h"
#include <misc.h>
#include "nisconf.h"
#include "nisconf.m"
#include <userconf.h>
#include "../../paths.h"
#include <subsys.h>

static const char NIS[] = "nis";
static const char NIS_DOMAIN[] = "domain";
static const char NIS_SERVER[] = "server";
static const char NIS_UPDATE[] = "updater";

static HELP_FILE help_nis ("nisconf","nis");

static CONFIG_FILE f_ypconf (ETC_YP_CONF,help_nis
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,subsys_netclient);

/*
	Read the yp.conf file and extract the ypserver.
	The rest is optionnal kept in the tb object
*/
static void nis_readypconf (SSTRINGS *tb, SSTRING *server)
{
	FILE_CFG *fin = f_ypconf.fopen ("r");
	if (fin != NULL){
		char buf[1000];
		while (fgets(buf,sizeof(buf)-1,fin)!=NULL){
			strip_end (buf);
			char keyw[100];
			char *pt = str_copyword(keyw,buf,sizeof(keyw));
			if (strcmp(keyw,"ypserver")==0){
				if (server != NULL) server->copyword (pt);
			}else if (tb != NULL){
				tb->add (new SSTRING (buf));
			}
		}
		fclose (fin);
	}
}
/*
	Manage the file /etc/conf.nis
	The construtor read the file. A missing file means that NIS
	is not configured.
*/
PUBLIC NIS_CONF::NIS_CONF()
{
	domain.setfrom (linuxconf_getval(NIS,NIS_DOMAIN));
	nis_readypconf (NULL,&server);
	update = linuxconf_getvalnum (NIS,NIS_UPDATE,0);
}

/*
	Check that the server is valid.
	The NIS server specification is optionnal: If the field is
	empty, this is ok.

	Signal an error message if not ok.

	Return != 0 if ok.
*/
PUBLIC int NIS_CONF::valid_server()
{
	int ret = 0;
	if (server.is_filled()){
		const char *serverp = server.get();
		if (net_isipnum(serverp)){
			ret = 1;
		}else if (gethostbyname(serverp) == NULL){
			xconf_error (MSG_U(E_NISSERV,"Unknown NIS server %s\n"),serverp);
		}else{
			ret = 1;
		}
	}else{
		ret = 1;
	}
	return ret;
}

/*
	Tells if NIS is properly configured
*/
PUBLIC int NIS_CONF::configok()
{
	int ret = 0;
	if (domain.is_filled()){
		ret = valid_server();
	}
	return ret;
}


/*
	Check the NIS domain name.
	Return -1 if not defined, 0 if left unchanged or 1 if changed
*/
PUBLIC int NIS_CONF::check_domain()
{
	int ret = -1;
	if (configok()){
		ret = 0;
		char buf[100];
		if (getdomainname(buf,sizeof(buf)-1)==-1
			|| domain.cmp(buf)!=0){
			ret = 1;
		}
	}
	return ret;
}
/*
	Set the NIS domain name.
	Return -1 if not defined, 0 if left unchanged or 1 if changed
*/
PUBLIC int NIS_CONF::set_domain()
{
	int ret = check_domain ();
	if (ret == 1){
		setdomainname(domain.get(),domain.getlen()+1);
		net_prtlog (NETLOG_CMD,MSG_U(X_SETNISDOM
			,"Set NIS domain to %s\n"),domain.get());
	}
	return ret;
}


/*
	Return the name of the optionnal NIS server or NULL.
*/
PUBLIC const char *NIS_CONF::getserver()
{
	return server.is_empty() ? (const char *)NULL : server.get();
}

/*
	Return the NIS domain name (Unrelated to the BIND domain name).
*/
PUBLIC const char *NIS_CONF::getdomain()
{
	return domain.get();
}

PUBLIC void NIS_CONF::write ()
{
	linuxconf_setcursys (subsys_netclient);
	linuxconf_replace(NIS,NIS_DOMAIN,domain);
	linuxconf_replace(NIS,NIS_UPDATE,update);
	SSTRINGS tb;
	// Read the file and write it back with the new ypserver value
	nis_readypconf (&tb,NULL);
	FILE_CFG *fout = f_ypconf.fopen ("w");
	if (fout != NULL){
		if (!server.is_empty()) fprintf (fout,"ypserver %s\n",server.get());
		for (int i=0; i<tb.getnb(); i++){
			fprintf (fout,"%s\n",tb.getitem(i)->get());
		}
		fclose (fout);
	}
	linuxconf_save();
}

/*
	Edit the configuration of the NIS client
	Return 0 if the edition was successful.
*/
PUBLIC int NIS_CONF::edit ()
{
	DIALOG dia;
	// bool empty = domain.is_empty();
	dia.newf_str (MSG_U(F_NISDOM,"NIS domain"),domain);
	dia.newf_str (MSG_U(F_NISSERV,"NIS server"),server);
	dia.newf_chk (MSG_U(F_UPDATE,"Update"),update
		,MSG_U(I_UPDATE,"the NIS database"));
	int ret = -1;
	int nof = 0;
	while (1){
		if (dia.edit(MSG_U(T_NISCONF,"NIS client configuration")
			,MSG_U(I_NISCONF,"You must enter the NIS domain\n"
			 "A server must be specified\n"
			 "if it can't be probed by a broadcast\n"
			 "The server is either specified as an IP number\n"
			 "or as a name.\n")
			,help_nis,nof) == MENU_ACCEPT){
			if (update && domain.is_empty()){
				xconf_error (MSG_U(E_UPDNODOMAIN
					,"You can set the update flag\n"
					 "without filling the NIS domain name"));
				nof = 0;
			}else if (valid_server()){
				ret = 0;
				set_domain();
				#if 0
				if (empty && domain.is_filled() && update){
					netconf_system_if ("nisconf_update",domain.get());
				}
				#endif
				break;
			}
		}else{
			break;
		}
	}
	if (ret != 0) dia.restore();
	return ret;
}

/*
	Return true if we are allowed to trigger the NIS database update command
*/
PUBLIC bool NIS_CONF::may_update()
{
	return update != 0;
}


/*
	Edite the file /etc/hosts
*/
void nis_edit()
{
	NIS_CONF nis;
	if (nis.edit() == 0) nis.write();
}
#if 0
/*
	Clone of the /bin/domainname command.
*/
int nis_domainname (int argc, char *argv[])
{
	/* #Specification: netconf / aliases / domainname
		netconf emulate the command /bin/domainname. Without argument
		it print the domainname (used by NIS). With one, it set the
		domain name.
	*/
	int ret = -1;
	if (argc == 1){
		char buf[100];
		if (getdomainname(buf,sizeof(buf)-1)==-1) strcpy (buf,"none");
		ret = 0;
		printf ("%s\n",buf);
	}else if (argc == 2){
		if (perm_rootaccess (MSG_U(P_SETTINGNIS
			,"Setting the NIS domain name"))){
			ret = setdomainname(argv[1],strlen(argv[1])+1);
		}
	}else{
		fprintf (stderr,MSG_U(E_USAGEDOM
			,"usage: domainname [ nisdomain ]\n"));
	}
	return ret;
}
#endif

#include <modregister.h>
static REGISTER_VARIABLE_LOOKUP_MSG nis_var_list[]={
	{"domain",NULL,P_MSG_R(F_NISDOM),nis_edit,NULL},
	{"server",NULL,P_MSG_R(F_NISSERV),nis_edit,NULL},
	{"update",NULL,P_MSG_R(F_UPDATE),nis_edit,NULL},
	{ NULL, NULL, NULL, NULL }
};
static REGISTER_VARIABLES nis_registry1("nis",nis_var_list);

