#include <sys/stat.h>
#include <unistd.h>
#include <sys/mman.h>
#include <string.h>
#include <stdlib.h>
#include "sysctl.h"

static CONFIG_FILE sysctl_config(
		"/etc/sysctl.conf",
		help_nil,
		CONFIGF_MANAGED|CONFIGF_OPTIONAL,
		"root","root",0644);

PRIVATE char *SYSCTL::dot_to_slash(const char *var)
{
	char *pvar = strdup(var);
	char *pvarp = pvar;
	for (;*pvarp != 0;pvarp++){
		if (*pvarp == '.') *pvarp = '/';
	}
	return pvar;
}

PRIVATE char *SYSCTL::slash_to_dot(const char *var)
{
	char *pvar = strdup(var);
	char *pvarp = pvar;
	for (;*pvarp != 0;pvarp++){
		if (*pvarp == '/') *pvarp = '.';
	}
	return pvar;
}

PUBLIC SYSCTL::SYSCTL()
	: vi(vip), modified(false)
{
	vip.quotchar = 0;
	vi.read(sysctl_config);
}

PUBLIC SYSCTL::~SYSCTL()
{
	if (modified){
		vi.write(sysctl_config,NULL);
	}
}
	
PUBLIC bool SYSCTL::probe(const char *var)
{
	const char *cfileval = this->get_from_file(var);
	if (cfileval != NULL){
		char *fileval = strdup(cfileval);
		const char *sysval = this->get_from_sys(var);
		if (sysval != NULL && strcmp(fileval,sysval) != 0){
			return true;
		}
		free(fileval);
	}
	return false;
}

PUBLIC const char *SYSCTL::get_from_file(const char *var)
{
	return vi.locateval(var,buf);
}

PUBLIC const char *SYSCTL::get_from_sys(const char *var)
{
	char *pvar = this->dot_to_slash(var);
	sprintf(buf,"/proc/sys/%s",pvar);
	free(pvar);
	
	FILE *file = fopen(buf,"r");
	if (file == NULL){
		return NULL;
	}
	int rsize = fread(buf,1,sizeof(buf)-1,file);
	if (rsize <= 0){
		return NULL;
	}
	
	buf[rsize] = 0;
	char *bufp = buf;
	while (*bufp != '\n' && *bufp != 0) bufp++;
	*bufp = 0;
	fclose(file);
	
	return buf;
}

PUBLIC const char *SYSCTL::get(const char *var)
{
	const char *ret = this->get_from_file(var);
	if (ret == NULL){
		ret = this->get_from_sys(var);
	}
	return ret;
}

PUBLIC void SYSCTL::set_to_file(const char *var, const char *val)
{
	modified = true;
	vi.update(var,val);
}

PUBLIC void SYSCTL::set_to_file(const char *var, int val)
{
	modified = true;
	sprintf(buf,"%d",val);
	vi.update(var,buf);
}

PUBLIC void SYSCTL::set_to_sys(const char *var, const char *val)
{
	modified = true;
	char *pvar = this->dot_to_slash(var);
	sprintf(buf,"/proc/sys/%s",pvar);
	free(pvar);
	FILE *file = fopen(buf,"w");
	if (file != NULL){
		fprintf(file,"%s",val);
	}
}

PUBLIC void SYSCTL::set_to_sys(const char *var, int val)
{
	modified = true;
	char *pvar = this->dot_to_slash(var);
	sprintf(buf,"/proc/sys/%s",pvar);
	free(pvar);
	FILE *file = fopen(buf,"w");
	if (file != NULL){
		fprintf(file,"%d",val);
	}
}

PUBLIC void SYSCTL::set(const char *var, const char *val)
{
	this->set_to_file(var,val);
	this->set_to_sys(var,val);
}

PUBLIC void SYSCTL::erase(const char *var)
{
	modified = true;
	char *pvar = this->slash_to_dot(var);
	VIEWITEM *it = vi.locateassign(var);
	free(pvar);
	if (it != NULL){
		vi.remove_del(it);
	}
}

