#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <misc.h>
#include <dialog.h>
#include "fstab.h"
#include "fstab.m"
#include <netconf.h>

extern FSTAB_HELP_FILE help_mount;
static FSTAB_HELP_FILE help_swap ("swap");
static FSTAB_HELP_FILE help_quotacheck ("quotacheck");

PRIVATE void FSTAB::fixroot(FSTAB_ENTRY *root, char *status)
{
	char *errmsg = NULL;
	struct stat roots;
	if (stat("/",&roots) < 0){
		errmsg = "directory /";
	}else{
		DIR *dp = opendir("/dev");
		if (dp == NULL){
			errmsg = "/dev";
		}else{
			const char *source = root->getsource();
			if (strncasecmp(source,"LABEL=",6)==0){
				const char *p = partition_findfromlabel(source+6);
				if (p != NULL) source = p;
			}
			char name[PATH_MAX+1];
			strcpy(name,"/dev/");
			struct dirent *dir;
			while ((dir = readdir(dp))!=NULL) {
				strcpy(name+5,dir->d_name);
				struct stat s;
				if (stat(name,&s) < 0){
					errmsg = name;
					break;
				}else if ((s.st_mode & S_IFMT) == S_IFBLK
					&& s.st_rdev == roots.st_dev){
					if (strcmp(source,name)!=0){
						root->setsource (name);
						strcat (status
							,MSG_U(N_ROOTDEF
								,"The root device has been changed to "));
						strcat (status,name);
						strcat (status,"\n\n");
					}
					break;
				}
			}
			closedir(dp);
		}
	}
	if (errmsg){
		xconf_error (MSG_U(E_CANTVLD,"Can't validate the root entry\n"
			"of your /etc/fstab file for the following\n"
			"reason:\n"
			"\n"
			"%s: %s"),errmsg,strerror(errno));
	}
}

/*
	Do some sanity check on one entry of /etc/fstab
	Return -1 if any error that can't be fixed.
*/
PUBLIC int FSTAB_ENTRY::check()
{
	/* #Specification: fstab / entry / checking
		fsconf make sure the entry has a valid (existing mount point).
		It check if it exist and it check if it is a directory.

		If it does not exist, the user is allowed to create it.
	*/
	int ret = 0;
	if (is_valid() && !is_swap()){
		struct stat st;
		const char *path = mpoint.get();
		ret = -1;
		if (path[0] == '\0'){
			xconf_error (MSG_U(E_NOMNTPT
				,"You must specify a "
				"mount point(directory)\n"
				"for entry %s\n"),source.get());
		}else if (context_isroot() && stat(path,&st)==-1){
			char buf[1000];
			sprintf (buf,MSG_U(Q_EXISTMNT,"Mount point %s\n"
				 "does not exist.\n"
				 "Do you want to create it ?\n")
				,path);
			if (dialog_yesno(MSG_U(T_CANTMOUNT,"Can't mount")
				,buf
				,help_mount) == MENU_YES){
				file_mkdirp (path,"root","root",0755);
				ret = 0;
			}
		}else if (context_isroot() && !S_ISDIR(st.st_mode)){
			xconf_error (MSG_U(E_ISNOTDIR
				,"%s is not a directory\n"
				 "please pick a different mount point\n")
				,path);
		}else{
			ret = 0;
		}
	}
	return ret;
}

/*
	Return the root device
*/
PUBLIC const char *FSTAB::getrootdev()
{
	const char *ret = "";
	for (int i=0; i<getnb(); i++){
		FSTAB_ENTRY *e = getitem(i);
		const char *mpoint = e->getmpoint();
		if (strcmp(mpoint,"/")==0){
			ret = e->getsource();
			break;
		}
	}
	return ret;
}

/*
	Make sure the /etc/fstab file contain minimally accurate entries.
	The caller will have to save it by checking was_modified().
*/
PUBLIC void FSTAB::check()
{
	const char *NO_SWAP_NEEDED = "# Please no swap";
	/* #Specification: /etc/fstab / minimal check
		The /etc/fstab is checked to make sure is contain
		the basic entries requiered to run successfully
		a Linux system.

		-It check for the root fs, both for presence and
		 accuracy. For example, if the user reorganise its
		 drive (linux was hdb and is now hda), or copy
		 a working (configured) installation to another
		 disk/partition, this utility will automagically
		 fix the root entry.

		-Check the /proc entry. It needs to be there.

		-Check for minimally a swap space. If there is none
		 the user will be prompt to create one. The user
		 will have the choice to say "No don't bother to
		 ask again, I don't need a swap". A special comment
		 line will be added to /etc/fstab to prevent
		 linuxconf from asking again and again.
	*/
	int swap_seen = 0;
	int proc_seen = 0;
	int no_swap_seen = 0;
	char status[1000];
	status[0] = '\0';
	for (int i=0; i<getnb(); i++){
		FSTAB_ENTRY *e = getitem(i);
		e->check();
		const char *mpoint = e->getmpoint();
		if (strcmp(mpoint,"/")==0){
			fixroot (e,status);
		}else if (strcmp(mpoint,"/proc")==0){
			proc_seen = 1;
		}else if (e->is_swap()){
			swap_seen = 1;
		}else if (strcmp(e->getcomment(),NO_SWAP_NEEDED)==0){
			no_swap_seen = 1;
		}
	}
	if (!proc_seen){
		strcat (status,MSG_U(N_PROCADD
			,"An entry for the /proc file system"
			 " has been added\n"
			 "The proc file system is needed for quite a few\n"
			 "system utilities such as \"ps\"\n\n"));
		add (new FSTAB_ENTRY("none /proc proc defaults"));
	}
	if (!swap_seen && !no_swap_seen){
		if (dialog_yesno (MSG_U(T_NOSWAP,"No swap !!!")
			,MSG_U(Q_NOSWAP,"Your linux system is lacking\n"
			 "a swap file or swap partition\n"
			 "\n"
			 "This is a severe weakness\n"
			 "Do you want to create a swap now ?")
			,help_swap)
			== MENU_YES){
			edit (FSTAB_ENTRY_SWAP);
		}else if (dialog_yesno(MSG_U(T_AREYOUSURE,"Are you sure")
			,MSG_U(Q_BUGYOU,"I will bug you about your system\n"
			 "lacking a swap every time I can\n"
			 "\n"
			 "Do you want me to shut off ?")
			,help_swap)
			== MENU_YES){
			add (new FSTAB_ENTRY(NO_SWAP_NEEDED));
		}
	}
	if (status[0] != '\0'){
		xconf_notice(MSG_U(N_SOMEMODS
			,"Some ajustements were made to the /etc/fstab\n"
			 "\n"
			 "%s"),status);
	}
}

void fstab_check()
{
	FSTAB fstab;
	fstab.check();
	if (fstab.was_modified()) fstab.write();
}

PUBLIC void FSTAB_ENTRY::quotacheck()
{
	const char *dirfs = mpoint.get();
	const char *uflag = "";
	const char *quota_user_file = "aquota.user";
	const char *quota_group_file = "aquota.group";
	if (!kernel_newer(2,4,0)){
		quota_user_file = "quota.user";
		quota_group_file = "quota.group";
	}
	if (has_quota_u()){
		char path[PATH_MAX];
		if (strcmp(dirfs,"/")==0){
			sprintf (path,"/%s",quota_user_file);
		}else{
			sprintf (path,"%s/%s",dirfs,quota_user_file);
		}
		if (file_type(path)==-1){
			uflag = "-u";
			net_prtlog (NETLOG_WHY,MSG_U(I_WHYQUOTAUSER
				,"User quota is enabled for partition %s, but %s is missing\n")
				,source.get(),path);
		}
	}
	const char *gflag = "";
	if (has_quota_g()){
		char path[PATH_MAX];
		sprintf (path,"%s/%s",dirfs,quota_group_file);
		if (file_type(path)==-1){
			gflag = "-g";
			net_prtlog (NETLOG_WHY,MSG_U(I_WHYQUOTAGROUP
				,"Group quota is enabled for partition %s, but %s is missing\n")
				,source.get(),path);
		}
	}
	if (uflag[0] != '\0' || gflag[0] != '\0'){
		bool yes = true;
		if (!simul_ison()){
			char msg[10000];
			sprintf (msg,MSG_U(I_QUOTACHECK
				,"You have enable disk quota management on\n"
				 "the file-system %s.\n"
				 "To make this effective, we must run the quotacheck utility.\n"
				 "It will browse through the directories and update the current\n"
				 "disk usage for each users and groups. This will take a while\n"
				 "\n"
				 "Should we do this now ?")
				,dirfs);
			yes = dialog_yesno (
				MSG_U(T_QUOTACHECK,"Installing quota on a file-system")
				,msg,help_quotacheck)==MENU_YES;
		}
		if (yes){
			char opt[PATH_MAX+10];
			sprintf (opt,"%s %s %s",uflag,gflag,dirfs);
			netconf_system_if ("quotacheck",opt);
			netconf_system_if ("quotaon",opt);
			/* #Specification: quota / after quotacheck / apply all
				There is a sequence where the quota defaults are not
				enforced properly.

				#
				The user set the quota flags on one partition. Linuxconf
				update the fstab, but nothing else is happening.

				The user visits the quota defaults and sets various limits.
				When he accept the limit, linuxconf try to apply those
				defaults by walking all users and groups, computing
				the effective limits for each. But this does not
				work since the quotacheck has not been done yet.

				The user quit from linuxconf. Linuxcong signals that
				quotacheck must be run. Fine, but the limits
				are not set.

				So at the end of the quotacheck, after the quotaon,
				Linuxconf force a walk of all users and groups to make
				sure the limits are properly entered.
			*/
			quota_applyall(source.get());
		}
	}
}

