/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: tkill.c,v 6.8 1999/07/13 16:48:38 jsquyres Exp $
 *
 *	Function:	- kills OTB environment
 *			- reads process IDs from the kill file, aka
 *			  reset file, aka lock file for historical reasons
 *			- makes a very good attempt at ensuring the
 *			  death of each pid in the file
 */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include <args.h>
#include <debug.h>
#include <terror.h>

#define TIMEOUT		10000000	/* timeout on process death */

/*
 * external functions
 */
extern void		microsleep();
extern char		*killname();
extern char		*sockname();
extern char		*iosockname();
extern void		lam_cleanup_objects();


int
main(argc, argv)

int			argc;
char			*argv[];

{
	FILE		*fp_kill;	/* kill file ptr */
	int		fd_kill;	/* kill file desc. */
	int		fl_debug;	/* debugging option */
	int		fl_pretend;	/* pretend option */
	int		fl_verbose;	/* verbose option */
	int		n;		/* favourite counter */
	int		pid;		/* lots of space to hold a PID */
	unsigned int	usec;		/* microseconds to sleep */
	char		*f_kill;	/* kill file */
	char		*f_sock;	/* socket file */
	char		*f_iosock;	/* io daemon socket file */
	char		syscmd[256];	/* system command */
	char            extra[1024];    /* place for ASCII messages */

/*
 * Initialize option parser.
 */
	validopts("dhvN");
	followed("f");

	if ((do_args(&argc, argv)) || (errno = (argc == 1) ? 0 : EUSAGE)) {
	    show_help("tkill", "usage", NULL);
	    exit(errno);
	}

	/* Root can only run this program is "-N" is supplied (i.e., if it
	 * was launched from recon */

	fl_pretend = opt_taken('N');
	if (!fl_pretend && (getuid() == 0 || geteuid() == 0)) {
	  show_help(NULL, "deny-root", NULL);
	  exit(EACCES);
	}

	if (opt_taken('h')) {
	    show_help("tkill", "usage", NULL);
	    exit(0);
	}

	fl_debug = opt_taken('d');
	fl_verbose = opt_taken('v') || fl_debug;
/*
 * Remove the socket file.
 */
	DBUG("tkill: removing socket file ...\n");
	f_sock = sockname();

	if (f_sock == 0) {
	  show_help(NULL, "lib-call-fail", "sockname", NULL);
	  exit(errno);
	}

	if (!fl_pretend) {
	    if (unlink(f_sock)) {
		if (errno != ENOENT) {
		  snprintf(extra, 1024, "unlink(\"%s\")", f_sock);
		  show_help(NULL, "system-call-fail", extra, NULL);
		}
	    }
	}
/*
 * Remove the IO daemon socket file.
 */
	DBUG("tkill: removing IO daemon socket file ...\n");
	f_iosock = iosockname();

	if (f_iosock == 0) {
	  show_help(NULL, "lib-call-fail", "iosockname", NULL);
	  exit(errno);
	}

	if (!fl_pretend) {
	    if (unlink(f_iosock)) {
		if (errno != ENOENT) {
		  snprintf(extra, 1024, "unlink(\"%s\")", f_iosock);
		  show_help(NULL, "system-call-fail", extra, NULL);
		}
	    }
	}
/*
 * Get the kill filename.
 */
	if (opt_taken('f')) {
	    f_kill = getparam('f');
	} else {
	    f_kill = killname();

	    if (f_kill == 0) {
	      show_help(NULL, "lib-call-fail", "killname", NULL);
	      exit(errno);
	    }
	}
/*
 * Dump debugging information.
 */
	DBUG("tkill: f_kill = \"%s\"\n", f_kill);
/*
 * Try to open the file.
 */
	if ((fd_kill = open(f_kill, O_RDWR, 0)) < 0) {

	    if (errno == ENOENT) {
		VERBOSE("tkill: nothing to kill: \"%s\"\n", f_kill);
		exit(0);
	    } else {
	      snprintf(extra, 1024, "open(\"%s\", O_RDWR)", f_kill);
	      show_help(NULL, "system-call-fail", extra, NULL);
	      exit(errno);
	    }
	}

	fp_kill = fdopen(fd_kill, "r");

	if (! fp_kill) {
	  snprintf(extra, 1024, "fdopen(\"%s\", \"r\")", f_kill);
	  show_help(NULL, "lib-call-fail", extra, NULL);
	  exit(errno);
	}

	VERBOSE("tkill: killing LAM...\n");

	n = fscanf(fp_kill, "%d", &pid);

	while (n > 0) {
	    DBUG("tkill: killing PID %d ...", pid);

	    if (fl_debug) 
	      fflush(stdout);
/*
 * Send SIGHUP to the process.
 */
	    if (fl_pretend) {
		DBUG("\n");
		n = fscanf(fp_kill, "%d", &pid);
		continue;
	    }

	    if (kill(pid, SIGHUP)) {

		if (errno == ESRCH) {
		    DBUG(" already dead");
		} else {
		    DBUG("\n");
		    perror("tkill (kill)");
		    exit(errno);
		}
	    } else {
		usec = 2;

		while ((kill(pid, SIGHUP) == 0) && (usec <= TIMEOUT)) {
		    microsleep(usec);
		    usec *= 2;
		}
	
		if (usec > TIMEOUT) {
		    DBUG(" trying -9 ...");
		    kill(pid, SIGKILL);
		    microsleep(500000);

		    if (kill(pid, SIGHUP)) {
			DBUG(" killed");
		    } else {
			DBUG(" cannot kill");

			snprintf(extra, 1024, "%d", pid);
			show_help("tkill", "cannot-kill", extra, NULL);
		    }
		} else {
		    DBUG(" killed");
		}
	    }
		
	    DBUG("\n");
	    n = fscanf(fp_kill, "%d", &pid);
	}

	if (fl_pretend) {
	    exit(0);
	}
/*
 * Remove the kill file.
 */
	DBUG("tkill: removing kill file ...\n");

	if (unlink(f_kill)) {
	  snprintf(extra, 1024, "unlink(\"%s\")", f_kill);
	  show_help(NULL, "system-call-fail", extra, NULL);
	  exit(errno);
	}
/*
 * Cleanup all registered objects.
 */
	lam_cleanup_objects();
/*
 * Delete any stray temporary files.
 */
	sprintf(syscmd, "%s %s*", RM, f_kill);
	system(syscmd);

	free(f_kill);
	free(f_sock);
	free(f_iosock);
	return(0);
}

