/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: wfence.c,v 1.0 1999/07/21 19:00:21 jsquyres Exp $
 *
 *	Function:	- window fence synchronization
 *	Accepts:	- assertion about the fence
 *			- window
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>


int
MPI_Win_fence(assertion, win)

int			assertion;
MPI_Win			win;

{
	MPI_Comm	comm;			/* window communicator */
	MPI_Request	*reqs;
	int		np;			/* # processes in window */
	int		fence_msg[1];		/* fence message */
	int		err;			/* error code */
	int		i;

	lam_initerr_m();
	lam_setfunc_m(BLKMPIWINFENCE);

	if (win == MPI_WIN_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIWINFENCE, lam_mkerr(MPI_ERR_WIN, 0)));
	}

	comm = win->w_comm;
/*
 * Check if started or posted.
 */
	if (win->w_flags & (LAM_WFSTARTED | LAM_WFPOSTED)) {
		return(lam_errfunc(comm,
			BLKMPIWINFENCE, lam_mkerr(MPI_ERR_EPOCH, 0)));
	}
/*
 * Send mesages to all window processes to inform them that this process
 * has entered the fence.  We can pre-allocate the request array in
 * MPI_Win_init() to trade off space for time.
 */
	np = comm->c_group->g_nprocs;

	reqs = (MPI_Request *) malloc(np * sizeof(MPI_Request));
	if (reqs == 0) {
		return(lam_errfunc(comm,
			BLKMPIWINFENCE, lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	fence_msg[0] = LAM_OSD_FENCE;

	for (i = 0; i < np; i++) {
		err = MPI_Isend(fence_msg, 1, MPI_INT,
			i, LAM_OSD_HEADER, comm, reqs + i);

		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIWINFENCE, err));
		}
	}
/*
 * Loop until everybody has entered the fence.
 */
	_mpi_req_blkclr_m();
	_mpi_req_blkset_m(win->w_req);

	while (win->w_nfence > 0) {

		err = lam_osd_blkadv(win);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIWINFENCE, err));
		}
	}
/*
 * Complete all outstanding one-sided requests.
 */
	err = lam_osd_complete(win, LAM_RQFOSORIG | LAM_RQFOSTARG);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINFENCE, err));
	}
/*
 * Complete the "entered fence" sends.
 */
	err = MPI_Waitall(np, reqs, MPI_STATUSES_IGNORE);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINFENCE, err));
	}
	free((char *) reqs);

	win->w_nfence = np;
	win->w_flags &= ~(LAM_WFACCEPOCH | LAM_WFEXPEPOCH | LAM_WFNOOUT);
/*
 * Barrier to avoid concurrent fences.
 */
	err = MPI_Barrier(win->w_comm);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(win->w_comm, BLKMPIWINFENCE, err));
	}

        lam_resetfunc_m(BLKMPIWINFENCE);
	return(MPI_SUCCESS);
}
