//LabPlot : ExplorerDialog.cc

#include <stdlib.h>
#include <stdio.h>
#include <iostream>
#include <math.h>
#include <qlabel.h>
#include <qhbox.h>
#include <qfontdialog.h>
#include <klocale.h>
#include <kiconloader.h>
#include "ExplorerDialog.h"
#include "Plot2DSimple.h"
#include "Plot2DSurface.h"
#include "Plot3D.h"
#include "PlotPie.h"
#include "PlotPolar.h"
#include "PlotTernary.h"
#include "PlotQWT3D.h"

#include "pixmaps/pixmap.h"

ExplorerItem::ExplorerItem(Worksheet *ws, Spreadsheet *ss, Plot *plot, Graph *graph)
	: ws(ws), ss(ss), plot(plot), graph(graph)
{}

ExplorerDialog::ExplorerDialog(MainWin *mw, const char *name)
	: Dialog(mw, name)
{
	kdDebug()<<"Explorer Dialog"<<endl;
	setCaption(i18n("Project Explorer"));

	lv = new ExplorerListView(vbox,0);
	lv->setSorting(-1);
	lv->addColumn(i18n("Type"));
	lv->addColumn(i18n("Name"));
	lv->addColumn(i18n("Info"));
	lv->setRootIsDecorated( true );
	connect(lv, SIGNAL( contextMenuRequested( QListViewItem *, const QPoint& , int ) ),
		this, SLOT( rmouse( QListViewItem *, const QPoint &, int ) ) );
	connect(lv, SIGNAL( pressed( QListViewItem *) ),this, SLOT( pressEvent(QListViewItem *) ) );	
	connect(lv, SIGNAL( mouseReleaseEvent( QMouseEvent *) ),this, SLOT( releaseEvent(QMouseEvent *) ) );

	for (int i=0;i<lv->columns();i++)
		lv->setColumnAlignment(i,Qt::AlignHCenter);

	QPixmap spreadsheetIcon = QPixmap(spreadsheet_xpm);
	QPixmap worksheetIcon = QPixmap(worksheet_xpm);
	QPixmap newIcon = KGlobal::iconLoader()->loadIcon("new", KIcon::Small);
	QPixmap deleteIcon = KGlobal::iconLoader()->loadIcon("editdelete", KIcon::Small);
	QPixmap copyIcon = KGlobal::iconLoader()->loadIcon("editcopy", KIcon::Small);

	QPopupMenu *newmenu = new QPopupMenu(lv);

	menu = new QPopupMenu( lv );
	menu->insertItem(deleteIcon,  i18n( "Delete" ),this,SLOT(deleteItem()) );
	menu->insertItem(copyIcon,  i18n( "Clone" ),this,SLOT(cloneItem()) );
	menu->insertSeparator();
	menu->insertItem( newIcon, i18n( "New" ), newmenu );
	newmenu->insertItem( newIcon, i18n( "New Spreadsheet" ), this, SLOT(newSpreadsheet()), CTRL+Key_Equal );
	newmenu->insertItem( worksheetIcon, i18n( "New Worksheet" ), this, SLOT(newWorksheet()), ALT+Key_X );

	updateList();

	QObject::connect(ok,SIGNAL(clicked()),SLOT(reject()));
	apply->hide();
	save->hide();
	cancel->hide();

	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(400,300);
}

void ExplorerDialog::updateList() {
	kdDebug()<<"ExplorerDialog::updateList()"<<endl;
	QWorkspace *workspace = mw->getWorkspace();
	QWidgetList wlist = workspace->windowList();

	int nr=0;
	list = new ExplorerItem[1000];	// TODO : max number of items ?
	lv->clear();
	
	QListViewItem *lvisheet=0;
	for (unsigned int i = 0; i < wlist.count(); ++i ) {
		kdDebug()<<" LIST : Sheet "<<i<<endl;
		
		lvisheet = new QListViewItem(lv,lvisheet);
		lvisheet->setOpen(true);
		
		if(((Spreadsheet *) wlist.at(i))->getWidgetType() == WSPREADSHEET ) {
			Spreadsheet *ss =  ((Spreadsheet *) wlist.at(i));
			list[nr++] = ExplorerItem(0,ss);
			
			lvisheet->setText(0,"Spreadsheet");
			lvisheet->setText(1,ss->Title());
			lvisheet->setText(2,ss->Info().join(" "));
		}
		else {
			Worksheet *ws =  ((Worksheet *) wlist.at(i));
			list[nr++] = ExplorerItem(ws,0);

			kdDebug()<<" 	Worksheet "<<ws->Title()<<endl;
			lvisheet->setText(0,"Worksheet");
			lvisheet->setText(1,ws->Title());
			QStringList info = ws->Info();
			info.remove(info.begin());
			lvisheet->setText(2,info.join(" "));
			
			QListViewItem *lviplot=0;
			for (unsigned int j=0;j<ws->NrPlots();j++) {
				lviplot = new QListViewItem(lvisheet,lviplot);
				lviplot->setOpen(true);
				Plot *plot = ws->getPlot(j);
//				kdDebug()<<" 	PLOT "<<plot->Title()->simpleTitle()<<endl;
				list[nr++] = ExplorerItem(ws,0,plot);
	
				QString ptype;
				QStringList info;
				switch(plot->Type()) {
				case P2D: ptype=i18n("2 D"); 
						info = ((Plot2DSimple *)plot)->Info();
						break;
				case PSURFACE: ptype=i18n("SURFACE"); 
						info = ((Plot2DSurface *)plot)->Info();
						break;
				case P3D: ptype=i18n("3 D"); 
						info = ((Plot2DSurface *)plot)->Info();
						break;
				case PPIE: ptype=i18n("PIE");
						info = ((PlotPie *)plot)->Info();
						break;
				case PPOLAR: ptype=i18n("POLAR");
						info = ((PlotPolar *)plot)->Info();
						break;
				case PTERNARY: ptype=i18n("TERNARY");
						info = ((PlotTernary *)plot)->Info();
						break;
				case PQWT3D: ptype=i18n("QWT 3D");
						info = ((PlotQWT3D *)plot)->Info();
						break;
				default:break;
				}
				lviplot->setText(0,ptype + ' ' +i18n("Plot"));
				lviplot->setText(1,plot->Title()->simpleTitle());
				info.remove(info.begin());
				lviplot->setText(2,info.join(" "));
				
				// list of graphs
//				kdDebug()<<"getting list of graphs for plot "<<j<<" of type "<<ptype<<endl;
				GraphList *graphlist = plot->getGraphList();
				for(unsigned int k=0;k<graphlist->Number();k++) {
					QListViewItem *lvigraph = new QListViewItem(lviplot,lv->lastItem());
					lvigraph->setOpen(true);
//					kdDebug()<<"getting graph "<<k<<endl;
					Graph *graph = graphlist->getGraph(k);
//					kdDebug()<<" 	GRAPH "<<graph->getLabel()->simpleTitle()<<endl;
					list[nr++] = ExplorerItem(ws,0,plot,graph);

					QString gtype;
//					kdDebug()<<"getting infos for graph "<<k<<endl;
					switch (graphlist->getStruct(k) ) {
					case GRAPH2D: gtype=i18n("2 D"); 
						info = ((Graph2D *) graph)->Info();
						break;
					case GRAPH3D: gtype=i18n("3 D"); 
						info = ((Graph3D *) graph)->Info();
						break;
					case GRAPHM: gtype=i18n("MATRIX"); 
						info = ((GraphM *) graph)->Info();
						break;
					case GRAPH4D: gtype=i18n("4 D"); 
						info = ((Graph4D *) graph)->Info();
						break;
					case GRAPHIMAGE: gtype=i18n("IMAGE"); 
						info = ((GraphIMAGE *) graph)->Info();
						break;
					default:
						break;
					}
//					kdDebug()<<"filling listview for graph "<<k<<endl;
					lvigraph->setText(0,gtype+ ' ' + i18n("Graph"));
					lvigraph->setText(1,graph->getLabel()->simpleTitle());
					info.remove(info.begin());
					info.remove(info.begin());
					lvigraph->setText(2,info.join(" "));
				}
			}
		}
	}
	for (int i=0;i<lv->columns();i++)
		lv->setColumnAlignment(i,Qt::AlignHCenter);
}

//! context menu
void ExplorerDialog::rmouse(QListViewItem *, const QPoint & point, int ) {
	menu->popup( point );
}

void ExplorerDialog::deleteItem() {
	kdDebug()<<"ExplorerDialog::deleteItem() : "<<endl;
	int item = lv->itemPos(lv->currentItem())/lv->currentItem()->height();
        kdDebug()<<"	ITEM = "<<item<<endl;
	
	ExplorerItem ei = list[item];
	Spreadsheet *ss = ei.getSpreadsheet();
	if(ss != 0) {
		ss->closeNow();	// dont ask questions. Needed if updateList should work
	}
	else {
		Worksheet *ws = ei.getWorksheet();
		Plot *plot = ei.getPlot();
		
		if(plot == 0) {
			ws->closeNow();
		}
		else {
			Graph *graph = ei.getGraph();
			if(graph == 0) {
				for(unsigned int i=0;i<ws->NrPlots();i++)
					if(plot == ws->getPlot(i))
						ws->deletePlot(i);
			 }
			else {
				GraphList *gl = plot->getGraphList();
				for(unsigned int i=0;i<gl->Number();i++)
					if(graph == gl->getGraph(i))
						gl->delGraph(i);
				ws->redraw();
			}
		}
	}

	updateList();
}

void ExplorerDialog::cloneItem() {
	kdDebug()<<"ExplorerDialog::cloneItem() : "<<endl;
	int item = lv->itemPos(lv->currentItem())/lv->currentItem()->height();
        kdDebug()<<"	ITEM = "<<item<<endl;
	
	ExplorerItem ei = list[item];
	Spreadsheet *ss = ei.getSpreadsheet();
	if(ss != 0) {
		mw->cloneSpreadsheet(ss);
	}
	else {
		Worksheet *ws = ei.getWorksheet();
		Plot *plot = ei.getPlot();
		
		if(plot == 0) {
			mw->cloneWorksheet(ws);
		}
		else {
			Graph *graph = ei.getGraph();
			if(graph == 0) {	// clone selected plot
				for(unsigned int i=0;i<ws->NrPlots();i++)
					if(plot == ws->getPlot(i))
						ws->clonePlot(i);
			 }
			else {		// clone selected Graph
				GraphList *gl = plot->getGraphList();
				for(unsigned int i=0;i<gl->Number();i++)
					if(graph == gl->getGraph(i)) {
						switch(gl->getStruct(i)) {
						case GRAPH2D: {
							gl->addGraph2D(gl->getGraph2D(i)->Clone());
							}; break;
						case GRAPH3D: {
							gl->addGraph3D(gl->getGraph3D(i)->Clone());
							}; break;
						case GRAPH4D: {
							gl->addGraph4D(gl->getGraph4D(i)->Clone());
							}; break;
						case GRAPHM: {
							gl->addGraphM(gl->getGraphM(i)->Clone());
							}; break;
						case GRAPHIMAGE: {
							gl->addGraphIMAGE(gl->getGraphIMAGE(i)->Clone());
							}; break;
						default: break;
						}
					}
				ws->redraw();
			}
		}
	}

	updateList();
}

void ExplorerDialog::pressEvent(QListViewItem *) {
	dragitem = lv->itemPos(lv->currentItem())/lv->currentItem()->height();
}

void ExplorerDialog::releaseEvent(QMouseEvent *e) {
	dropitem = lv->itemPos(lv->currentItem())/lv->currentItem()->height();
	
	// do nothing on same item
	if(dropitem == dragitem) return;
	
	QPopupMenu *dropmenu = new QPopupMenu(lv);
	
	QPixmap copyIcon = KGlobal::iconLoader()->loadIcon("editcopy", KIcon::Small);
	QPixmap moveIcon = KGlobal::iconLoader()->loadIcon("editmove", KIcon::Small);
	QPixmap cancelIcon = KGlobal::iconLoader()->loadIcon("cancel", KIcon::Small);
	
	dropmenu->insertItem(copyIcon,  i18n( "Copy" ),this,SLOT(copyItem()) );
	dropmenu->insertItem(moveIcon,  i18n( "Move" ),this,SLOT(moveItem()) );
	dropmenu->insertSeparator();
	dropmenu->insertItem(cancelIcon,  i18n( "Cancel" ));
	
	dropmenu->popup( e->globalPos() );
}

void ExplorerDialog::copyItem(bool remove) {
	kdDebug()<<"	copy item "<<dragitem<<" to item "<<dropitem<<endl;

	ExplorerItem e1 = list[dragitem];
	ExplorerItem e2 = list[dropitem];
	Spreadsheet *ss1 = e1.getSpreadsheet();
	if(ss1 != 0) {
//		kdDebug()<<"	copy spreadsheet"<<endl;
		mw->cloneSpreadsheet(ss1);
	}
	else {
		kdDebug()<<"	copy worksheet/plot/graph"<<endl;
		Worksheet *ws1 = e1.getWorksheet();
		Worksheet *ws2 = e2.getWorksheet();
		Plot *plot1 = e1.getPlot();
		Plot *plot2 = e2.getPlot();
		if(plot2 == 0)		// move graph to worksheet
			plot2 = ws2->getPlot(0);
		
		if(plot1 == 0) {
//			kdDebug()<<"	copy worksheet"<<endl;
			mw->cloneWorksheet(ws1);
		}
		else {
//			kdDebug()<<"	copy plot or graph"<<endl;
			Graph *graph = e1.getGraph();
			if(graph == 0) {	// copy plot1 to ws2
//				kdDebug()<<"	COPY PLOT "<<plot1->Title()<<" to WORKSHEET "<<ws2->Title()<<endl;
				for(unsigned int i=0;i<ws1->NrPlots();i++)
					if(plot1 == ws1->getPlot(i)) {
						ws2->setPlot(ws2->NrPlots(),ws1->clonePlot(i,false));
						ws2->setNrPlots(ws2->NrPlots()+1);
						if(remove) ws1->deletePlot(i);
					}
			 }
			else {		// copy graph to plot2
//				kdDebug()<<"	copy GRAPH "<<graph->getLabel()->simpleTitle()<<endl;
				GraphList *gl1 = plot1->getGraphList();
				GraphList *gl2 = plot2->getGraphList();
				for(unsigned int i=0;i<gl1->Number();i++) {
					if(graph == gl1->getGraph(i)) {
//				kdDebug()<<"	copy GRAPH "<<i<<" ("<<graph->Name()<<") to PLOT "<<plot2->Title()<<endl;
						// clone label
						Label *l = new Label();
						*l = *(graph->getLabel());
						
						switch(gl1->getStruct(i)) {
						case GRAPH2D: {
							Graph2D newg = *(gl1->getGraph2D(i));
							newg.setLabel(l);
							gl2->addGraph2D(&newg);
							}; break;
						case GRAPH3D: {
							Graph3D newg = *(gl1->getGraph3D(i));
							newg.setLabel(l);
							gl2->addGraph3D(&newg);
							}; break;
						case GRAPH4D: {
							Graph4D newg = *(gl1->getGraph4D(i));
							newg.setLabel(l);
							gl2->addGraph4D(&newg);
							}; break;
						case GRAPHM: {
							GraphM newg = *(gl1->getGraphM(i));
							newg.setLabel(l);
							gl2->addGraphM(&newg);
							}; break;
						case GRAPHIMAGE: {
							GraphIMAGE newg = *(gl1->getGraphIMAGE(i));
							newg.setLabel(l);
							gl2->addGraphIMAGE(&newg);
							}; break;
						default: break;
						}
						
						if(remove) gl1->delGraph(i);
					}
				}
			}
			ws2->redraw();
		}
	}

	updateList();
}

void ExplorerDialog::moveItem() {
	kdDebug()<<"	move item "<<dragitem<<" to item "<<dropitem<<endl;
	copyItem(true);
}
