
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <linux/kd.h>

#include "typedefs.h"
#include "leds.h"
#include "log.h"

/* console device */
static const char *console = "/dev/console";

/* LED states */
static int led_state[3] = { LED_NUM, LED_CAP, LED_SCR };

#ifdef KBDFLAGS
/* key states */
static int key_state[3] = { K_NUMLOCK, K_CAPSLOCK, K_SCROLLLOCK };
#endif

/* LED file descriptor */
static int fd = 0;

/***********************************************************************************/
/***********************************************************************************/
/***********************************************************************************/

static char get_led_status ()
{
   char ledval;
   if (!fd) return (-1);
   if (ioctl (fd,KDGETLED,&ledval) < 0) log_printf (LOG_ERROR,"ioctl(KDGETLED): %s\n",strerror (errno));
   return (ledval);
}

static int set_led_status (char ledval)
{
#ifdef KBDFLAGS
   char keyval;
   int i;
#endif
   if (!fd) return (-1);
#ifdef KBDFLAGS
   if (ioctl (fd,KDGKBLED,&keyval) < 0)
	 {
		log_printf (LOG_ERROR,"ioctl(KDGKBLED): %s\n",strerror (errno));
		return (-1);
	 }
   for (i = 0; i < 3; i++) if (keyval & key_state[i]) ledval |= led_state[i];
#endif
   if (ioctl (fd,KDSETLED,ledval) < 0)
	 {
		log_printf (LOG_ERROR,"ioctl(KDSETLED): %s\n",strerror (errno));
		return (-1);
	 }
   return (0);
}

/***********************************************************************************/
/***********************************************************************************/
/***********************************************************************************/

const char *led_init ()
{
   static char errmsg[1024];
   if (fd) return (NULL);
   if ((fd = open (console,O_RDONLY)) < 0)
	 {
		sprintf (errmsg,"couldn't open %s for reading\n",console);
		return (errmsg);
	 }
   return (NULL);
}

void led_set (int n)
{
   if (n >= 0 && n < 3) set_led_status (get_led_status () | led_state[n]);
}

void led_clear (int n)
{
   if (n >= 0 && n < 3) set_led_status (get_led_status () & ~led_state[n]);
}

void led_setall ()
{
   char ledval = get_led_status ();
   ledval |= LED_NUM | LED_CAP | LED_SCR;
   set_led_status (ledval);
}

void led_clearall ()
{
   char ledval = get_led_status ();
   ledval &= ~LED_NUM | ~LED_CAP | ~LED_SCR;
   set_led_status (ledval);
}

void led_close ()
{
   led_clearall ();
   close (fd);
}

