/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
/*
  visibility-polygon.cpp: ipelet for finding the visibility polygon of
  a point inside a polygon

  Author: Chris Gray <cgray@win.tue.nl>
  License: Same as ipe (GPL with CGAL exception)
  Date: Thu Aug 11 14:51:03 CEST 2005

  Commentary: This ipelet computes the visibility polygon of a point
  inside a polygon.  The algorithm is adapted from the book _Art
  Gallery Theorems and Algorithms_ by Joseph O'Rourke (which was in
  turn based on the algorithm by Lee from 1985).

 */

#include "ipelet.h"
#include "ipepath.h"
#include "ipepage.h"
#include "ipevisitor.h"
#include "ipemark.h"

#include <list>
using std::list;

static const char *aboutText =
"Written by Chris Gray (www.win.tue.nl/~cgray/ipelets.html)";

// --------------------------------------------------------------------

int TurnType (IpeVector *a, IpeVector *b, IpeVector *c);
class VisibilityStack;

class VisibilityPolygonIpelet : public Ipelet {
public:
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  virtual const char *Label() const { return "Visibility Polygon"; }
  virtual const char *About() const { return aboutText; }
  virtual void Run(int, IpePage *page, IpeletHelper *helper);
private:
  virtual VisibilityStack *FindVisibilityPolygon(IpeVector **poly,
						 IpeVector *point,
						 int npoints);
  virtual IpeVector **RenumberPoly(IpeVector **poly, IpeVector *point,
				   int npoints);

};

class VisibilityStack {
private:
  IpeVector **poly;
  double *polyAlphas;
  IpeVector *point;
  int npoints;
  int stackSize;
  list<IpeVector *> myStack;
  list<double> alphaStack;
  IpeVector *top() { return myStack.front(); }
  double alphaTop() { return alphaStack.front(); }
  void push (IpeVector *pt, double alpha)
  { myStack.push_front(pt); alphaStack.push_front(alpha); stackSize++; }
  void pop ()
  { IpeVector *pt = top(); myStack.pop_front(); delete pt; stackSize--;
  alphaStack.pop_front(); }
  int stopPop (int i, IpeVector *prevPoint, double prevAlpha,
	       IpeVector &outPoint);

public:
  void SetPoly (IpeVector **p, int num) { poly = p; npoints = num; }
  void SetPoint (IpeVector *p) { point = p; }
  void Run ();
  IpeVector **ToPolygon ();
  VisibilityStack() { poly = NULL; point = NULL; stackSize = 0; npoints = 0; }
  int Length() { return stackSize; }
  ~VisibilityStack();
};

VisibilityStack::~VisibilityStack ()
{
  int i;
  for (i = 0; i < npoints; i++) {
    delete poly[i];
  }
  delete [] poly;
  delete [] polyAlphas;
}

// Returns 0, 1, or 2.
// 0 means "don't stop"
// 1 means "stop because case a happened"
// 2 means "stop because case b happened"
// If 1 or 2 is returned, outPoint is filled with the appropriate point.
int VisibilityStack::stopPop(int i, IpeVector *prevPoint,
			     double prevAlpha, IpeVector &outPoint)
{
  double currentAlpha = alphaTop();
  IpeVector *currentPoint = top();

  IpeSegment seg(*(point), *currentPoint);
  IpeSegment seg2(*(poly[i]), *(poly[(i + 1) % npoints]));
  IpeSegment seg3(*(poly[(i + 1) % npoints]), *(poly[(i + 2) % npoints]));
  IpeSegment seg4(*currentPoint, *prevPoint);
  if ((fabs(currentAlpha - prevAlpha) <= 1e-10
       || fabs(fabs(currentAlpha - prevAlpha) - 2 * M_PI) <= 1e-10)
      && (seg4.Intersects(seg3, outPoint)
	  || seg4.Intersects(seg2, outPoint))) {
    IpeVector v1 = outPoint - *point;
    IpeVector v2 = *currentPoint - *point;
    if (v2.SqLen() < v1.SqLen()) {
      return 2;
    }
  }
  if (prevAlpha >= polyAlphas[(i + 1) % npoints]
      && polyAlphas[(i + 1) % npoints] > currentAlpha) {
    IpeSegment seg(*(point), *(poly[(i + 1) % npoints]));
    IpeSegment seg2(*prevPoint, *currentPoint);
    seg2.Intersects(seg.Line(), outPoint);
    return 1;
  }

  return 0;
}

void VisibilityStack::Run ()
{
  int i;
  IpeSegment *W = new IpeSegment();
  IpeVector y;
  int stopReason = 0;

  if (npoints == 0) return;

  polyAlphas = new double[npoints];

  // fill in the polyAlphas array
  polyAlphas[0] = 0.0;
  for (i = 1; i < npoints; i++) {
    IpeVector v1 = *(poly[i]) - *point;
    IpeVector v2 = *(poly[i - 1]) - *point;
    polyAlphas[i] = polyAlphas[i - 1]
      + TurnType(point, poly[i - 1], poly[i])
      * acos((v1.iX * v2.iX + v1.iY * v2.iY) / (v1.Len() * v2.Len()));
  }

  push(new IpeVector(*poly[0]), 0.0);
  enum state { PUSH, POP, WAIT } state = PUSH;
  for (i = 0; i < npoints;) {
    int iplus1modn = (i + 1) % npoints;
    switch (state) {
    case PUSH:
      if (polyAlphas[iplus1modn] < 2 * M_PI
	  || (fabs(polyAlphas[iplus1modn] - 2 * M_PI) <= 1e-10)) {
	push(new IpeVector(*poly[iplus1modn]), polyAlphas[iplus1modn]);
	i++;
	iplus1modn = (i + 1) % npoints;
	if (i == npoints) goto out;
	if (polyAlphas[iplus1modn] >= polyAlphas[i] || i == npoints - 1) {
	  // nothing
	} else {
	  if (TurnType(poly[i - 1], poly[i], poly[iplus1modn]) >= 0) {
	    state = POP;
	  } else {
	    // Set W to s_t \infty in the direction from x to s_t
	    state = WAIT;
	    W->iP = *(top());
	    IpeVector v = W->iP - *point;
	    v *= 1e42;
	    W->iQ = W->iP + v;
	  }
	}
      } else {
	// push the intersection of the ray x v_0 with v_i v_{i + 1}
	// onto the stack and then call wait with W = v_0 s_t
	state = WAIT;
	// calculate the intersection between the horizontal line from
	// v_0 and v_i v_{i + 1}
	double pointY = point->iY;
	double edgeY1 = poly[i]->iY;
	double edgeY2 = poly[iplus1modn]->iY;
	if ((pointY < edgeY1 && pointY > edgeY2)
	    || (pointY > edgeY1 && pointY < edgeY2)) {
	  double a = (pointY - edgeY2) / (edgeY1 - edgeY2);
	  double edgeX1 = poly[i]->iX;
	  double edgeX2 = poly[iplus1modn]->iX;
	  double xprime = (a * edgeX1) + ((1 - a) * edgeX2);
	  // push the intersection with alpha 2\pi
	  push(new IpeVector(xprime, pointY), 2 * M_PI);
	  // set W to be v_0 s_t
	  W->iQ = *(poly[0]);
	  W->iP = *(top());
	} else {
	  // something bad happened.
	  assert(0);
	}
      }
      break;
    case POP:
      // pop until one of two things happens:
      // a) prevAlpha >= polyAlphas[i + 1] > alphaTop
      // b) prevAlpha = alphaTop >= polyAlphas[i + 1] and the
      // intersection of the ray from point through prevAlpha to the
      // next segment happens between prevAlpha and alphaTop
      double prevAlpha;
      IpeVector *prevPoint;
      do {
	prevAlpha = alphaTop();
	prevPoint = top();
	pop();
      } while (!(stopReason = stopPop(i, prevPoint, prevAlpha, y)));

      // what you do depends on whether a or b happened
      if (stopReason == 1) {
	IpeVector v = y - *point;
	push(new IpeVector(y), v.Angle().Normalize(0.0));
	i++;
	if (i == npoints) {
	  goto out;
	}
	if (polyAlphas[i] > polyAlphas[(i + 1) % npoints]) {
	  state = POP;
	} else {
	  if (TurnType(poly[i - 1], poly[i], poly[(i + 1) % npoints]) < 0) {
	    i--;
	    state = PUSH;
	  } else {
	    //i--;
	    state = WAIT;
	    W->iQ = *(poly[i]);
	    W->iP = *(top());
	  }
	}
      } else {
	state = WAIT;
	W->iP = *(top());
	W->iQ = y;
      }

      break;
    case WAIT:
      // i is incremented until v_i v_{i + 1} intersects W at point y
      // from the correct direction.  When that occurs, y is pushed on
      // the stack, and either push or pop is called depending on
      // whether polyAlphas[i + 1] >= polyAlphas[i] or vice versa,
      // respectively.

      IpeVector pt;
      IpeSegment currentSeg;
      do {
	i++;
	currentSeg.iP = *(poly[i]);
	currentSeg.iQ = *(poly[(i + 1) % npoints]);
      } while (!(currentSeg.Intersects(*W, pt) &&
		 TurnType(&currentSeg.iP, &(W->iQ), &currentSeg.iQ) > 0));

      // push pt on the stack, after computing alpha

      IpeVector diff = pt - *point;
      push (new IpeVector(pt), diff.Angle().Normalize(0.0));

      // now figure out what to set the state to.
      if (polyAlphas[i] < polyAlphas[(i + 1) % npoints]) {
	state = PUSH;
      } else {
	state = POP;
      }

      break;
    }
  }
 out:;
  delete W;
}

IpeVector **VisibilityStack::ToPolygon()
{
  IpeVector **output = new IpeVector*[stackSize];
  int i = 0;

  while (stackSize) {
    output[i] = new IpeVector(*top());
    pop();
    i++;
  }
  return output;
}

IpeVector **VisibilityPolygonIpelet::RenumberPoly(IpeVector **poly,
						  IpeVector *point,
						  int npoints)
{
  int i;
  int nearestPointBetween = -1;
  int reverse;
  double nearest = 1e42; /* FIXME: bounding circle */
  IpeVector *nearestPoint = new IpeVector();
  IpeVector **output = new IpeVector*[npoints + 2];

  IpeVector farPoint(1e42, point->iY); /* FIXME: bounding circle */
  IpeSegment horizSegment(*point, farPoint);

  nearestPoint->iY = point->iY;

  for (i = 0; i < npoints; i++) {
    int j = (i + 1) % npoints;
    IpeVector pt;
    IpeSegment seg(*(poly[i]), *(poly[j]));

    if (horizSegment.Intersects(seg, pt)) {
      if (pt.iX > point->iX && pt.iX < nearest) {
	nearest = pt.iX;
	nearestPointBetween = j;
	nearestPoint->iX = pt.iX;
      }
    }
  }

  output[0] = nearestPoint;
  if (poly[(nearestPointBetween - 1 + npoints) % npoints]->iY
      < poly[nearestPointBetween]->iY) {
    reverse = 1;
  } else {
    reverse = -1;
    nearestPointBetween = (nearestPointBetween - 1 + npoints) % npoints;
  }
  for (i = 0; i < npoints; i++) {
    output[i + 1] = new IpeVector(*(poly[(reverse * i + nearestPointBetween
					  + npoints) % npoints]));
  }
  output[npoints + 1] = new IpeVector(*nearestPoint);
  return output;

}

int TurnType (IpeVector *a, IpeVector *b, IpeVector *c)
{
  double area2 = (b->iX - a->iX) *
    (c->iY - a->iY) - (c->iX - a->iX) * (b->iY - a->iY);

  if (area2 > 0) {
    return 1;
  } else if (area2 < 0) {
    return -1;
  } else {
    return 0;
  }
}

VisibilityStack *VisibilityPolygonIpelet::FindVisibilityPolygon
(IpeVector **poly, IpeVector *point, int npoints)
{
  int i;
  IpeVector **renumbered = RenumberPoly (poly, point, npoints);
  npoints++;
  VisibilityStack *st = new VisibilityStack;

  for (i = 0; i < npoints - 1; i++) {
    delete poly[i];
  }
  delete [] poly;

  st->SetPoly (renumbered, npoints + 1);
  st->SetPoint (point);
  st->Run ();

  return st;
}

void VisibilityPolygonIpelet::Run(int, IpePage *page, IpeletHelper *helper)
{
  IpePage::iterator it;
  int length = 0;
  int have_point = 0;
  int i, j, k;

  for (it = page->begin(); it != page->end(); it++) {
    if (it->Select() && it->Object() && it->Object()->AsMark()) {
      have_point = 1;
    } else if (it->Select() && it->Object() && it->Object()->AsPath()) {
      IpePath *path = it->Object()->AsPath();
      for (j = 0; j < path->NumSubPaths(); j++) {
	if (path->SubPath(j)->Type() == IpeSubPath::ESegments) {
	  const IpeSubPath *sp = it->Object()->AsPath()->SubPath(j);
	  if (sp->AsSegs()) {
	    length = sp->AsSegs()->NumSegments() + 1;
	  }
	}
      }
    }
  }
  if (length < 2 || have_point == 0) {
    helper->Message("Too little selected");
    return;
  }
  IpeVector **points = new IpeVector*[length];
  IpeVector *point = NULL;
  i = 0;
  for (it = page->begin(); it != page->end(); it++) {
    if (it->Select() && it->Object()) {
      IpeMatrix m = it->Object()->Matrix();
      if (it->Object()->AsMark()) {
	point = new IpeVector(m * it->Object()->AsMark()->Position());
      } else if (it->Object()->AsPath()) {
	IpePath *path = it->Object()->AsPath();
	for (j = 0; j < path->NumSubPaths(); j++) {
	  if (path->SubPath(j)->Type() == IpeSubPath::ESegments) {
	    const IpeSegmentSubPath *sp = path->SubPath(j)->AsSegs();
	    for (k = 0; k < sp->NumSegments(); k++) {
	      points[i] = new IpeVector(m * sp->Segment(k).CP(0));
	      i++;
	    }
	    points[i] = new IpeVector(m * sp->Segment(k - 1).CP(1));
	    i++;
	  }
	}
      }
    }
  }

  VisibilityStack *output = FindVisibilityPolygon (points, point, length);
  delete point;

  length = output->Length();

  IpePath *path = new IpePath(helper->Attributes());
  IpeSegmentSubPath *sp = new IpeSegmentSubPath;

  IpeVector **newpoints = output->ToPolygon();

  for (i = 1; i < length; i++) {
    sp->AppendSegment (*(newpoints[i - 1]), *(newpoints[i]));
  }
  sp->SetClosed(true);
  path->AddSubPath(sp);
  page->push_back(IpePgObject(IpePgObject::EPrimary, helper->CurrentLayer(),
			      path));

  for (i = 0; i < length; i++) {
    delete newpoints[i];
  }
  delete [] newpoints;

  delete output;
}

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new VisibilityPolygonIpelet;
}
