// --------------------------------------------------------------------
// Ipelet for aligning objects
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipelet.h"
#include "ipepath.h"
#include "ipetext.h"
#include "ipepage.h"
#include "ipevisitor.h"

// --------------------------------------------------------------------

class AlignIpelet : public Ipelet {
public:
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  virtual int NumFunctions() const { return 8; }
  virtual const char *Label() const { return "Align"; }
  virtual const char *SubLabel(int function) const;
  virtual const char *KeySequence(int function) const;
  virtual void Run(int, IpePage *page, IpeletHelper *helper);
};

// --------------------------------------------------------------------

const char * const sublabel[] = {
  "Top",
  "Bottom",
  "Left",
  "Right",
  "Center",
  "H Center",
  "V Center",
  "Baseline"
};

const char * const shortcut[] = {
  "Shift+T", "Shift+B", "Shift+L", "Shift+R",
  "Shift+C", "Shift+H", "Shift+V", 0 };

const char *AlignIpelet::SubLabel(int function) const
{
  return sublabel[function];
}

const char *AlignIpelet::KeySequence(int function) const
{
  return shortcut[function];
}

// --------------------------------------------------------------------

void AlignIpelet::Run(int fn, IpePage *page, IpeletHelper *helper)
{
  IpePage::iterator it = page->PrimarySelection();
  if (it == page->end()) {
    helper->Message("Nothing selected");
    return;
  }
  IpeRect pbox = it->BBox();
  IpeVector pref = pbox.Min();
  if (it->Object()->AsText())
    pref = it->Object()->Matrix() * it->Object()->AsText()->Position();

  int count = 0;
  for (IpePage::iterator it1 = page->begin(); it1 != page->end(); ++it1) {
    if (it1->Select() == IpePgObject::ESecondary) {
      IpeRect box = it1->BBox();
      ++count;
      IpeVector v(0, 0);
      IpeVector ref = box.Min();
      if (it1->Object()->AsText())
	ref = it1->Object()->Matrix() * it1->Object()->AsText()->Position();
      switch (fn) {
      case 0: // top
	v.iY = pbox.Max().iY - box.Max().iY;
	break;
      case 1: // bottom
	v.iY = pbox.Min().iY - box.Min().iY;
	break;
      case 2: // left
	v.iX = pbox.Min().iX - box.Min().iX;
	break;
      case 3: // right
	v.iX = pbox.Max().iX - box.Max().iX;
	break;
      case 4: // center
	v.iX = 0.5 * ((pbox.Min().iX + pbox.Max().iX) -
		      (box.Min().iX + box.Max().iX));
	v.iY = 0.5 * ((pbox.Min().iY + pbox.Max().iY) -
		      (box.Min().iY + box.Max().iY));
	break;
      case 5: // h center
	v.iX = 0.5 * ((pbox.Min().iX + pbox.Max().iX) -
		      (box.Min().iX + box.Max().iX));
	break;
      case 6: // v center
	v.iY = 0.5 * ((pbox.Min().iY + pbox.Max().iY) -
		      (box.Min().iY + box.Max().iY));
	break;
      case 7: // baseline
	v.iY = pref.iY - ref.iY;
	break;
      }
      it1->Transform(IpeMatrix(v));
    }
  }
  if (!count)
    helper->Message("No objects to align");
}

// --------------------------------------------------------------------

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new AlignIpelet;
}

// --------------------------------------------------------------------
