// -*- C++ -*-
// --------------------------------------------------------------------
// Ipe style sheet
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPESTYLE_H
#define IPESTYLE_H

#include "ipebase.h"
#include "ipecolor.h"

class IpeObject;

// --------------------------------------------------------------------

class IPE_EXPORT IpeStyleSheet {
 public:
  IpeStyleSheet(IpeRepository *rep);
  ~IpeStyleSheet();

  static IpeStyleSheet *Standard(IpeRepository *rep);

  void AddTemplate(IpeAttribute name, const IpeObject *obj);
  const IpeObject *FindTemplate(IpeAttribute sym) const;

  void Add(IpeAttribute name, IpeAttribute value);
  IpeAttribute Find(IpeAttribute sym) const;

  void SaveAsXml(IpeStream &stream) const;
  void SaveCascadeAsXml(IpeStream &stream) const;

  void AllNames(IpeKind kind, IpeAttributeSeq &seq) const;
  void AllNames(IpeAttributeSeq *attr) const;

  //! Return whether this is the standard style sheet built into Ipe.
  inline bool IsStandard() const { return iStandard; }

  //! Return preamble of this style sheet (not including the cascade).
  inline IpeString Preamble() const { return iPreamble; }
  IpeString TotalPreamble() const;
  //! Set LaTeX preamble.
  inline void SetPreamble(const IpeString &str) { iPreamble = str; }

  void SetMargins(const IpeVector &tl, const IpeVector &br);
  void FindMargins(IpeVector &tl, IpeVector &br) const;

  void SetShading(const IpeShading &s);
  IpeShading FindShading() const;

  void AddCMap(IpeString s);
  void AllCMaps(std::vector<IpeString> &seq) const;

  inline const IpeStyleSheet *Cascade() const;
  inline IpeStyleSheet *Cascade();

  void SetCascade(IpeStyleSheet *sheet);

  //! Return name of style sheet.
  inline IpeString Name() const { return iName; }
  //! Set name of style sheet.
  inline void SetName(const IpeString &name) { iName = name; }

  //! Return attribute repository.
  inline IpeRepository *Repository() { return iRepository; }
  //! Return attribute repository (const version).
  inline const IpeRepository *Repository() const { return iRepository; }

private:
  IpeStyleSheet(const IpeStyleSheet &rhs);
  IpeStyleSheet &operator=(const IpeStyleSheet &rhs);
 private:
  typedef std::map<int, const IpeObject *> TemplateMap;
  typedef std::map<int, IpeAttribute> Map;

  bool iStandard;
  IpeString iName;
  IpeStyleSheet *iCascade;
  TemplateMap iTemplates;
  Map iMap;
  IpeString iPreamble;
  IpeVector iTLMargin, iBRMargin;
  IpeRepository *iRepository;

  IpeShading iShading;

  std::vector<IpeString> iCMaps;
};

// --------------------------------------------------------------------

//! Return next sheet in cascade (const version).
inline const IpeStyleSheet *IpeStyleSheet::Cascade() const
{
  return iCascade;
}

//! Return the next level style sheet.
inline IpeStyleSheet *IpeStyleSheet::Cascade()
{
  return iCascade;
}

// --------------------------------------------------------------------
#endif
