/*
 *  Hamlib CI-V backend - main header
 *  Copyright (c) 2000-2015 by Stephane Fillod
 *
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef _ICOM_H
#define _ICOM_H 1

#include "hamlib/rig.h"
#include "cal.h"
#include "tones.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#define BACKEND_VER "0.7"

/*
 * defines used by comp_cal_str in rig.c
 * STR_CAL_LENGTH is the length of the S Meter calibration table
 * STR_CAL_S0 is the value in dB of the lowest value (not even in table)
 * MULTIB_SUBCMD allows the dsp rigs ie pro models to use multibyte subcommands for all the extra
 * parameters and levels.
 */
#define STR_CAL_LENGTH 16
#define STR_CAL_S0 -54
#define MULTIB_SUBCMD

/*
 * minimal channel caps.
 * If your rig has better/lesser, don't modify this define but clone it,
 * so you don't change other rigs.
 */
#define IC_MIN_MEM_CAP {	\
	.freq = 1,	\
	.mode = 1,	\
	.width = 1,	\
}

/*
 * common channel caps.
 * If your rig has better/lesser, don't modify this define but clone it,
 * so you don't change other rigs.
 */
#define IC_MEM_CAP {	\
	.freq = 1,	\
	.mode = 1,	\
	.width = 1,	\
	.tx_freq = 1,	\
	.tx_mode = 1,	\
	.tx_width = 1,	\
	.rptr_offs = 1,	\
}

/*
 * S-Meter data for uncalibrated rigs
 */
#define UNKNOWN_IC_STR_CAL { 2, {{ 0, -60}, { 255, 60}} }


struct ts_sc_list {
	shortfreq_t ts;	/* tuning step */
	unsigned char sc;	/* sub command */
};

/**
 * \brief Pipelined tuning state data structure.
 */
typedef struct rig_pltstate {
  freq_t freq;
  freq_t next_freq;
  rmode_t mode;
  rmode_t next_mode;
  pbwidth_t width;
  pbwidth_t next_width;
  struct timeval timer_start;
  struct timeval timer_current;
  int usleep_time; /* dependent on radio module & serial data rate */
} pltstate_t;

struct icom_priv_caps {
	unsigned char re_civ_addr;	/* the remote dlft equipment's CI-V address*/
	int civ_731_mode; /* Off: freqs on 10 digits, On: freqs on 8 digits */
	int no_xchg; /* Off: use VFO XCHG to set other VFO, On: use set VFO to set other VFO */
	const struct ts_sc_list *ts_sc_list;
	int settle_time; /*!< Receiver settle time, in ms */
	int (*r2i_mode)(RIG *rig, rmode_t mode, pbwidth_t width,
		unsigned char *md, signed char *pd); /*< backend specific code
						       to convert bandwidth and
						       mode to cmd tokens */
	void (*i2r_mode)(RIG *rig, unsigned char md, int pd,
		rmode_t *mode, pbwidth_t *width);    /*< backend specific code
						       to convert response
						       tokens to bandwidth and
						       mode */
	int serial_full_duplex; /*!< Whether RXD&TXD are not tied together */
};


struct icom_priv_data {
	unsigned char re_civ_addr;	/* the remote equipment's CI-V address*/
	int civ_731_mode; /* Off: freqs on 10 digits, On: freqs on 8 digits */
	int no_xchg; /* Off: use VFO XCHG to set other VFO, On: use set VFO to set other VFO */
	pltstate_t *pltstate;	/* only on optoscan */
};

extern const struct ts_sc_list r8500_ts_sc_list[];
extern const struct ts_sc_list ic737_ts_sc_list[];
extern const struct ts_sc_list r75_ts_sc_list[];
extern const struct ts_sc_list r7100_ts_sc_list[];
extern const struct ts_sc_list r9000_ts_sc_list[];
extern const struct ts_sc_list r9500_ts_sc_list[];
extern const struct ts_sc_list ic756_ts_sc_list[];
extern const struct ts_sc_list ic756pro_ts_sc_list[];
extern const struct ts_sc_list ic706_ts_sc_list[];
extern const struct ts_sc_list ic7000_ts_sc_list[];
extern const struct ts_sc_list ic7100_ts_sc_list[];
extern const struct ts_sc_list ic7200_ts_sc_list[];
extern const struct ts_sc_list ic910_ts_sc_list[];
extern const struct ts_sc_list ic718_ts_sc_list[];

extern const pbwidth_t rtty_fil[];	/* rtty filter passband width; available on 746pro and 756pro rigs */

pbwidth_t icom_get_dsp_flt(RIG *rig, rmode_t mode);

int icom_init(RIG *rig);
int icom_cleanup(RIG *rig);
int icom_set_freq(RIG *rig, vfo_t vfo, freq_t freq);
int icom_get_freq(RIG *rig, vfo_t vfo, freq_t *freq);
int icom_set_rit(RIG *rig, vfo_t vfo, shortfreq_t rit);
int icom_set_mode_with_data(RIG *rig, vfo_t vfo, rmode_t mode, pbwidth_t width);
int icom_set_mode(RIG *rig, vfo_t vfo, rmode_t mode, pbwidth_t width);
int icom_get_mode_with_data(RIG *rig, vfo_t vfo, rmode_t *mode, pbwidth_t *width);
int icom_get_mode(RIG *rig, vfo_t vfo, rmode_t *mode, pbwidth_t *width);
int icom_set_vfo(RIG *rig, vfo_t vfo);
int icom_set_rptr_shift(RIG *rig, vfo_t vfo, rptr_shift_t rptr_shift);
int icom_get_rptr_shift(RIG *rig, vfo_t vfo, rptr_shift_t *rptr_shift);
int icom_set_rptr_offs(RIG *rig, vfo_t vfo, shortfreq_t rptr_offs);
int icom_get_rptr_offs(RIG *rig, vfo_t vfo, shortfreq_t *rptr_offs);
int icom_set_split_freq(RIG *rig, vfo_t vfo, freq_t tx_freq);
int icom_get_split_freq(RIG *rig, vfo_t vfo, freq_t *tx_freq);
int icom_set_split_mode(RIG *rig, vfo_t vfo, rmode_t tx_mode, pbwidth_t tx_width);
int icom_get_split_mode(RIG *rig, vfo_t vfo, rmode_t *tx_mode, pbwidth_t *tx_width);
int icom_set_split_vfo(RIG *rig, vfo_t vfo, split_t split, vfo_t tx_vfo);
int icom_get_split_vfo(RIG *rig, vfo_t vfo, split_t *split, vfo_t *tx_vfo);
int icom_mem_get_split_vfo(RIG *rig, vfo_t vfo, split_t *split, vfo_t *tx_vfo);
int icom_set_ts(RIG *rig, vfo_t vfo, shortfreq_t ts);
int icom_get_ts(RIG *rig, vfo_t vfo, shortfreq_t *ts);
int icom_set_ptt(RIG *rig, vfo_t vfo, ptt_t ptt);
int icom_get_ptt(RIG *rig, vfo_t vfo, ptt_t *ptt);
int icom_get_dcd(RIG *rig, vfo_t vfo, dcd_t *dcd);
int icom_set_ctcss_tone(RIG *rig, vfo_t vfo, tone_t tone);
int icom_get_ctcss_tone(RIG *rig, vfo_t vfo, tone_t *tone);
int icom_set_ctcss_sql(RIG *rig, vfo_t vfo, tone_t tone);
int icom_get_ctcss_sql(RIG *rig, vfo_t vfo, tone_t *tone);
int icom_set_dcs_code(RIG *rig, vfo_t vfo, tone_t code);
int icom_get_dcs_code(RIG *rig, vfo_t vfo, tone_t *code);
int icom_set_bank(RIG *rig, vfo_t vfo, int bank);
int icom_set_mem(RIG *rig, vfo_t vfo, int ch);
int icom_vfo_op(RIG *rig, vfo_t vfo, vfo_op_t op);
int icom_scan(RIG *rig, vfo_t vfo, scan_t scan, int ch);
int icom_set_level(RIG *rig, vfo_t vfo, setting_t level, value_t val);
int icom_get_level(RIG *rig, vfo_t vfo, setting_t level, value_t *val);
int icom_set_func(RIG *rig, vfo_t vfo, setting_t func, int status);
int icom_get_func(RIG *rig, vfo_t vfo, setting_t func, int *status);
int icom_set_parm(RIG *rig, setting_t parm, value_t val);
int icom_get_parm(RIG *rig, setting_t parm, value_t *val);
int icom_set_conf(RIG *rig, token_t token, const char *val);
int icom_get_conf(RIG *rig, token_t token, char *val);
int icom_set_powerstat(RIG *rig, powerstat_t status);
int icom_get_powerstat(RIG *rig, powerstat_t *status);
int icom_set_ant(RIG * rig, vfo_t vfo, ant_t ant);
int icom_get_ant(RIG * rig, vfo_t vfo, ant_t *ant);
int icom_decode_event(RIG *rig);

extern const struct confparams icom_cfg_params[];

extern const struct rig_caps ic703_caps;
extern const struct rig_caps ic706_caps;
extern const struct rig_caps ic706mkii_caps;
extern const struct rig_caps ic706mkiig_caps;
extern const struct rig_caps ic707_caps;
extern const struct rig_caps ic718_caps;
extern const struct rig_caps ic725_caps;
extern const struct rig_caps ic726_caps;
extern const struct rig_caps ic728_caps;
extern const struct rig_caps ic735_caps;
extern const struct rig_caps ic736_caps;
extern const struct rig_caps ic737_caps;
extern const struct rig_caps ic738_caps;
extern const struct rig_caps ic746_caps;
extern const struct rig_caps ic7410_caps;
extern const struct rig_caps ic746pro_caps;
extern const struct rig_caps ic756_caps;
extern const struct rig_caps ic756pro_caps;
extern const struct rig_caps ic756pro2_caps;
extern const struct rig_caps ic756pro3_caps;
extern const struct rig_caps ic751_caps;
extern const struct rig_caps ic7600_caps;
extern const struct rig_caps ic761_caps;
extern const struct rig_caps ic765_caps;
extern const struct rig_caps ic7700_caps;
extern const struct rig_caps ic775_caps;
extern const struct rig_caps ic78_caps;
extern const struct rig_caps ic7800_caps;
extern const struct rig_caps ic7000_caps;
extern const struct rig_caps ic7100_caps;
extern const struct rig_caps ic7200_caps;
extern const struct rig_caps ic781_caps;
extern const struct rig_caps ic820h_caps;
extern const struct rig_caps ic821h_caps;
extern const struct rig_caps ic910_caps;
extern const struct rig_caps ic9100_caps;
extern const struct rig_caps ic970_caps;
extern const struct rig_caps icrx7_caps;
extern const struct rig_caps icr10_caps;
extern const struct rig_caps icr20_caps;
extern const struct rig_caps icr71_caps;
extern const struct rig_caps icr72_caps;
extern const struct rig_caps icr75_caps;
extern const struct rig_caps icr7000_caps;
extern const struct rig_caps icr7100_caps;
extern const struct rig_caps icr8500_caps;
extern const struct rig_caps icr9000_caps;
extern const struct rig_caps icr9500_caps;
extern const struct rig_caps ic271_caps;
extern const struct rig_caps ic275_caps;
extern const struct rig_caps ic471_caps;
extern const struct rig_caps ic475_caps;
extern const struct rig_caps ic1275_caps;

extern const struct rig_caps omnivip_caps;
extern const struct rig_caps delta2_caps;

extern const struct rig_caps os456_caps;
extern const struct rig_caps os535_caps;

extern const struct rig_caps ic92d_caps;
extern const struct rig_caps id1_caps;
extern const struct rig_caps id5100_caps;

#endif /* _ICOM_H */
