/*
	$Id: resourcetype_font.cpp,v 1.1.1.1 2000/04/09 12:18:00 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Font resource type

*/

#include "Core/precomp.h"
#include <API/Core/IOData/inputsource_provider.h>
#include <API/Core/IOData/outputsource.h>
#include <API/Core/Resources/resource.h>
#include <API/Core/Resources/resource_manager.h>
#include <API/Core/Resources/resourceoptions.h>
#include <API/Core/Resources/resourcetype.h>
#include <API/Core/System/error.h>
#include <API/Core/System/clanstring.h>
#include <API/Core/Display/palette.h>
#include <API/Core/Display/surfaceprovider.h>
#include <API/Core/SurfaceProviders/provider_pcx.h>
#include <API/Core/SurfaceProviders/provider_targa.h>
#include <API/Core/Font/font_description.h>
#include <Core/Font/resourcetype_font.h>
#include <API/Core/System/cl_assert.h>

/******************************************************************************
						class CL_Res_Font
******************************************************************************/

CL_Res_Font::CL_Res_Font() : CL_ResourceType("font")
{
}

CL_Resource *CL_Res_Font::create_from_location(
	std::string name,
	std::string location,
	CL_ResourceOptions *options,
	CL_ResourceManager *parent)
{
	CL_String ext = CL_String(location).right(4);
	ext.to_lower();

	bool is_font_type = false;
	if (options->exists("type")) 
	{
		if (options->get_value("type") != "font") return NULL;
		is_font_type = true;
	}

	if (is_font_type ||
		options->exists("font") || ext == ".pcx" || ext == ".tga")
	{
		return new CL_Font_Resource(
			name,
			location,
			options,
			parent);
	}

	return NULL;
}

CL_Resource *CL_Res_Font::create_from_serialization(
	std::string name,
	CL_ResourceManager *parent)
{
	return new CL_Font_Resource(
		name,
		parent);
}
/******************************************************************************
						class CL_Font_Resource
******************************************************************************/

CL_Font_Resource::CL_Font_Resource(
	std::string name,
	std::string location,
	CL_ResourceOptions *options,
	CL_ResourceManager *parent)
: CL_Resource("font", name)
{
	this->location = location;
	this->options = options;
	this->parent = parent;
	from_datafile = false;
	load_count = 0;

	font_desc = NULL;
}

CL_Font_Resource::CL_Font_Resource(
	std::string name,
	CL_ResourceManager *parent)
: CL_Resource("font", name)
{
	location = "";
	options = NULL;
	this->parent = parent;
	from_datafile = true;
	load_count = 0;

	font_desc = NULL;
}

CL_Font_Resource::~CL_Font_Resource()
{
	delete options;
//	delete font_desc;
}

void CL_Font_Resource::serialize_save(CL_OutputSource *output)
{
	load();

	output->write_int32(font_desc->space_len);
	output->write_int32(font_desc->subtract_width);
	output->write_string(font_desc->letters.c_str());
/*
	output->write_short16(font_desc->letters.length()+1);
	output->write(font_desc->letters.c_str(), font_desc->letters.length()+1);
*/	
	unsigned int len = font_desc->letters.length();
	if (len > font_desc->letter_providers.size())
	{
		CL_String err;
		err << "Font loading error - ";
		err << "Letters expected:  " << len;
		err << ", letters available: " << font_desc->letter_providers.size();
		throw CL_Error(err.get_string());
	}
	else if (len < font_desc->letter_providers.size())
	{
		std::cout << "ClanLib Font Warning: Font letters do not match!" << std::endl;
		std::cout << "ClanLib Font Warning: Letters expected:  " << len << std::endl;
		std::cout << "ClanLib Font Warning: Letters available: "
			<< font_desc->letter_providers.size() << std::endl;
	}
	
	if (len > 0)
	{
		font_desc->letter_providers[0]->lock();
	}
	for(unsigned int i=0; i<len; i++)
	{
		CL_SurfaceProvider *provider = font_desc->letter_providers[i];
		cl_assert(provider != NULL);

		provider->lock();

		int width = provider->get_width();
		int height = provider->get_height();
		int no_sprs = provider->get_num_frames();
		int bytes_per_pixel = (provider->get_depth()+7)/8;
		int transcol = provider->uses_src_colorkey() ? (int)provider->get_src_colorkey() : -1;
//		int format = provider->get_pixel_format();
		int red_mask = provider->get_red_mask();
		int green_mask = provider->get_green_mask();
		int blue_mask = provider->get_blue_mask();
		int alpha_mask = provider->get_alpha_mask();
		int translate_x = provider->get_translate_x();
		int translate_y = provider->get_translate_y();
		int pitch = provider->get_pitch();

		CL_Palette *pal = provider->get_palette();
		unsigned char *data = (unsigned char *) provider->get_data();
		cl_assert(data != NULL);

		output->write_int32(width);
		output->write_int32(height);
		output->write_int32(no_sprs);
		output->write_int32(transcol);
		
		char palette_bool = (pal != NULL) ? 1 : 0;
		output->write_char8(palette_bool);
		
		if (palette_bool) output->write(pal->palette, 256*3);

		char indexed_bool = (provider->is_indexed()) ? 1 : 0;
		output->write_char8(indexed_bool);
		
//		output->write_int32(format);
		output->write_int32(red_mask);
		output->write_int32(green_mask);
		output->write_int32(blue_mask);
		output->write_int32(alpha_mask);
		output->write_int32(provider->get_depth());

		if (transcol == -1)
		{
			int bytes_pr_line = width*bytes_per_pixel;
			for (int y=0; y<height*no_sprs; y++)
			{
				int img_y = y+translate_y;

				output->write(
					data+translate_x*bytes_per_pixel+img_y*pitch,
					bytes_pr_line);
			}
		}
		else
		{
			for (int y=0; y<height*no_sprs; y++)
			{
				int img_y = y+translate_y;

				output->write(
					data+translate_x*bytes_per_pixel+img_y*pitch,
					bytes_per_pixel*width);
			}
		}

		provider->unlock();
	}
	if (len > 0)
	{
		font_desc->letter_providers[0]->unlock();
	}
	
	unload();
}

void CL_Font_Resource::load()
{
	load_count++;
	if (font_desc != NULL) return; // already loaded.

	if (from_datafile) load_from_datafile();
	else load_from_file();
}

void CL_Font_Resource::unload()
{
	load_count--;
	if (load_count == 0)
	{
		delete font_desc;
		font_desc = NULL;
	}
}

void CL_Font_Resource::load_from_datafile()
{
	CL_InputSourceProvider *provider = parent->get_resource_provider();

	font_desc = new CL_Font_Description(
		provider->open_source(get_name().c_str()));
}

void CL_Font_Resource::load_from_file()
{
	CL_String ext = CL_String(location).right(4);
	ext.to_lower();
	
	if (ext == ".tga" || options->exists("alphafont"))
	{
		if (options->exists("spacelen") == false)
			throw CL_Error("Font missing spacelen option.");

		if (options->exists("trans_limit") == false)
			throw CL_Error("Font missing trans_limit option.");

		if (options->exists("subtract_width") == false)
			throw CL_Error("Font missing subtract_width option.");

		if (options->exists("letters") == false)
			throw CL_Error("Font missing letters option.");

		int space_len = CL_String(options->get_value("spacelen")).get_as_int();
		float trans_limit = CL_String(options->get_value("trans_limit")).get_as_float();
		int subtract_width = CL_String(options->get_value("subtract_width")).get_as_int();
	
		font_desc = new CL_Font_Description(
			new CL_TargaProvider(location, NULL),
			trans_limit,
			space_len,
			subtract_width,
			options->get_value("letters").c_str());
	}
	else
	{
		if (options->exists("x") == false)
			throw CL_Error("Font missing x coordinate option.");

		if (options->exists("y") == false)
			throw CL_Error("Font missing y coordinate option.");

		if (options->exists("spacelen") == false)
			throw CL_Error("Font missing spacelen option.");

		if (options->exists("letters") == false)
			throw CL_Error("Font missing letters option.");

		int x = CL_String(options->get_value("x")).get_as_int();
		int y = CL_String(options->get_value("y")).get_as_int();
		int space_len = CL_String(options->get_value("spacelen")).get_as_int();
		int subtract_width = 0;
		if (options->exists("subtract_width"))
			subtract_width = CL_String(
				options->get_value("subtract_width")).get_as_int();

		int num_tcols = 1;
		int *tcols = NULL;
		if (num_tcols > 0)
		{
			tcols = new int[num_tcols];
			tcols[0] = 0;
		}

		font_desc = new CL_Font_Description(
			new CL_PCXProvider(location, NULL),
			x, y,
			tcols, num_tcols,
			space_len,
			subtract_width,
			options->get_value("letters").c_str());
	}
}
