/*
	$Id: restype_surface.cpp,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/
#include <Core/precomp.h>

#include <API/Core/Display/palette.h>
#include <API/Core/Display/surface.h>
#include <API/Core/Display/surfaceprovider.h>
#include <API/Core/Resources/resource.h>
#include <API/Core/Resources/resource_manager.h>
#include <API/Core/Resources/resourceoption.h>
#include <API/Core/Resources/resourceoptions.h>
#include <API/Core/Resources/resourcetype.h>
#include <API/Core/SurfaceProviders/sprite.h>
#include <API/Core/SurfaceProviders/sprite_subarray_provider.h>
#include <API/Core/SurfaceProviders/sprite_subsection_provider.h>
#include <API/Core/SurfaceProviders/provider_targa.h>
#include <API/Core/SurfaceProviders/provider_pcx.h>
#include <API/Core/SurfaceProviders/provider_bmp.h>
#include <API/Core/SurfaceProviders/provider_ppm.h>
#include <API/Core/IOData/outputsource.h>
#include <API/Core/System/cl_assert.h>
#include <API/Core/System/error.h>
#include <API/Core/System/clanstring.h>
#include <Core/Display/Generic/restype_surface.h>

#include "restype_surface.h"

/******************************************************************************
						class CL_ResourceType_Surface
******************************************************************************/

std::list<CL_ResourceSource_Surface*> CL_Res_Surface::surface_sources;

CL_Res_Surface::CL_Res_Surface() : CL_ResourceType("surface")
{
}

CL_Resource *CL_Res_Surface::create_from_location(
	std::string name,
	std::string location,
	CL_ResourceOptions *options,
	CL_ResourceManager *parent)
{
	CL_String ext = CL_String(name).right(4);
	ext.to_lower();

	bool is_surface_type = false;
	if (options->exists("type"))
	{
		if (options->get_value("type") != "surface") return NULL;
		is_surface_type = true;
	}

	for (
		std::list<CL_ResourceSource_Surface*>::iterator it = surface_sources.begin();
		it != surface_sources.end();
		it++)
	{
		if ((*it)->can_create(ext, options))
		{
			is_surface_type = true;
			break;
		}
	}
	
	if (is_surface_type)
	{
		return new CL_SurfaceFileResource(name, location, options, parent);
	}

	return NULL;
}

CL_Resource *CL_Res_Surface::create_from_serialization(
	std::string name,
	CL_ResourceManager *parent)
{
	return new CL_SurfaceDatafileResource(name, parent);
}

/******************************************************************************
						class CL_WritableSurfaceResource
******************************************************************************/

CL_WritableSurfaceResource::CL_WritableSurfaceResource(std::string name)
: CL_Resource("surface", name)
{
	loaded = false;
	loaded_surface = NULL;
	load_count = 0;
}

void CL_WritableSurfaceResource::serialize_save(CL_OutputSource *output)
{
	CL_SurfaceProvider *provider = create_provider();
	cl_assert(provider != NULL);

	provider->lock();

	int width = provider->get_width();
	int height = provider->get_height();
	int no_sprs = provider->get_num_frames();
	int bytes_per_pixel = (provider->get_depth()+7)/8;
	int transcol = provider->uses_src_colorkey() ? (int) provider->get_src_colorkey() : -1;
	int red_mask = provider->get_red_mask();
	int green_mask = provider->get_green_mask();
	int blue_mask = provider->get_blue_mask();
	int alpha_mask = provider->get_alpha_mask();
	int depth = provider->get_depth();
//	int format = provider->get_pixel_format();
	int translate_x = provider->get_translate_x();
	int translate_y = provider->get_translate_y();
	int pitch = provider->get_pitch();

	CL_Palette *pal = provider->get_palette();
	unsigned char *data = (unsigned char *) provider->get_data();
	cl_assert(data != NULL);

	output->write_int32(width);
	output->write_int32(height);
	output->write_int32(no_sprs);
	output->write_int32(transcol);
	
	char palette_bool = (pal != NULL) ? 1 : 0;
	output->write_char8(palette_bool);
	
	if (palette_bool) output->write(pal->palette, 256*3);

	char indexed_bool = (provider->is_indexed()) ? 1 : 0;
	output->write_char8(indexed_bool);
	
//	output->write_int32(format);
	output->write_int32(red_mask);
	output->write_int32(green_mask);
	output->write_int32(blue_mask);
	output->write_int32(alpha_mask);
	output->write_int32(depth);

	if (!provider->uses_src_colorkey())
	{
		int bytes_pr_line = width*bytes_per_pixel;
		for (int y=0; y<height*no_sprs; y++)
		{
			int img_y = y+translate_y;

			output->write(
				data+translate_x*bytes_per_pixel+img_y*pitch,
				bytes_pr_line);
		}
	}
	else
	{
		for (int y=0; y<height*no_sprs; y++)
		{
			int img_y = y+translate_y;

			output->write(data+translate_x*bytes_per_pixel+img_y*pitch, bytes_per_pixel*width);
		}
	}

	provider->unlock();

	delete provider;
}

void CL_WritableSurfaceResource::load()
{
	load_count++;
	if (loaded) return;

	CL_SurfaceProvider *provider = create_provider();

	cl_assert(provider != NULL);
	
	loaded_surface = CL_Surface::create(provider, true);
	loaded = true;
}

void CL_WritableSurfaceResource::unload()
{
	load_count--;
	if (load_count == 0)
	{
		delete loaded_surface;
		loaded = false;
	}
}

CL_Surface *CL_WritableSurfaceResource::create_surface()
{
	if (!loaded) load();

	return loaded_surface;
}

/******************************************************************************
						class CL_SurfaceFileResource
******************************************************************************/

CL_SurfaceFileResource::CL_SurfaceFileResource(
	std::string name,
	std::string location,
	CL_ResourceOptions *_options,
	CL_ResourceManager *_parent)
: CL_WritableSurfaceResource(name)
{
	surface_location = location;
	options = _options;
	parent = _parent;
}

CL_SurfaceFileResource::~CL_SurfaceFileResource()
{
	delete options;
}

CL_SurfaceProvider *CL_SurfaceFileResource::create_provider()
{
	CL_SurfaceProvider *ret_provider = NULL;

	CL_String ext = CL_String(surface_location).right(4);
	ext.to_lower();

	for (
		std::list<CL_ResourceSource_Surface*>::iterator it = CL_Res_Surface::surface_sources.begin();
		it != CL_Res_Surface::surface_sources.end();
		it++)
	{
		if (options->exists((*it)->get_name()))
		{
			if ((*it)->can_create(ext, options))
			{
				ret_provider = (*it)->create(surface_location, options, parent);
				break;
			}
			else
			{
				CL_String err;
				err << "Surface provider " << (*it)->get_name() << " cannot handle input format!";
				throw CL_Error(err.get_string());
			}
		}
	}
	
	if (ret_provider == NULL)
	{
		for (
			std::list<CL_ResourceSource_Surface*>::iterator it = CL_Res_Surface::surface_sources.begin();
			it != CL_Res_Surface::surface_sources.end();
			it++)
		{
			if ((*it)->can_create(ext, options))
			{
				ret_provider = (*it)->create(surface_location, options, parent);
				break;
			}
		}
	}
	if (ret_provider == NULL) return NULL;

/*
	if (options->exists("tga") ||  ext == ".tga")
	{
		ret_provider = new CL_TargaProvider(surface_location, NULL);
	}
	else if (options->exists("pcx") || ext == ".pcx")
	{
		ret_provider = new CL_PCXProvider(surface_location, NULL);
	}
	else if (options->exists("png") || ext == ".png")
	{
		ret_provider = new CL_PNGProvider(surface_location, NULL);
	}
	else
	{
		return NULL;
	}
*/

	if (options->exists("x") && options->exists("y") &&
		options->exists("width") && options->exists("height"))
	{
		if (options->exists("array"))
		{
			int sep = options->get_value("array").find('x');
			if (sep == -1) return NULL;

			ret_provider = new CL_SpriteSubarrayProvider(
				ret_provider,
				CL_String(options->get_value("x")).get_as_int(),
				CL_String(options->get_value("y")).get_as_int(),
				CL_String(options->get_value("width")).get_as_int(),
				CL_String(options->get_value("height")).get_as_int(),
				CL_String(options->get_value("array")).mid(0, sep).get_as_int(),
				CL_String(options->get_value("array")).mid(sep+1).get_as_int());
		}
		else
		{
			ret_provider = new CL_SpriteSubsectionProvider(
				ret_provider,
				CL_String(options->get_value("x")).get_as_int(),
				CL_String(options->get_value("y")).get_as_int(),
				CL_String(options->get_value("width")).get_as_int(),
				CL_String(options->get_value("height")).get_as_int());
		}
	}

	CL_ResourceOption *tcol_option = options->get_option("tcol");
	if (tcol_option != NULL)
	{
		if (tcol_option->multi_valued())
		{
			std::list<std::string> *tcol_list = tcol_option->get_all_values();
			unsigned int *trans_table = new unsigned int[tcol_list->size()];

			int pos = 0;
			for (
				std::list<std::string>::iterator it = tcol_list->begin();
				it != tcol_list->end();
				it++)
			{
				int tval = CL_String(*it).get_as_int();
				{
					trans_table[pos++] = tval;
				}
			}

			ret_provider = new CL_MaskTranscolProvider(
					ret_provider,
					true,
					trans_table,
					tcol_list->size());
			delete[] trans_table;
		}
		else
		{
			CL_String tcol = tcol_option->get_value();
			int tval = tcol.get_as_int();
			if (tval >= 0 && tval < 256)
			{
				unsigned int v = tval;
				ret_provider = new CL_MaskTranscolProvider(
						ret_provider,
						true,
						&v,
						1);
			}
		}
	}

	return ret_provider;
}

/******************************************************************************
						class CL_SurfaceDatafileResource
******************************************************************************/

CL_SurfaceDatafileResource::CL_SurfaceDatafileResource(
	std::string name,
	CL_ResourceManager *_parent)
: CL_WritableSurfaceResource(name)
{
	surface_location = "";
	options = NULL;
	parent = _parent;
}

CL_SurfaceDatafileResource::~CL_SurfaceDatafileResource()
{
	delete options;
}

CL_SurfaceProvider *CL_SurfaceDatafileResource::create_provider()
{
	return new CL_SpriteProvider(get_name().c_str(), parent->get_resource_provider());
}

/******************************************************************************
						class CL_MaskTranscolProvider
******************************************************************************/

CL_MaskTranscolProvider::CL_MaskTranscolProvider(
	CL_SurfaceProvider *_parent,
	bool _delete_provider,
	const unsigned int * _mask_indices,
	int _num_mask_indices)
{
	parent = _parent;
	cl_assert(parent != NULL);
	//cl_assert(parent->get_depth() == 8);

	delete_provider = _delete_provider;
	num_mask_indices = _num_mask_indices;
	mask_indices = new unsigned int[num_mask_indices];
	memcpy(mask_indices, _mask_indices, num_mask_indices*sizeof(unsigned int));
}

CL_MaskTranscolProvider::~CL_MaskTranscolProvider()
{
	if (delete_provider) delete parent;
	delete[] mask_indices;
}

unsigned int CL_MaskTranscolProvider::get_pitch() const
{
	return parent->get_pitch();
}

int CL_MaskTranscolProvider::get_translate_x() const
{
	return parent->get_translate_x();
}

int CL_MaskTranscolProvider::get_translate_y() const
{
	return parent->get_translate_y();
}

unsigned int CL_MaskTranscolProvider::get_depth() const
{
	return parent->get_depth();
}

bool CL_MaskTranscolProvider::is_indexed() const
{
	return parent->is_indexed();
}

unsigned int CL_MaskTranscolProvider::get_width() const
{
	return parent->get_width();
}

unsigned int CL_MaskTranscolProvider::get_height() const
{
	return parent->get_height();
}

unsigned int CL_MaskTranscolProvider::get_num_frames() const
{
	return parent->get_num_frames();
}

unsigned int CL_MaskTranscolProvider::get_red_mask() const
{
	return parent->get_red_mask();
}

unsigned int CL_MaskTranscolProvider::get_green_mask() const
{
	return parent->get_green_mask();
}

unsigned int CL_MaskTranscolProvider::get_blue_mask() const
{
	return parent->get_blue_mask();
}

unsigned int CL_MaskTranscolProvider::get_alpha_mask() const
{
	return parent->get_alpha_mask();
}

CL_Palette *CL_MaskTranscolProvider::get_palette() const
{
	return parent->get_palette();
}

unsigned int CL_MaskTranscolProvider::get_src_colorkey() const
{
	return mask_indices[0];
}

bool CL_MaskTranscolProvider::uses_src_colorkey() const
{
//	return parent->uses_src_colorkey();
	return true;
}

void *CL_MaskTranscolProvider::get_data() const
{
	return parent->get_data();
}

void CL_MaskTranscolProvider::perform_lock()
{
	parent->lock();

	switch (get_depth())
	{
	case 8:
		{
			int i;
			bool trans_table[256];
			for (i=0;i<256;i++) trans_table[i] = false;
			if (parent->uses_src_colorkey()) trans_table[parent->get_src_colorkey()] = true;
			for (i=0;i<num_mask_indices;i++) trans_table[mask_indices[i]] = true;

			unsigned char *pdata = (unsigned char *) parent->get_data();
			for (unsigned int s=0;s<parent->get_num_frames();s++)
			{
				for (unsigned int y=0;y<parent->get_height();y++)
				{
					for (unsigned int x=0;x<parent->get_width();x++)
					{
						if (trans_table[*pdata]) *pdata = mask_indices[0];
						pdata++;
					}
					pdata += parent->get_pitch()-parent->get_width();
				}
			}
		}
		break;

	case 15:
	case 16:
		{
			unsigned int frames = parent->get_num_frames();
			unsigned int height = parent->get_height();
			unsigned int width = parent->get_width();
			unsigned int pitch = parent->get_pitch();

			for (unsigned int s=0;s<frames;s++)
			{
				for (unsigned int y=0;y<height;y++)
				{
					unsigned char *data =
						((unsigned char *) parent->get_data()) + (y+s*height)*pitch;

					unsigned short *pdata = (unsigned short *) data;

					for (unsigned int x=0;x<width;x++)
					{
						for (int i=0; i<num_mask_indices; i++)
						{
							if (pdata[x] == mask_indices[i])
							{
								// Set the whole pixel to 0 (Even if we should only set the alpha value
								pdata[x] = 0;
							}
						}
					}
				}
			}
		}
		break;

	case 24:
		cl_assert(false); // not implemented in 24 bpp. That depth SUCKS! :-(
		break;

	case 32:
		{
			unsigned int frames = parent->get_num_frames();
			unsigned int height = parent->get_height();
			unsigned int width = parent->get_width();
			unsigned int pitch = parent->get_pitch();

			for (unsigned int s=0;s<frames;s++)
			{
				for (unsigned int y=0;y<height;y++)
				{
					unsigned char *data =
						((unsigned char *) parent->get_data()) + (y+s*height)*pitch;

					unsigned int *pdata = (unsigned int *) data;

					for (unsigned int x=0;x<width;x++)
					{
						for (int i=0; i<num_mask_indices; i++)
						{
							if (pdata[x] == mask_indices[i])
							{
								// Set the whole pixel to 0 (Even if we should only set the alpha value
								pdata[x] = 0;
							}
						}
					}
				}
			}
		}
		break;
	}
}

void CL_MaskTranscolProvider::perform_unlock()
{
	parent->unlock();
}
