/*
	$Id: sound.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="Sound"

#ifndef header_sound
#define header_sound

#include <stdlib.h>
#include "../../stl.h"

#include "soundbuffer.h"

class CL_SoundCard
//: Soundcard interface in ClanLib.
// CL_SoundCard is the interface to a soundcard. It is used to control the
// main mixer volume and other global settings. It also provides access to
// the microphone and line in as streamed soundproviders.
{
public:
	std::string name;
	//: Name of the soundcard.

	int card_no;
	//: Card number.

	virtual ~CL_SoundCard() { ; }

	virtual CL_StreamSoundProvider *get_microphone()=0;
	//: Returns the streamed soundprovider representing the microphone.

	virtual CL_StreamSoundProvider *get_line_in()=0;
	//: Returns the streamed soundprovider representing the line in.

	virtual void stop_all()=0;
	//: Stops all sample playbacks on the soundcard.
	
	virtual void set_global_volume(int volume)=0;
	//: Sets the main/mixer volume on the soundcard.

	virtual void set_global_pan(int pan)=0;
	//: Sets the main panning position on the soundcard.
};

class CL_Sound
//: Sound interface in ClanLib.
// This class provides a list of all soundcards available on the system. It
// is also a wrapper class for the CL_SoundCard class, providing an easier
// access to a single selected soundcard.
// <br>
// All the functions that share name with those in CL_SoundCard have the
// same functionality. The only difference is, that the ones CL_Sound
// operate on a selected soundcard. This saves the trouble of passing around
// a pointer to the soundcard, when only a single one is used anyway.
//!also: CL_SoundCard - Soundcard interface class.
//!also: CL_SoundBuffer - Sample class in ClanLib.
{
public:
	virtual ~CL_Sound() { ; }

	static std::vector<CL_SoundCard*> cards;
	//: The list of soundcards available on this system.
	
	static std::list<CL_SoundCard*> preload_cards;
	//: List of all cards where soundbuffers are automatically preloaded.

	static CL_StreamSoundProvider *get_microphone();
	//: Returns the streamed soundprovider representing the microphone.
	//!retval: The microphone.

	static CL_StreamSoundProvider *get_line_in();
	//: Returns the streamed soundprovider representing the line in.
	//!retval: The line in.

	static void clear_card_preload();
	//: Clears the list of preloaded soundcards.

	static void add_card_preload(CL_SoundCard *card);
	//: Adds 'card' to the preloaded soundcards list. This means all soundbuffers 
	//: will be automatically preloaded onto the card. By default the first card 
	//: is always on the preload list.
	//: <br>
	//: If a card isn't on the list, the soundbuffer's data will first be loaded when 
	//: the sample is actually played for the first time.
	//!param: card - The card to be added to the preload list.

	static void remove_card_preload(CL_SoundCard *card);
	//: Removes the specified card from the preload list.
	//!param: card - The card to be removed from the list.

	static void select_card(CL_SoundCard *card);
	//: Change the current selected soundcard to 'card'.
	//!param: card - The new current selected soundcard.
	
	static void select_card(int card_no);
	//: Change the current selected soundcard to soundcard number 'card_no'.
	//!param: card_no - Card number in the CL_Sound::cards list.

	static CL_SoundCard *get_current_card();
	//: Returns the current selected soundcard.
	//!retval: Current soundcard.
};

#endif
