#include "cp_types.h"
#include "cp_proto.h"

/* ======================  build_j_function ====================== */
/* This is a specialized routine to construct circlepacking complexes
associated with functions like the "j-function" of classical function
theory. Parameters n0 and n1 are specified "branch orders" (recall that
"multiplicity" is one larger than order and the the number of faces is
2*multiplicity). maxsize puts a cap on size. 

Every vertex of the intended complex is to be associated with 0, 1, 
or infinity (recorded in util_flag as 1,2,3). Every face has one 
of each type of vert, half with one orientation, half with the 
other (ie. shaded/unshaded). 

Build as many generations as possible from an initial 2*(n1+1)-flower,
center a 1-type vert; continually circulate around the boundary,
each time you reach a 0-type or 1-type bdry vert v, you enclose
it with enough faces (with vertices properly labeled) so it has
degree 2*(n0+1) or 2*(n1+1), respectively.
*/

int build_j_function(struct p_data *p,int n0,int n1,int maxsize)
{
  int i,j,k,next_bdry,aft_bdry,fore_bdry,cur_bdry,num;
  int N,dead,alive,vert,*util=NULL;
  int *new_flower,*aft_flower,*fore_flower;
  struct K_data *pK_ptr;

  /* initiate pack and expand to hold maxsize */
  if (maxsize<10 || n0<1 || n1<1 || !seed_pack(p,0,2*(n1+1))
      || !alloc_pack_space(p,maxsize+10*(n0+1)*(n1+1),1)) 
    return 0;
  pK_ptr=p->packK_ptr;
  pK_ptr[1].util_flag=2; /* 1-type vert at center */
  for (i=1;i<=n1+1;i++)
    {
      pK_ptr[2*i].util_flag=1; /* 0-type vert */
      pK_ptr[2*i+1].util_flag=3; /* inf-type vert */
    }
  cur_bdry=next_bdry=2; /* get started traveling around the 
			   ever-expanding bdry, adding faces 
			   at 0- and 1-type bdry verts. */
  /* find the next bdry subject to added faces for later use. */
  while (pK_ptr[(next_bdry=pK_ptr[next_bdry].
		 flower[0])].util_flag==3)
    {
      if (next_bdry==cur_bdry) /* error, bomb */
	return 0;
    }
  /* set intended flower multiplicities */
  if (pK_ptr[cur_bdry].util_flag==1) /* 0-type vert */
    N=n0+1;
  else N=n1+1; /* 1-type vert */

  /* main while loop */

  while (p->nodecount<maxsize)
    {
      if (!pK_ptr[cur_bdry].bdry_flag
	  || pK_ptr[cur_bdry].util_flag==3) /* done with this one */
	{
	  cur_bdry=next_bdry;
	  if (pK_ptr[cur_bdry].util_flag==1) /* 0-type vert */
	    N=n0+1;
	  else N=n1+1; /* 1-type vert */

	  /* find the next bdry subject to added faces for later use. */
	  while (pK_ptr[(next_bdry=pK_ptr[next_bdry].
			 flower[0])].util_flag==3)
	    {
	      if (next_bdry==cur_bdry) /* error, bomb */
		return 0;
	    }
	}
      
      /* cur_bdry shouldn't be inf-type */
      if (pK_ptr[cur_bdry].util_flag==3) goto BACK_TO_WHILE;

      if (pK_ptr[cur_bdry].util_flag==1) /* 0-type vert */
	N=n0+1;
      else N=n1+1; /* 1-type vert */
      fore_bdry=pK_ptr[cur_bdry].flower[0];
      aft_bdry=pK_ptr[cur_bdry].flower[pK_ptr[cur_bdry].num];

      /* break into cases depending on face count at cur_bdry */

      if (pK_ptr[cur_bdry].num>2*N) /* too many faces already */
	return -1;
      if (pK_ptr[cur_bdry].num==2*N) /* have all the necessary faces, just
			   close up and check neighbors */
	{
	  if (!identify_nghbs(p,cur_bdry,&alive,&dead)) 
	    return 0;
	  if (next_bdry==dead) next_bdry=alive;
	  else if (next_bdry>dead) next_bdry--;
	  if (cur_bdry>dead) cur_bdry--;
	  pK_ptr[cur_bdry].bdry_flag=0;

	  /* too many faces at the consolidated neighbor */
	  if ((pK_ptr[alive].util_flag==1 && 
	       pK_ptr[alive].num>2*(n0+1))
	      || (pK_ptr[alive].util_flag==2 &&
		  pK_ptr[alive].num>2*(n1+1))) 
	    return -1;
	  goto BACK_TO_WHILE;
	}
      if (pK_ptr[cur_bdry].num==2*N-1) /* only identify existing nghbs */
	{
	  /* create new flower space for cur_bdry */
	  new_flower=(int *)calloc((size_t)(2*N+1),sizeof(int));
	  for (i=0;i<=pK_ptr[cur_bdry].num;i++)
	    new_flower[i]=pK_ptr[cur_bdry].flower[i];
	  new_flower[2*N]=new_flower[0];
	  free(pK_ptr[cur_bdry].flower);
	  pK_ptr[cur_bdry].flower=new_flower;
	  pK_ptr[cur_bdry].num++;
	  pK_ptr[cur_bdry].bdry_flag=0;

	  /* fix fore_bdry */
	  fore_flower=(int *)calloc((size_t)(pK_ptr[fore_bdry].num+2),
				    sizeof(int));
	  for (j=0;j<=pK_ptr[fore_bdry].num;j++)
	    fore_flower[j]=pK_ptr[fore_bdry].flower[j];
	  fore_flower[pK_ptr[fore_bdry].num+1]=aft_bdry;
	  free(pK_ptr[fore_bdry].flower);
	  pK_ptr[fore_bdry].flower=fore_flower;
	  pK_ptr[fore_bdry].num++;

	  /* fix aft_bdry */
	  aft_flower=(int *)calloc((size_t)(pK_ptr[aft_bdry].num+2),
				    sizeof(int));
	  for (j=0;j<=pK_ptr[aft_bdry].num;j++)
	    aft_flower[j+1]=pK_ptr[aft_bdry].flower[j];
	  aft_flower[0]=fore_bdry;
	  free(pK_ptr[aft_bdry].flower);
	  pK_ptr[aft_bdry].flower=aft_flower;
	  pK_ptr[aft_bdry].num++;

	  /* too many faces at neighbors? */
	  if ((pK_ptr[fore_bdry].util_flag==1 && 
	       pK_ptr[fore_bdry].num>2*(n0+1))
	      || (pK_ptr[fore_bdry].util_flag==2 &&
		  pK_ptr[fore_bdry].num>2*(n1+1))
	      || (pK_ptr[aft_bdry].util_flag==1 && 
		  pK_ptr[aft_bdry].num>2*(n0+1))
	      || (pK_ptr[aft_bdry].util_flag==2 &&
		  pK_ptr[aft_bdry].num>2*(n1+1))) 
	    return -1;
	  goto BACK_TO_WHILE;
	}
      else /* have to add one face and check aft_bdry */
	{
	  /* create new vert, flower */
	  vert=p->nodecount+1;
	  p->nodecount++;
	  pK_ptr[vert].num=1;
	  pK_ptr[vert].bdry_flag=1;
	  pK_ptr[vert].flower=(int *)calloc((size_t)(2),sizeof(int));
	  pK_ptr[vert].flower[0]=cur_bdry;
	  pK_ptr[vert].flower[1]=aft_bdry;

	  /* fix cur_bdry flower */
	  new_flower=(int *)calloc((size_t)(2*N+1),sizeof(int));
	  for (i=0;i<=pK_ptr[cur_bdry].num;i++)
	    new_flower[i]=pK_ptr[cur_bdry].flower[i];
	  new_flower[pK_ptr[cur_bdry].num+1]=vert;
	  free(pK_ptr[cur_bdry].flower);
	  pK_ptr[cur_bdry].flower=new_flower;
	  pK_ptr[cur_bdry].num++;

	  /* set util_flag's */
	  if (pK_ptr[cur_bdry].util_flag==1 
	      && pK_ptr[aft_bdry].util_flag==2)
	    pK_ptr[vert].util_flag=3;
	  else if (pK_ptr[cur_bdry].util_flag==2 
	      && pK_ptr[aft_bdry].util_flag==1)
	    pK_ptr[vert].util_flag=3;
	  else if (pK_ptr[cur_bdry].util_flag==3
	      && pK_ptr[aft_bdry].util_flag==2)
	    pK_ptr[vert].util_flag=1;
	  else if (pK_ptr[cur_bdry].util_flag==2 
	      && pK_ptr[aft_bdry].util_flag==3)
	    pK_ptr[vert].util_flag=1;
	  else if (pK_ptr[cur_bdry].util_flag==1 
	      && pK_ptr[aft_bdry].util_flag==3)
	    pK_ptr[vert].util_flag=2;
	  else if (pK_ptr[cur_bdry].util_flag==3 
	      && pK_ptr[aft_bdry].util_flag==1)
	    pK_ptr[vert].util_flag=2;

	  /* fix up aft_bdry */
	  num=pK_ptr[aft_bdry].num;
	  aft_flower=(int *)calloc((size_t)(num+2),sizeof(int));
	  for (i=0;i<=num;i++)
	    aft_flower[i+1]=pK_ptr[aft_bdry].flower[i];
	  aft_flower[0]=vert;
	  free(pK_ptr[aft_bdry].flower);
	  pK_ptr[aft_bdry].flower=aft_flower;
	  pK_ptr[aft_bdry].num++;

	  /* too many faces at the consolidated neighbor */
	  if ((pK_ptr[aft_bdry].util_flag==1 && 
	       pK_ptr[aft_bdry].num>2*(n0+1))
	      || (pK_ptr[aft_bdry].util_flag==2 &&
		  pK_ptr[aft_bdry].num>2*(n1+1))) 
	    return -1;

	}
    BACK_TO_WHILE:
      continue;
    } /* end of main while */

  /* set radii, etc */
  p->hes=-1; 
  for (j=1;j<=p->nodecount;j++)
    {
      if (pK_ptr[j].bdry_flag) p->packR_ptr[j].rad=5.0; 
      /* bdry radii essentially infinite */
      else p->packR_ptr[j].rad=.5;
    }
  p->alpha=1;
  p->gamma=p->beta=2;

  /* save util_flags */
  util=(int *)calloc((size_t)(p->nodecount+1),sizeof(int));
  for (j=1;j<=p->nodecount;j++) util[j]=pK_ptr[j].util_flag;

  /* fix packing up */
  complex_count(p,0);
  facedraworder(p,0);
  set_aim_default(p);

  /* shade alternate faces */

  for (j=1;j<=p->facecount;j++)
    {
      i=util[p->faces[j].vert[0]];
      k=util[p->faces[j].vert[1]];
      if ((i==1 && k==2)
	  || (i==2 && k==3)
	  || (i==3 && k==1))
	p->faces[j].color=FG_COLOR;
      else p->faces[j].color=BG_COLOR;
    }
  if (util) free(util);
  return p->nodecount;
} /* build_j_function */

int identify_nghbs(struct p_data *p,int v,int *alive,int *dead)
     /* Simply identify two bdry neighbors of bdry vert v to
	close up flower at v; plan to keep fore_vert, adjusting
	it's flower, then throwing out aft_vert as a node number
	and making required adjustments */
{
  int i,j,k,fore_num,fore_vert,aft_vert,*new_flower;
  struct K_data *pK_ptr=p->packK_ptr;

  if (v<1 || v>p->nodecount || !pK_ptr[v].bdry_flag || pK_ptr[v].num<3) 
    return 0;
  *alive=fore_vert=pK_ptr[v].flower[0];
  *dead=aft_vert=pK_ptr[v].flower[pK_ptr[v].num];

  /* make new flower for fore_vert */
  fore_num=pK_ptr[fore_vert].num+pK_ptr[aft_vert].num;
  new_flower=(int *)calloc((size_t)(fore_num+1),sizeof(int));
  for (i=0;i<=pK_ptr[fore_vert].num;i++)
    new_flower[i]=pK_ptr[fore_vert].flower[i];
  for (i=pK_ptr[fore_vert].num+1;i<=fore_num;i++)
    new_flower[i]=pK_ptr[aft_vert].flower[i-pK_ptr[fore_vert].num];

  /* fix flower of v */
  pK_ptr[v].flower[pK_ptr[v].num]=fore_vert;
  pK_ptr[v].bdry_flag=0;

  /* go to flowers of neighbors of aft_vert and replace aft_vert
     with fore_vert */
  for (j=0;j<=pK_ptr[aft_vert].num;j++)
    {
      k=pK_ptr[aft_vert].flower[j];
      for (i=0;i<=pK_ptr[k].num;i++)
	if (pK_ptr[k].flower[i]==aft_vert)
	  pK_ptr[k].flower[i]=fore_vert;
    }

  /* shift all higher index info */
  free(pK_ptr[aft_vert].flower);
  for (k=aft_vert;k<p->nodecount;k++)
    pK_ptr[k]=pK_ptr[k+1];

  /* all references to aft_vert should be gone now; just have
     to shift all the node indices to fill the hole */
  for (i=1;i<p->nodecount;i++)
    for (j=0;j<=pK_ptr[i].num;j++)
      if (pK_ptr[i].flower[j]>aft_vert) 
	pK_ptr[i].flower[j]--;
  p->nodecount--;
  if ((*alive)>(*dead)) (*alive)=(*alive)-1;
  return 1;
} /* identify_nghbs */

