/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <math.h> /* floor() */
#include <unistd.h>

#include "cdw_main_window.h"
#include "gettext.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_window.h"
#include "cdw_processwin.h"
#include "cdw_debug.h"


/* *** some processwin layout constants *** */

/* first goes primary information line, telling user what happens now */
#define SUBWINDOW_TXT1_ROW 2
/* then some additional text info (e.g. "press any key" or "fixating") */
#define SUBWINDOW_TXT2_ROW 4
#define PROCESSWIN_PERMANENT_INFO_TEXT_ROW 6
/* some data that might be most important to the user */
#define PROCESSWIN_ETA_ROW 7
#define PROCESSWIN_AMOUNT_ROW 8
/* some visualization - less important than numeric data presented above,
   because user can get data presented on progress bar from percentage
   information - both are calculated using the same data, but progress bar
   is less accurate; "CENTRAL_ROW" means that there is subwindow border
   around this one-line progress bar*/
#define PROCESSWIN_PROGRESSBAR_CENTRAL_ROW 10
/* boring stuff at the bottom - common user might not care */
#define PROCESSWIN_FIFO_ROW 12


/* this string is used to erase parts of processwin - it has to be
 * filled with spaces (and ended with \0') first */
#define EMPTY_STRING_LEN PROCESSWIN_COLS - 2 - 1
static char processwin_empty_string[EMPTY_STRING_LEN + 1];

/* This variable keeps track of changes of percentage of task done.
   Progressbar and percentage value is updated only when the percentage
   changes, avoiding useless refreshes. It must be set to zero every time
   new processwin is created. Perhaps 'float' type is too fine-grained and
   processwin is updated too often. Or perhaps it is not so. */
static float static_percent;



static void cdw_processwin_display_text_info(int row, const char *string);

static struct {
	/* has the window been created and displayed? can
	   other modules print to the window? */
	bool active;

	WINDOW *window;

	/* area where progress bar is displayed;
	   may be used only by some instances of processwin */
	WINDOW *progressbar_sub;

	WINDOW *subwindow_txt1;
	WINDOW *subwindow_txt2;

	int n_cols;
	int n_rows;
	int begin_y;
	int begin_x;

	int subwindow_txt1_row;
	int subwindow_txt2_row;
} processwin;



/**
 * \brief Create UI window in which progress of some process (perhaps created by fork()) is shown
 *
 * Create UI window to show progress of process, make some initialization, too.
 * Put \p window_title string in title area of window.
 * Put \p window_label in first row of window (below title).
 *
 * \param window_title - title of processwin window
 * \param window_label - initial information displayed in window
 * \param show_progress - should progress bar be displayed?
 *
 * \return CDW_OK on success
 * \return CDW_GEN_ERROR if function failed to create progress window
 */
cdw_rv_t cdw_processwin_create(const char *window_title, const char *window_label, bool show_progress)
{
	if (processwin.active) {
		cdw_vdm ("ERROR: called the function for active process window\n");
		return CDW_ERROR;
	}

	memset(processwin_empty_string, ' ', EMPTY_STRING_LEN + 1);
	processwin_empty_string[EMPTY_STRING_LEN] = '\0';

	static_percent = 0.0;
	processwin.n_cols = PROCESSWIN_COLS;
	processwin.n_rows = PROCESSWIN_ROWS;
	processwin.begin_y = (LINES - processwin.n_rows) / 2;
	processwin.begin_x = (COLS - processwin.n_cols) / 2;
	processwin.subwindow_txt1_row = SUBWINDOW_TXT1_ROW;
	processwin.subwindow_txt2_row = SUBWINDOW_TXT2_ROW;

	processwin.window = newwin(processwin.n_rows,
				   processwin.n_cols,
				   processwin.begin_y,
				   processwin.begin_x);

	if (processwin.window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create process window with newwin()\n");
		return CDW_ERROR;
	}

	keypad(processwin.window, TRUE);
	wbkgd(processwin.window, COLOR_PAIR(CDW_COLORS_DIALOG));
	werase(processwin.window);
	wtimeout(processwin.window, -1);

	processwin.subwindow_txt1 = derwin(processwin.window, 2,
					   processwin.n_cols - 2,
					   processwin.subwindow_txt1_row, 1);

	if (processwin.subwindow_txt1 == (WINDOW *) NULL) {
		delwin(processwin.window);
		processwin.window = (WINDOW *) NULL;
		cdw_vdm ("ERROR: failed to create subwindow_txt1 with derwin()\n");
		return CDW_ERROR;
	}
	processwin.subwindow_txt2 = derwin(processwin.window, 2,
					   processwin.n_cols - 2,
					   processwin.subwindow_txt2_row, 1);
	if (processwin.subwindow_txt2 == (WINDOW *) NULL) {
		delwin(processwin.subwindow_txt1);
		processwin.subwindow_txt1 = (WINDOW *) NULL;
		delwin(processwin.window);
		processwin.window = (WINDOW *) NULL;
		cdw_vdm ("ERROR: failed to create subwindow_txt2 with derwin()\n");
		return CDW_ERROR;
	}

	cdw_window_add_strings(processwin.window, window_title, (char *) NULL);

	if (show_progress) {
		cdw_processwin_add_progress_bar();
	}

	processwin.active = true;

	/* display only primary label, secondary is empty */
	cdw_processwin_display_main_info(window_label);
	cdw_processwin_display_sub_info((char *) NULL);
	cdw_processwin_wrefresh();

	return CDW_OK;
}





/**
 * \brief Destroy UI window created by processwin_create()
 *
 * This is just wrapper for simple destroying of curses window
 * Set second argument as true if you want to wait for user key
 * before destroying the window - user will have time and opportunity
 * to read \p status_string. The string will be displayed
 * as primary information string in processwin. Secondary information
 * string will be sth like "press any key". If \p is false, then
 * \p status_string won't be displayed, and window will be destroyed
 * without waiting for any user action.
 *
 * \param status_string - text displayed in process window
 * \param wait - decides if the window should wait for a key from
 *                    user before closing.
 */
void cdw_processwin_destroy(const char *status_string, bool wait)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}
	if (wait) {
		cdw_processwin_display_main_info(status_string);
		/* 2TRANS: message displayed in progress window:
		   user must press any key to close the window after
		   task is completed */
		cdw_processwin_display_sub_info(_("Press any key"));
		cdw_processwin_wrefresh();
		cdw_processwin_wgetch();
	}

	if (processwin.progressbar_sub) {
		cdw_processwin_delete_progress_bar();
	}

	delwin(processwin.subwindow_txt1);
	processwin.subwindow_txt1 = (WINDOW *) NULL;
	delwin(processwin.subwindow_txt2);
	processwin.subwindow_txt1 = (WINDOW *) NULL;

	delwin(processwin.window);
	processwin.window = (WINDOW *) NULL;
	processwin.active = false;

	cdw_main_ui_main_window_wrefresh_part(processwin.n_rows,
					      processwin.n_cols,
					      processwin.begin_y,
					      processwin.begin_x);

	return;
}





bool cdw_processwin_is_active(void)
{
	return processwin.active;
}





void cdw_processwin_add_progress_bar(void)
{
	/* build progress bar */
	/* 27 = 25 + 2: 2 chars for borders and 25 chars for progress bar (4% per char) */
	int n_cols = 27;
	int n_rows = 3; /* upper border + progress bar strip + lower border */
        int begin_y = PROCESSWIN_PROGRESSBAR_CENTRAL_ROW - 1;
	int begin_x = (processwin.n_cols - n_cols) / 2;
	processwin.progressbar_sub = derwin(processwin.window, n_rows, n_cols,
					    begin_y, begin_x);
	box(processwin.progressbar_sub, 0, 0);

	/* this puts "0% done" so that user knows that some process
	   will start in few seconds - it looks better than no
	   information at all */
	cdw_processwin_display_progress(0, 100, "");
	cdw_processwin_wrefresh();

	return;
}





void cdw_processwin_delete_progress_bar(void)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}

	if (processwin.progressbar_sub == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: called the function for process window without progress bar\n");
		return;
	}

	/* on some occasions cdw_processwin_delete_progress_bar()
	   is called not because we want to destroy whole
	   process window, but only because we want to erase
	   its part showing progress; therefore we need to
	   erase "amount" information, progress bar and ETA
	   string; in case of progress bar window simple delete
	   and refresh won't be enough */
	mvwprintw(processwin.window, PROCESSWIN_AMOUNT_ROW, 1, "%s", processwin_empty_string);
	mvwprintw(processwin.window, PROCESSWIN_ETA_ROW, 1, "%s", processwin_empty_string);

	werase(processwin.progressbar_sub);
	delwin(processwin.progressbar_sub);
	processwin.progressbar_sub = (WINDOW *) NULL;

	return;
}





/**
 * \brief Display ETA string in progress window
 *
 * The function itself doesn't know anything about calculation of ETA, it just
 * knows if and where to display it. You can erase place where usually
 * ETA string is displayed by providing null string ((char *) NULL) as an argument.
 *
 * \param eta_string - string containing ETA information
 */
void cdw_processwin_display_eta(const char *eta_string)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}

	cdw_processwin_erase_eta();

	if (! eta_string || (!strcmp(eta_string, ""))) {
		;
       	} else {
		int col = (processwin.n_cols / 2) - ((int) strlen(eta_string) / 2);
		mvwprintw(processwin.window, PROCESSWIN_ETA_ROW, col, "%s", eta_string);
	}

	wrefresh(processwin.window);

	return;
}





void cdw_processwin_erase_eta(void)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
	} else {
		mvwprintw(processwin.window, PROCESSWIN_ETA_ROW, 1, "%s", processwin_empty_string);
	}
	return;
}





/**
 * \brief Draw progress bar strip in processwin
 *
 * Draw progress bar strip (not the borders, those should be created
 * when processwin itself is created) of length=value. Maximal value of
 * variable 'value' should be normalized by caller to width of processwin.
 *
 * \param value - current length of progress bar to draw
 * \param color1 - color name used for drawing a progress bar
 * \param color2 - name of normal color of process window
 * \param c - character to be used to draw bar
 */
void cdw_processwin_draw_progress_bar(int value, cdw_colors_t color1, cdw_colors_t color2, chtype c)
{
	cdw_assert (processwin.progressbar_sub != (WINDOW *) NULL,
		    "ERROR: called %s() for processwin without progressbar\n", __func__);

	const int start_col = 1;
	(void) wattrset(processwin.progressbar_sub, COLOR_PAIR(color1));
	for (int i = 0; i <= value; i++) {
		mvwaddch(processwin.progressbar_sub, 1, start_col + i, c);
	}

	(void) wattrset(processwin.progressbar_sub, COLOR_PAIR(color2));
	wrefresh(processwin.progressbar_sub);

	return;
}





/**
 * Show (in processwin window) information about progress of process. Do this only if necessary.
 *
 * Check if value describing current status (in terms of percentage of task
 * done) has changed. If yes, then update processwin with new information
 * (which requires call of ncurses wrefresh()). If not then do nothing.
 *
 * \param current - current status of task - this is value of arbitrary unit,
 *                  reflecting current amount of work done
 * \param total - total size of task - this is value of arbitrary unit (but
 *                  the same as of 'current'), reflecting maximal amount of
 *                  work that has to be done
 * \param description - string containing data or information
 *                  prepared by caller (can be NULL or empty)
 */
void cdw_processwin_display_progress_conditional(long current, long total, const char *description)
{
	/* assertion may fail if caller botched calculations of
	   current or total, causing overflows */
	cdw_assert (current >= 0 && total >= 0,
		    "ERROR: value of \"current\" or \"total\" is negative: %ld / %ld\n", current, total);
	/* new_static_percent reflects current amount of work done
	   at given specific call to this function */
	float new_static_percent = 0.0;
	if (total != 0) {
		/* FIXME: don't allow passing total == 0,
		   improve interface of the function */
		new_static_percent = (float) current / (float) total;
	}

	/* static_percent reflects amount of work done at
	   previous call to this function */
	if (new_static_percent - static_percent > 0.0001) {
		/* if they differ then information in processwin should be updated */
		static_percent = new_static_percent;
		cdw_processwin_display_progress(current, total, description);
	}

	return;
}





/**
 * \brief Show (in processwin window) information about progress of process.
 *
 * Show to user updated information about progress of current process. This
 * can be done in two ways: by displaying text line prepared by caller or by
 * showing percentage information.
 *
 * Third argument can have any character, even descriptive, but since this
 * function is called cdw_processwin_display_progress(), it would be nice if
 * this argument hold some sort of numerical information, perhaps with units
 * of measure.
 *
 * First and second argument should be provided without any preprocessing,
 * because they will be recalculated to percents like this:
 * percentage_info = (current_amount * 100) / total_amount.
 *
 * Percentage information is shown both in form of "x% done" text and in form
 * of progress bar.
 *
 * All information is displayed in one line, in following format:
 * "xx% done (current_value_string)". Both parts ('xx%done' and
 * 'current_value_string') can be omitted (set to 0 or (char *) NULL).
 *
 * \p description should be no longer than (PROCESSWIN_COLS - 2) chars,
 * including ending '\0'. It should be even shorter to make place for
 * numeric data displayed in the same line as \p description.
 *
 * \param current_amount - value representing current state (stage) of process
 * \param total_amount - value representing final, expected state (stage) of process
 * \param description - string containing details on data or other information
 */
void cdw_processwin_display_progress(long current_amount, long total_amount, const char *description)
{
	bool has_descr = false; /* is description string non-empty? */
	char progress_string[PROCESSWIN_MAX_RTEXT_LEN + 1];

	/* first check for NULLness, so strlen arg will be safe */
	if (description && (strlen(description)) ) {
		has_descr = true;
	}

	memset(progress_string, ' ', PROCESSWIN_MAX_RTEXT_LEN + 1);
	progress_string[PROCESSWIN_MAX_RTEXT_LEN] = '\0';
	mvwprintw(processwin.window, PROCESSWIN_AMOUNT_ROW, 1, "%s", progress_string);

	if (total_amount) {
		/* we are able to calculate percentage and display progress bar */

		int percent_done = (int) ((current_amount * 100) / total_amount);

		if (has_descr) {
			snprintf(progress_string, PROCESSWIN_MAX_RTEXT_LEN + 1,
				 /* 2TRANS: status message displayed in progress
				    window: %d%% is percent amount of task finished,
				    %s is previously prepared string displaying
				    some additional information about %d, e.g. units */
				 _("%d%% done (%s)"), percent_done, description);
		} else {
			snprintf(progress_string, PROCESSWIN_MAX_RTEXT_LEN + 1,
				 /* 2TRANS: status message displayed in progress
				    window: %d%% is percent amount of task finished */
				 _("%d%% done"), percent_done);
		}
		cdw_processwin_draw_progress_bar((percent_done / 4) - 1,
						 CDW_COLORS_DIALOG,
						 CDW_COLORS_DIALOG, ACS_BLOCK);
	} else { /* total_amount == 0, so there is no way we can show percentage information */
		if (has_descr) {
			strncpy(progress_string, description, PROCESSWIN_MAX_RTEXT_LEN);
			progress_string[PROCESSWIN_MAX_RTEXT_LEN] = '\0';
		} else {
			progress_string[0] = '\0';
		}
	}

	int col = (processwin.n_cols / 2) - ((int) strlen(progress_string) / 2);
	mvwprintw(processwin.window, PROCESSWIN_AMOUNT_ROW, col, "%s", progress_string);

	wrefresh(processwin.window);

	return;
}





void cdw_processwin_display_main_info(const char *string)
{
	cdw_processwin_display_text_info(1, string);
	return;
}

void cdw_processwin_display_sub_info(const char *string)
{
	cdw_processwin_display_text_info(2, string);
	return;
}





/**
 * \brief Show (in processwin window) text information about status of process
 *
 * Display string with information, or erase (blank) it. The string is
 * displayed in first or second topmost row, depending on value of
 * first argument. Accepted values of \p row are 1 or 2.
 *
 * Set second argument to (char *) NULL if you want to erase it, or set
 * it to pointer to a valid string if you want to display the string.
 *
 * \param row - index of row that you want to update
 * \param string - string displayed in processwin
 */
void cdw_processwin_display_text_info(int row, const char *string)
{
	cdw_assert (row == 1 || row == 2, "ERROR: incorrect id of row to print in (%d), should be 1 or 2\n", row);
	if (!processwin.active) {
		/* warning, not error/assert; on some rare occasions
		   calling this function for null window may be ok */
		cdw_vdm ("WARNING: called the function for inactive process window\n");
		return;
	}

	WINDOW *window = (WINDOW *) NULL;
	if (row == 1) {
		window = processwin.subwindow_txt1;
	} else {
		window = processwin.subwindow_txt2;
	}
	cdw_assert (window != (WINDOW *) NULL, "ERROR: attempting to print to NULL window\n");

	werase(window);
	if (string != (char *) NULL && (strcmp(string, ""))) {
		cdw_window_print_message(window, string, CDW_ALIGN_CENTER);
	} else {
		/* user may pass null pointer or empty string ("")
		   to show that given text area should be erased */
	}
	wrefresh(window);

	return;
}





/**
 * \brief Display fifo buffer information
 *
 * This function displays test string in form of "Fifo: %d%%" in
 * process window. It is useful when writing data to a disc.
 * If fifo value is lower than 25, the text background is highlighted.
 *
 * \param fifo_level - FIFO buffer usage value (0 - 100)
 */
void cdw_processwin_display_fifo_and_speed(int fifo_level, int speed_decimal, int speed_fract)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}

	mvwprintw(processwin.window, PROCESSWIN_FIFO_ROW, 1, "%s", processwin_empty_string);

	if (fifo_level >= 0) {
		/* display fifo */

		char fifo_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
		/* FIXME: digits disappear when fifo_level == 0; probably a problem
		   with format string */
		/* 2TRANS: this is message displayed in progress window, describing
		   FIFO buffer usage (in percents) during some resources-consuming
		   tasks */
		snprintf(fifo_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("FIFO: %3.d%%"), fifo_level);

		/* http://howto-pages.org/cdwriting/07.php
		   "If the level keeps dropping below half full then either the FIFO
		   is too small or you're trying to write too fast." */
		if (fifo_level <= 25) {
			wattrset(processwin.window, COLOR_PAIR(CDW_COLORS_ERROR));
		} else if (fifo_level <= 50) {
			wattrset(processwin.window, COLOR_PAIR(CDW_COLORS_WARNING));
		} else {
			;
		}

		int col = 0;
		if (speed_decimal < 0) {
			/* speed information won't be displayed,
			   there is more room for FIFO string */
			col = (processwin.n_cols / 2) - ((int) strlen(fifo_string) / 2);
		} else {
			/* FIFO string should be moved to left side
			   of process window */
			col = (processwin.n_cols / 4) - ((int) strlen(fifo_string) / 2);
		}

		mvwprintw(processwin.window, PROCESSWIN_FIFO_ROW, col, "%s", fifo_string);
		wattrset(processwin.window, COLOR_PAIR(CDW_COLORS_DIALOG));
	}

	if (speed_decimal >= 0) {
		/* display speed (in the same line as FIFO) */

		char speed_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
		/* 2TRANS: this is message displayed in process progress window;
		   "Speed" is writing speed, %d and %d are decimal and fractional
		   parts of writing speed, e.g. "Speed: 2.5x" */
		sprintf(speed_string, _("Speed: %2d.%dx"), speed_decimal, speed_fract);
		cdw_vdm ("INFO: speed: decimal = %d, fract = %d -> string = \"%s\"\n",
			 speed_decimal, speed_fract, speed_string);

		int col = 0;
		if (fifo_level < 0) {
			/* FIFO information won't be displayed,
			   there is more room for speed string */
			col = (processwin.n_cols / 2) - ((int) strlen(speed_string) / 2);
		} else {
			/* speed string should be moved to right side
			   of process window */
			col = (3 * processwin.n_cols / 4) - ((int) strlen(speed_string) / 2);
		}

		mvwprintw(processwin.window, PROCESSWIN_FIFO_ROW, col, "%s", speed_string);
	}

	wrefresh(processwin.window);

	return;
}




void cdw_processwin_erase_fifo_and_speed(void)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}
	mvwprintw(processwin.window, PROCESSWIN_FIFO_ROW, 1, "%s", processwin_empty_string);

	return;
}




void cdw_processwin_wrefresh(void)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
	} else {
		redrawwin(processwin.window);
		wrefresh(processwin.window);
	}
	return;
}




void cdw_processwin_wgetch(void)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}
	/* sometimes the process window closes without proper user key,
	   I still don't know why; I think that there is an error with
	   wgetch() checking if read is blocking or not (or if there is
	   a delay or not); I'm using loop + usleep() to simulate
	   blocking read */
	int rv = wgetch(processwin.window);
	while (rv == ERR) {
		usleep(100000);
		rv = wgetch(processwin.window);
	}
	return;
}




/**
 * \brief Very simple wrapper for functions refreshing processwin window
 *
 * First the window is touchwin()ed and then it is wrefresh()ed.
 */
void cdw_processwin_force_refresh(void)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}
	touchwin(processwin.window);
	wrefresh(processwin.window);

	return;
}





/**
 * \brief Reset any information about progress accumulated so far in progress window
 */
void cdw_processwin_reset_progress(void)
{
	if (!processwin.active) {
		cdw_vdm ("ERROR: called the function for inactive process window\n");
		return;
	}
	cdw_assert (processwin.progressbar_sub, "ERROR: calling %s() for processwin without progressbar\n", __func__);

	static_percent = 0.0;

	char progress_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	memset(progress_string, ' ', PROCESSWIN_MAX_RTEXT_LEN + 1);
	progress_string[PROCESSWIN_MAX_RTEXT_LEN] = '\0';
	mvwprintw(processwin.window, PROCESSWIN_AMOUNT_ROW, 1, "%s", progress_string);

	int ncols = getmaxx(processwin.progressbar_sub);
	for (int i = 1; i < ncols - 1; i++) {
		mvwprintw(processwin.progressbar_sub, 1, i, " ");
	}

	wrefresh(processwin.progressbar_sub);
	wrefresh(processwin.window);

	return;
}


