/*
cdcd - Command Driven CD player
Copyright (C) 1998-99 Tony Arcieri
Copyright (C) 2001 Fabrice BAUZAC

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <stdio.h>

/* Should be declared in <stdio.h>, but I need this line for an
   unknown reason.  */
extern int vasprintf (char **, const char *, va_list);

#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include "cdcd.h"
#include "str.h"

/* Returns a malloc()ed string containing the lowercase version of S */
char *
strlowera (const char *s)
{
  char *r, *t;
  t = r = (char*) xmalloc (strlen (s) + 1);
  while (*s)
    *t++ = tolower (*s++);
  *t=0;
  return r;
}

/* case-insensitive strstr() */
const char *
strcasestr (const char *u, const char *v)
{
  char *uu, *vv, *ret;
  int i;
  uu = strlowera (u);
  vv = strlowera (v);
  ret = strstr (uu, vv);  /* ret is a pointer inside uu.  */
  if (!ret)
    return NULL;
  i = ret - uu;  /* i is an index.  */
  free (uu);
  free (vv);
  return u + i;  /* u+i is a pointer inside u.  */
}

/* I don't know why, but these seem to not always work */

#if 0
/* Change a whole string to lower case */
int
strnlower(char *dest, const char *src, int len)
{
  int index;

  for (index = 0; index < len; index++)
    {
      dest[index] = tolower (src[index]);
      if (src[index] == '\0')
	return index;
    }
   
  return index;
}
#endif

#if 0
/* Case insensitive substring search with size limits*/
int
strncasestr (const char *haystack, int haystacklen,
	     const char *needle, int needlelen)
{
  char *haystacklower, *needlelower;
  int r;

  haystacklower = (char *) xmalloc (strlen (haystack) + 1);
  needlelower = (char *) xmalloc (strlen (needle) + 1);
   
  strnlower (haystacklower, haystack, haystacklen);
  strnlower (needlelower, needle, needlelen);
   
  r = strstr (haystacklower, needlelower)? 1: 0;
  free (haystacklower);
  free (needlelower);

  return r;
}
#endif

#if 0
/* Case insensitive substring search */
int
strcasestr (const char *haystack, const char *needle)
{
  int r;
  r = strncasestr (haystack, strlen (haystack), needle, strlen (needle));
  printf ("strcasestr({%s}, {%s}) = %d\n", haystack, needle, r);
  return r;
}
#endif



static int
space_needed (int wlen, struct pprintfd *ppd)
{
  int ret = wlen;
  if (ppd->col)
    ret += ppd->liad? 2: 1;
  return ret;
}

static void
print_word (int width, char *w, struct pprintfd *ppd)
{
  int l;
  int needed;

  l = strlen (w);

  needed = space_needed (l, ppd);
  if (ppd->col && ppd->col + needed > width)
    {
      putchar ('\n');
      ppd->col = 0;
      needed = space_needed (l, ppd);
    }

  if (ppd->col)
    {
      if (ppd->liad)
	putchar (' ');
      putchar(' ');
    }

  fputs (w, stdout);
  
  ppd->col += needed;

  ppd->liad = w[l - 1] == '.'? 1: 0;
}

/* These are the magic functions.  */

/* vpdprintf does a printf() with word breaks; the width is MAXCOL.
   Be careful, no \n is appended at the end of the last line, and all
   \n's are removed from FMT ! */
void
vpdprintf (int maxcol, struct pprintfd *ppd, char *fmt, va_list v)
{
  char *s, *p;
  char *word = NULL;
  vasprintf (&s, fmt, v);  /* I hope asprintf & vasprintf exist
			      everywhere.  Otherwise, I think I'm
			      going to use Glib: g_strdup_vsprintf.
			      Any other idea, anyone? */
  for (p = s; *p; p++)
    {
      if (!word && !isspace (*p)) /* We have just found a new word */
	word = p;
      else if (word && isspace (*p)) /* The current word is finished */
	{
	  *p = 0;
	  print_word (maxcol, word, ppd);
	  word = NULL;
	}
    }

  if (word)
    print_word (maxcol, word, ppd);
}

void
pdprintf (int maxcol, struct pprintfd *ppd, char *fmt, ...)
{
  va_list v;
  va_start (v, fmt);
  vpdprintf (maxcol, ppd, fmt, v);
  va_end (v);
}

void
vpprintf (int maxcol, char *fmt, va_list v)
{
  struct pprintfd ppd = {0, 0};
  vpdprintf (maxcol, &ppd, fmt, v);
}

void
pprintf (int maxcol, char *fmt, ...)
{
  va_list v;
  va_start (v, fmt);
  vpprintf (maxcol, fmt, v);
  va_end (v);
}

/* cv2v0 converts argc+argv style parameters to argv0 style parameters */
/* argv0 is a NULL-terminated array of pointers to strings (char*) */
char **
cv2v0 (int argc, char **argv)
{
  char **r;
  int i;
  r = (char**) xmalloc ((argc + 1) * sizeof (*r)); /* don't forget the
						      NULL pointer */
  for (i=0; i<argc; ++i)
    {
      r[i] = (char*) xmalloc (strlen (argv[i])+1);
      strcpy (r[i], argv[i]);
    }
  r[i] = NULL;
  return r;
}

/* freev0 frees a argv0 array */
void
freev0 (char **v)
{
  char **p = v;

  if (!v)
    return;

  while (*p)
    free (*p++);
  free (v);
}

/* Tokenization with ", ', `.  This is meant to be a (limited?)
   replacement of history_tokenize ().  However we don't use
   history_tokenize() because of its (bogus?) behaviour on empty lines
   (it returns NULL).  */
char **
my_tokenize (char *s)
{
  char **p;
  char *ss = s;
  int q = 0;
  char *word = NULL;
  int p_size = 0;

  p = (char**) xmalloc (sizeof (*p) * ++p_size);
  *p = NULL;

  while (*ss)
    {
      if (!isspace (*ss) && !word)
	word = ss;
      if (isspace (*ss) && word && !q)
	{
	  *ss = 0;
	  p = (char**) xrealloc (p, sizeof (*p) * ++p_size);
	  p[p_size-2] = (char*) xmalloc (strlen (word)+1);
	  strcpy (p[p_size-2], word);
	  p[p_size-1] = NULL;
	  word = NULL;
	}
      else if (*ss == '\'' || *ss == '\"' || *ss == '`')
	{
	  if (q == *ss)
	    q = 0;
	  else
	    q = *ss;
	}
      else if (*ss == '\\')
	ss++;

      ss++;
    }

  if (word)
    {
      p = (char**) xrealloc (p, sizeof (*p) * ++p_size);
      p[p_size-2] = (char*) xmalloc (strlen (word) + 1);
      strcpy (p[p_size-2], word);
      p[p_size-1] = NULL;
    }

  return p;
}

/*
  It should be easier to understand this with some examples:
  hello "           becomes hello (there is no need to quote)
  hello world "     becomes "hello world"
  hello world '     becomes 'hello world'
  it's nice "       becomes "it's nice"
  it's nice '       becomes 'it\'s nice'
  dumb thing\'"` `  becomes `dumb thing\\'"\`` (only \ and ` are backslashed)

  Q must be one of ', ", `.

  This function has not been extensively tested, it should behave like
  the previous examples.  If not, correct the function, not the examples!
*/
char *
quote_armor (char *s, char q)
{
  char *ret, *t;
  int need_quotes = 0;

  {
    /* Add quotes if and only if the following characters are in S */
    char *forbidden = " \t\\'\"`";
    int i;
    for (i=0; forbidden[i]; ++i)
      if (strchr (s, forbidden[i]))
	{
	  need_quotes = 1;
	  break;
	}
  }

  /* At max, s is filled with q or '\', so everything must be
     backslashed: strlen(s) * 2 */
  t = ret = (char*) xmalloc (2 * strlen (s) + (need_quotes? 2: 0) + 1);

  if (need_quotes)
    *t++ = q;

  while (*s)
    {
      if (*s == '\\' || *s == q)
	{
	  /* We know that need_quotes is set to 1 because there is '\\'
	     or q inside.  */
	  *t++ = '\\';
	  *t++ = *s++;
	}
      else
	*t++ = *s++;
    }

  if (need_quotes)
    *t++ = q;

  *t = 0;
  ret = (char*) xrealloc (ret, strlen(ret)+1); /* shrink */
  return ret;
}

/* Unquote one string: suppress non-protected quotes and some
   backslashes */
char *
unquote_armor (char *s)
{
  int q = 0;
  char *ret;
  char *p;
  p = ret = (char*) xmalloc (strlen (s)+1);

  while (*s)
    {
      if (*s=='\'' || *s=='\"' || *s=='`')
	{
	  if (q == *s)
	    {
	      s++;
	      q = 0;
	    }
	  else if (!q)
	    {
	      q = *s;
	      s++;
	    }
	  else
	    *p++ = *s++;
	}
      else if (*s == '\\')
	{
	  s++;
	  if (*s)
	    *p++ = *s++;
	}
      else
	*p++ = *s++;
    }

  *p = 0;
  return ret;
}

/* Unquote an entire parameter list */
char **
unquote_params (char **argv)
{
  char **ret;
  int argc;

  /* Count upwards... */
  for (argc = 0; argv[argc]; argc++)
    ;

  ret = (char**) xmalloc (sizeof (*ret) * (argc + 1));
  ret[argc] = NULL;

  if (!argc)
    return ret;

  /* This only works if argc > 0.  */

  /* ... and downwards.  */
  for (; argc; --argc)
    ret[argc-1] = unquote_armor (argv[argc-1]);

  return ret;
}
