/**
  \brief Audio inputs of a video device
  
  This class describes an audio input on the card... Hopefully there are as 
  many audio input as video inputs (and listed in the same order, but well,
  you never know :))
  
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/ioctl.h>

#include "VideoAudioInput.h"
#include "VideoDeviceLinux.h"

CVideoAudioInput::CVideoAudioInput(CVideoDeviceLinux *video, int channel)
{
   struct video_audio va;
   char buf[33];
   
   pVideo = video;
   Channel = channel;
   
   Flags = 0;

   // Query data 
   va.audio = Channel;
   if (ioctl(pVideo->m_CamFD, VIDIOCGAUDIO, &va) == 0) {
     strncpy(buf, va.name, 32);
     buf[32] = '\0';
     Name = buf;
     
     Flags = va.flags;
   }
   else
     qWarning("CVideoAudioInput: Warning: no channel info available.");
}

// public

int CVideoAudioInput::GetNumber() const
{
   return Channel;
}

QString CVideoAudioInput::GetName() const
{
   return Name;
}

/** Returns true if this audio input can be muted */
bool CVideoAudioInput::IsMutable()
{
   return (Flags & VIDEO_AUDIO_MUTABLE);
}

bool CVideoAudioInput::IsMuted()
{
   struct video_audio va;

   if (Flags & VIDEO_AUDIO_MUTABLE) {
     if (ioctl(pVideo->m_CamFD, VIDIOCGAUDIO, &va) == 0) {
       return (va.flags & VIDEO_AUDIO_MUTE);
     }
   }
   return FALSE;
}

/** 
  \brief Return volume setting (if supported)
  \return An integer in the range 0..65535, or -1 if not supported or on error
*/  
int CVideoAudioInput::GetVolume() const
{
   struct video_audio va;

  if (Flags & VIDEO_AUDIO_VOLUME) {
    if (ioctl(pVideo->m_CamFD, VIDIOCGAUDIO, &va) == 0)
      return va.volume;
   }
   return -1;
}

int CVideoAudioInput::GetBass() const
{
   struct video_audio va;

  if (Flags & VIDEO_AUDIO_BASS) {
    if (ioctl(pVideo->m_CamFD, VIDIOCGAUDIO, &va) == 0)
      return va.bass;
   }
   return -1;
}

int CVideoAudioInput::GetTreble() const
{
   struct video_audio va;

  if (Flags & VIDEO_AUDIO_TREBLE) {
    if (ioctl(pVideo->m_CamFD, VIDIOCGAUDIO, &va) == 0)
      return va.treble;
   }
   return -1;
}



// public slots

void CVideoAudioInput::Mute(bool mute_on)
{
   struct video_audio va;

   if (Flags & VIDEO_AUDIO_MUTABLE) {
     if (ioctl(pVideo->m_CamFD, VIDIOCGAUDIO, &va) == 0) {
       if (mute_on)
         va.flags |= VIDEO_AUDIO_MUTE;
       else
         va.flags &= ~VIDEO_AUDIO_MUTE;
       ioctl(pVideo->m_CamFD, VIDIOCSAUDIO, &va);
     }
   }
}


void CVideoAudioInput::SetVolume(int volume)
{
}


void CVideoAudioInput::SetBass(int bass)
{
}


void CVideoAudioInput::SetTreble(int treble)
{
}

