/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Searcher.h"
#include "Shared.h"
#include "Config.h"
#include "Tree.h"
#include <qlayout.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qregexp.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qradiobutton.h>
#include <qbuttongroup.h> 

/*------- local constants:
-------------------------------------------------------------------*/
const int     Searcher::OVER_STRETCH        = 100;
const int     Searcher::SPACING_WIDTH       = 2;
const int     Searcher::MARGIN_WIDTH        = 2;
const QString Searcher::CAPTION             = QT_TR_NOOP( "Searcher" );
const QString Searcher::BTN_RETURN_LABEL    = QT_TR_NOOP( "&Goto" );
const QString Searcher::BEGIN_FROM_LABEL    = QT_TR_NOOP( "&From:" );
const QString Searcher::WHAT_LABEL          = QT_TR_NOOP( "&What:" );
const QString Searcher::GBOX_SETTINGS_LABEL = QT_TR_NOOP( " Settings " );
const QString Searcher::GBOX_RESULTS_LABEL  = QT_TR_NOOP( " Results " );
const QString Searcher::END_OF_SEARCH       = QT_TR_NOOP( " Finish. Conformable to mask: %1" );
const QString Searcher::HDR_FILE            = QT_TR_NOOP( "File" );
const QString Searcher::HDR_DIR             = QT_TR_NOOP( "Directory" );

	
//*******************************************************************
// Searcher                                              CONSTRUCTOR
//*******************************************************************
Searcher::Searcher( QWidget* const in_parent )
: QDialog          ( in_parent, 0, FALSE )
, d_from_cbx       ( 0 )
, d_what_cbx       ( 0 )
, d_table          ( 0 )
, d_info_line      ( 0 )
, d_start_btn      ( new QPushButton( tr(Shared::RunBtnLabel), this ))
, d_break_btn      ( new QPushButton( tr(Shared::BreakBtnLabel), this ))
, d_return_btn     ( new QPushButton( tr(BTN_RETURN_LABEL), this ))
, d_cancel_btn     ( new QPushButton( tr(Shared::CancelBtnLabel), this ))
, d_tree_btn       ( 0 )
, d_break          ( FALSE )
, d_counter        ( 0 )
, d_goto_path      ( Shared::EmptyStr )
{
	setCaption( tr(CAPTION) );
	setFont( Config::instance()->lfs_default_font() );
	
	QVBoxLayout* const main_layout = new QVBoxLayout( this );
	main_layout->setSpacing( SPACING_WIDTH );
	main_layout->setMargin( MARGIN_WIDTH );
		
	// ******** TOP LAYOUT *******
	QHBoxLayout* const top_layout = new QHBoxLayout;
	top_layout->setSpacing( SPACING_WIDTH );
	top_layout->setMargin( MARGIN_WIDTH );
		
	// ------- COMBO BOX LAYOUT -------
	QVBoxLayout* const cbx_layout = new QVBoxLayout;
	cbx_layout->setSpacing( SPACING_WIDTH );
	cbx_layout->setMargin( MARGIN_WIDTH );
				
	QGroupBox* const gbox = new QGroupBox( 1, Qt::Horizontal, tr(GBOX_SETTINGS_LABEL), this );

	{	// BEGIN FROM - katalog, od ktorego zaczynamy szukac
		QLabel* label = new QLabel( tr(BEGIN_FROM_LABEL), gbox );
		
		QFrame* const frm = new QFrame( gbox );
		QHBoxLayout* const from_layout = new QHBoxLayout( frm );

		d_from_cbx = create_combo_box( frm );
		from_layout->addWidget( d_from_cbx );
		label->setBuddy( d_from_cbx );
		d_from_cbx->setCurrentText( QDir::homeDirPath() );
		
		from_layout->addWidget( d_tree_btn = new QPushButton( frm ));
		Shared::add_icon( d_tree_btn, "tree.png" );
		connect( d_tree_btn, SIGNAL( clicked() ), this, SLOT( slot_tree() ));
		from_layout->setStretchFactor( d_from_cbx, Shared::OverStretch );
	}
	
	{	// WHAT - maska szukanych plikow
		QLabel* label = new QLabel( tr(WHAT_LABEL), gbox );
		d_what_cbx = create_combo_box( gbox );
		label->setBuddy( d_what_cbx );
		cbx_layout->addWidget( gbox );
	}
				
	top_layout->addLayout( cbx_layout );
	top_layout->setStretchFactor( cbx_layout, OVER_STRETCH );	
			
	// ------- BUTTON LAYOUT --------
	QVBoxLayout* const btn_layout = new QVBoxLayout;
	btn_layout->setSpacing( SPACING_WIDTH );
	btn_layout->setMargin( MARGIN_WIDTH );
	btn_layout->addWidget( d_start_btn );
	btn_layout->addWidget( d_break_btn );
	btn_layout->addWidget( d_return_btn );
	btn_layout->addWidget( d_cancel_btn );
	btn_layout->addStretch( OVER_STRETCH );
	top_layout->addLayout( btn_layout );
			
	main_layout->addLayout( top_layout );

	// ******* BOTTOM LAYOUT *******
	QHBoxLayout* const bottom_layout = new QHBoxLayout;
	main_layout->addLayout( bottom_layout );
	main_layout->setStretchFactor( bottom_layout, OVER_STRETCH );
	bottom_layout->setSpacing( SPACING_WIDTH );
	bottom_layout->setMargin( MARGIN_WIDTH );
			
	{
		QGroupBox* const gbox = new QGroupBox( 1, Qt::Horizontal, tr(GBOX_RESULTS_LABEL), this );
						
		// TABLE
		d_table = new QListView( gbox );
		d_table->setPaletteBackgroundColor( Config::instance()->lfs_default_bkg_color() );
		d_table->addColumn( tr(HDR_FILE) );
		d_table->addColumn( tr(HDR_DIR) );
		d_table->setSorting ( 0, TRUE );
		d_table->setShowSortIndicator( TRUE );
		d_table->setSelectionMode( QListView::Single );
		d_table->setAllColumnsShowFocus( TRUE );
		// INFO LINE
		d_info_line = new QLineEdit( gbox );
		d_info_line->setReadOnly( TRUE );
		
		bottom_layout->addWidget( gbox );
	}
	
	d_break_btn->setDisabled( TRUE );
	
	connect( d_start_btn, SIGNAL( clicked() ), this, SLOT( slot_start() ));
	connect( d_break_btn, SIGNAL( clicked() ), this, SLOT( slot_break() ));
	connect( d_return_btn, SIGNAL( clicked() ), this, SLOT( slot_goto() ));
	connect( d_cancel_btn, SIGNAL( clicked() ), this, SLOT( slot_cancel() ));

	d_what_cbx->setFocus();
}
// end of Searcher

//*******************************************************************
// create_combo_box                                          PRIVATE
 //*******************************************************************
QComboBox* Searcher::create_combo_box( QWidget* const in_parent )
{
	QComboBox* const cbx = new QComboBox( in_parent );
	cbx->setEditable( TRUE );
	cbx->setInsertionPolicy( QComboBox::AtTop );
	cbx->setAutoCompletion( TRUE );
	cbx->setDuplicatesEnabled( FALSE );
	return cbx;
}
// end of create_combo_box

//*******************************************************************
// polish                                          PRIVATE inherited
//*******************************************************************
void Searcher::polish()
{
	Shared::polish( this, 50, 50 );
}
// end of polish

//*******************************************************************
// slot_start                                           PRIVATE slot
//*******************************************************************
void Searcher::slot_start()
{
	if( FALSE == d_from_cbx->currentText().isEmpty() ) {
		if( FALSE == d_what_cbx->currentText().isEmpty() ) {
			d_from_cbx->setDisabled( TRUE );
			d_what_cbx->setDisabled( TRUE );

			d_start_btn->setDisabled( TRUE );
			d_break_btn->setEnabled( TRUE );
			d_return_btn->setDisabled( TRUE );
			d_cancel_btn->setDisabled( TRUE );
			//--------------------------------------------
			befor();
			parse( d_from_cbx->currentText() );
			after();
			//--------------------------------------------
		}
		else {
			d_what_cbx->setFocus();
		}
	}
	else {
		d_from_cbx->setFocus();
	}
}
// end of slot_start

//*******************************************************************
// slot_break                                           PRIVATE slot
//*******************************************************************
void Searcher::slot_break()
{
	d_break = TRUE;
	
	d_from_cbx->setEnabled( TRUE );
	d_what_cbx->setEnabled( TRUE );

	d_start_btn->setEnabled( TRUE );
	d_break_btn->setDisabled( TRUE );
	d_return_btn->setEnabled( TRUE );
	d_cancel_btn->setEnabled( TRUE );
}
// end of slot_break

//*******************************************************************
// slot_goto                                          PRIVATE slot
//*******************************************************************
void Searcher::slot_goto()
{
	const QListViewItem* const item = d_table->currentItem();
	if( item ) {
		const QString dir = item->text( 1 );
		if( "/" == dir ) d_goto_path = dir + item->text( 0 );
		else             d_goto_path = dir + "/" + item->text( 0 );
	}
	accept();
}
// end of slot_return

//*******************************************************************
// slot_cancel                                          PRIVATE slot
//*******************************************************************
void Searcher::slot_cancel()
{
	reject();
}
// end of slot_cancel

//*******************************************************************
// slot_tree                                            PRIVATE slot
//*******************************************************************
void Searcher::slot_tree()
{
	Tree* tree = new Tree( this, QString::null );
	if( tree ) {
		QString selected_dir = QString::null;
		if( QDialog::Accepted == tree->exec() ) {
			selected_dir = tree->get_dir();
		}
		delete tree;
		tree = 0;
		
		if( selected_dir ) {
			d_from_cbx->setCurrentText( selected_dir );
		}
	}
}
// end of slot_tree

//*******************************************************************
// parse                                                     PRIVATE
//*******************************************************************
void Searcher::parse( const QString& in_path )
{
	QFileInfo* finfo = 0;
				
	// pokaz w linii statusowej aktualnie przetwarzany katalog
	display_path( in_path );
	// daj czas systemowi
	Shared::idle();
	
	d_finfo.setFile( in_path );

	if(( FALSE == d_break ) && d_finfo.isDir() && !d_finfo.isSymLink() ) {
		if( d_finfo.isReadable() && d_finfo.isExecutable() ) {

			const QDir dir( in_path );

			// Pobieramy informacje o plikach/katalogach,
			// ktorych nazwa jest zgodna ze wzorcem.
			// To sa informacje tylko do wyswietlenia.
			const QFileInfoList* const file_list = dir.entryInfoList( d_what_cbx->currentText(), QDir::All, QDir::Unsorted );
			if( file_list ) {
				QFileInfoListIterator it( *file_list );
				while(( FALSE == d_break ) && ((	finfo = it.current() ) != 0 )) {
					if( Shared::is_regular_file( finfo->fileName() )) {
						++d_counter;
						display( finfo );
					}
					Shared::idle();
					++it;
				}
			}

			// Pobieramy informacje o wszystkich podkatalogach.
			// Musimy te podkatalogi tez przeszukac.
			const QFileInfoList* const dirs_list = dir.entryInfoList( QDir::Dirs, QDir::Unsorted );
			if( dirs_list ) {
				QFileInfoListIterator it( *dirs_list );
				while(( FALSE == d_break ) && (( finfo = it.current() ) != 0 )) {
					if( Shared::is_regular_file( finfo->fileName() )) {
						if( "/" == in_path ) parse( in_path + finfo->fileName() );
						else                 parse( in_path + "/" + finfo->fileName() );
					}
					Shared::idle();
					++it;
				}
			}
		}
	}
}
// end of parse

//*******************************************************************
// befor                                                     PRIVATE
//-------------------------------------------------------------------
// Przygotowanie wszystkiego do czytania dysku.
//*******************************************************************
void Searcher::befor()
{
	d_break = FALSE;
	d_counter = 0;
	d_table->clear();

	QApplication::setOverrideCursor( Qt::WaitCursor );
	Shared::idle();
}
// end of befor

//*******************************************************************
// after                                                     PRIVATE
//-------------------------------------------------------------------
// Porzadki po czytaniu dysku.
//*******************************************************************
void Searcher::after()
{
	QApplication::restoreOverrideCursor();
	slot_break();
	display_path( tr(END_OF_SEARCH).arg( QString::number( d_counter )));
	d_table->setCurrentItem ( d_table->firstChild() );
	d_table->setFocus();
	Shared::idle();
}
// end of after

//*******************************************************************
// display                                                   PRIVATE
//*******************************************************************
void Searcher::display( const QFileInfo* const in_finfo )
{
	if( in_finfo->isReadable() ) {
		new QListViewItem( d_table, in_finfo->fileName(), in_finfo->dirPath() );
		d_table->adjustColumn( 0 );
		d_table->adjustColumn( 1 );
	}
}
// end of display

//*******************************************************************
// display_path                                              PRIVATE
//*******************************************************************
void Searcher::display_path( const QString& in_path )
{
	static QFontMetrics fm( font() );
	QString path = in_path;
	
	Shared::clip_path( fm, d_info_line->width(), path );
	d_info_line->setText( path );
	Shared::idle();
}
// end of display_path

