/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Copier.h"
#include "Shared.h"
#include "Config.h"
#include <qdir.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qpushbutton.h>
#include "CanOverwrite.h"
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const char* const Copier::MkdirError      = QT_TR_NOOP( "Can't create the directory: %1" );
const char* const Copier::Rename          = QT_TR_NOOP( "Rename" );
const char* const Copier::NewFileName     = QT_TR_NOOP( "New file name:" );
const char* const Copier::NotReadableFile = QT_TR_NOOP( "You can't read from this file: %1.\nCheck permission access." );
const char* const Copier::NotWriteableDir = QT_TR_NOOP( "You can't wite to this directory: %1.\nCheck permission access." );


//*******************************************************************
// OneFileCopier
//*******************************************************************
Copier::Copier  ( QWidget* const in_parent, const ViewTable::SelectedItems& in_items, const QString& in_dst_dir )
: CopyDialog    ( in_parent, FALSE )
, d_items       ( in_items )
, d_dst_dir     ( in_dst_dir )
, d_block_size  ( BufferSize )
, d_ask_again   ( TRUE )
, d_break       ( FALSE )
, d_running     ( FALSE )
{
    connect( d_left_btn, SIGNAL( clicked() ), this, SLOT( run() ));
    connect( d_right_btn, SIGNAL( clicked() ), this, SLOT( cancel() ));
}
// end of Copier

//*******************************************************************
// cancel                                               PRIVATE slot
//*******************************************************************
void Copier::cancel()
{
    if( d_running ) {
        d_break = TRUE;
    }
    else {
        accept();
    }
}
// end of cancel


//###################################################################
//#                                                                 #
//#                     P R O C E S S I N G                         #
//#                                                                 #
//###################################################################


//*******************************************************************
// run                                                  PRIVATE slot
//*******************************************************************
void Copier::run()
{
    // Uzytkownik nie ma uprawnien do zapisu w katalu docelowym.
    if( !Shared::is_dir_ok( d_dst_dir )) {
        QMessageBox::warning( this, Shared::ProgramName, tr( NotWriteableDir ).arg( d_dst_dir ));
        reject();
    }
    // Uzytkownik ma prawo zapisu do katalogu docelowego.
    else {
        runned();
        d_running = TRUE;

        ViewTable::SelectedItems::const_iterator it = d_items.begin();
        while( !d_break && ( it != d_items.end() )) {
            copy_next( (*it)->fi(), d_dst_dir );
            ++it;
        }
        accept();
    }
}
// end of run

//*******************************************************************
// copy_next                                                 PRIVATE
//*******************************************************************
void Copier::copy_next( const QFileInfo& in_fi, const QString& in_dst_dir )
{
    Shared::idle();
	
    if( !d_break ) {
        if( Shared::is_regular_file( in_fi.fileName() )) {
            // KATALOG ---------------------------------------------
            if( in_fi.isDir() ) {
                const QDir dir( in_fi.absFilePath() );
                if( dir.isReadable() ) {
                    QDir dst( in_dst_dir );
                    dst.mkdir( in_fi.fileName(), FALSE );
					if( dst.cd( in_fi.fileName(), FALSE )) {
                        copy_subdir( in_fi.absFilePath(), dst.absPath() );
                    }
                    else {
                        //  Nie mozna utworzyc podkatalogu.
                        QMessageBox::warning( this, Shared::ProgramName, tr( MkdirError ).arg( in_fi.fileName() ));
					}
				}
                else {
                    // Brak praw dostepu. Nic nierob. Wracamy.
                    QMessageBox::warning( this, Shared::ProgramName, tr(Shared::NotReadableDir).arg( in_fi.absFilePath() ));
				}
            }
            // NORMALNY PLIK ---------------------------------------
            else {
                copy_file( in_fi.absFilePath(), in_dst_dir );
            }
        }
    }
}
// end of copy_next

//*******************************************************************
// copy_subdir                                               PRIVATE
//*******************************************************************
void Copier::copy_subdir( const QString& in_src_dir, const QString& in_dst_dir )
{
	if( !d_break ) {
		QDir dir( in_src_dir );
		QFileInfoListIterator it( *dir.entryInfoList( QDir::All | QDir::Hidden | QDir::System ));
	
		while( it.current() ) {
			copy_next( *it.current(), in_dst_dir );
			++it;
		}

		if( d_remove_cb->isChecked() ) {
			Shared::remove_path_and_content( in_src_dir );
		}
	}
}
// end of copy_subdir

//*******************************************************************
// can_copy                                                  PRIVATE
//*******************************************************************
bool Copier::can_copy( const QString& in_src, QString& in_dst )
{
    bool       retval = TRUE;
    static int answer = CanOverwrite::OVER;
	
    if( d_ask_again ) {
        if( QFile::exists( in_dst ) ) {
            CanOverwrite* dialog = new CanOverwrite( this, in_dst );
            if( dialog ) {
                answer = dialog->exec();
                if( answer != CanOverwrite::RENAME ) {
                    d_ask_again = dialog->ask_again();
                }
                delete dialog;
                dialog = 0;
            }
        }
    }

    switch( answer ) {
        case CanOverwrite::RENAME:
            rename( in_dst );
            break;
        case CanOverwrite::OVER:
            break;
        case CanOverwrite::CANCEL:
            d_break = TRUE;
		case CanOverwrite::SKIP:
            retval = FALSE;
            break;
        case CanOverwrite::UPDATE:
            retval = can_update( in_src, in_dst );
            break;
    }
    return retval;
}
// end of can_copy

//*******************************************************************
// rename                                                    PRIVATE
//*******************************************************************
void Copier::rename( QString& inout_fpath )
{
    QString fpath = inout_fpath;
    Shared::auto_rename( fpath );
	
    const QFileInfo fi( fpath );
    const QString fname = fi.fileName();

    bool ok;
    const QString new_fname = QInputDialog::getText( tr( Rename ), tr( NewFileName ), QLineEdit::Normal, fname, &ok, this );
    if( ok && !new_fname.isEmpty() ) {
        QString fpath = fi.dirPath( TRUE );
        if( !fpath.endsWith( "/" )) fpath += "/";
        inout_fpath = fpath + new_fname;
    }
}
// end of rename

//*******************************************************************
// can_update                                                PRIVATE
//*******************************************************************
bool Copier::can_update( const QString& in_src, const QString& in_dst )
{
    const QFileInfo src_fi( in_src );
    const QFileInfo dst_fi( in_dst );
    return ( src_fi.lastModified() > dst_fi.lastModified() );
}
// end of can_update

//*******************************************************************
// copy_file                                                 PRIVATE
//-------------------------------------------------------------------
// Kopiowanie pliku z jednego katalogu do innego.
//
// PARAMETRY:
// 'in_src_path' - Sciezka pliku zrodlowego (czyli katalog + plik )
//                 np. /home/piotr/gnu/nazwa_pliku
// 'in_dst_dir'  - Katalog, do ktorego ma byc przekopiowany plik
//                 np. /home/piotr/backup/
//*******************************************************************
void Copier::copy_file( const QString& in_src_path, const QString& in_dst_dir )
{
    const QFileInfo fi( in_src_path );
	
    if( !fi.isReadable() ) {
        QMessageBox::warning( this, Shared::ProgramName, tr( NotReadableFile ).arg( in_src_path ));
    }
    else {
        // Tworzymy sciezke pliku wynikowego.
        // Wyodrebniamy ze sciezki zrodlowej tylko nazwe pliku i
        // dodajemy ja do katalogu docelowego.
        QString dst_path = in_dst_dir + "/" + QFileInfo( in_src_path ).fileName();

        // Pokazujemy uzytkownikowi co bedziemy kopiowac.
        display_info( in_src_path, dst_path );

        // Najpierw sprawdzamy czy w ogole mozemy kopiowac plik.
        // Jezeli sa jakiekolwiek watpliwosci (np. plik juz istnieje)
        // zgode na kopiowanie musi wydac uzytkownik.
        if( can_copy( in_src_path, dst_path ) ) {
            // Po uzyskaniu zgody na kopiowanie bierzemy sie do roboty.
            QFile in( in_src_path );
            QFile out( dst_path );
            if( in.open( IO_ReadOnly ) ) {
                const unsigned int nbytes = in.size();	// Liczba bajtow do przekopiowania.
                unsigned int copied = 0;                // Licznik aktualnie przekopiowanych bajtow.
			
				if( TRUE == out.open( IO_WriteOnly | IO_Truncate ) ) {
                    reset_progress();
				
                    // Petla kopiujaca.
                    while( !in.atEnd() ) {
                        const unsigned int n = in.readBlock( d_buffer, d_block_size );
                        out.writeBlock( d_buffer, n );
						set_progress( copied += n, nbytes );
					}
					
					// czynnosci po kopiowaniu
                    set_progress( nbytes, nbytes );
					out.close();
                    if( d_remove_cb->isChecked() ) {
                        Shared::remove_file( in_src_path );
                    }
                }
                else {
                    read_open_error( dst_path );
                }
                in.close();
            }
            else {
                write_open_error( in_src_path );
            }
        }
    }
}
// end of copy_file
