/*
 * Copyright (C) 1999  Rob Crittenden (rcrit@greyoak.com)
 * Copyright (C) 1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define BNETTIME_INTERNAL_ACCESS
#include <stddef.h>
#include <stdio.h>
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <errno.h>
#include "compat/strerror.h"
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include "eventlog.h"
#include "bn_type.h"
#include "bnettime.h"


/*
 * By comparing the hex numbers from packet dumps with the times displayed
 * on the login screen, bnettime seems to be in units of 10E-7 seconds. It
 * is stored in a 64bit value, and represented as two distinct numbers when
 * it is used as a string in the protocol.
 *
 * example: 01/03/99 05:13    0x01be370ae40228e0    "29243146 3825346784"
 *
 * The top number seems to be in units of about 7 minutes because the bottom
 * number rolls over every 2^32 10E-7 seconds, which is:
 * (2^32)*.0000001
 * 429.4967296 // seconds
 * 429/60
 * 7.15827882666666666666 // minutes
 *
 * The epoch was determined using a binary search looking for Jan 1, 1970 to
 * be displayed after setting the last game time to a number. It was only
 * determined to the first 32 bits. The last 32 bits must be close to zero
 * though because it already seems to be pretty accurate.
 *
 * If I was clever I should be able to do this without floating point math.
 * I don't feel like being clever though :)
 */

#define SEC_PER_USEC .0000001

#define UNIX_EPOCH_UPPER 19764578. /* 0:00 1 Jan 1970 GMT */
#define UNIX_EPOCH_LOWER 0.

#define SEC_PER_UPPER 429.4967296 /* (2^32)*10E-7 */
#define SEC_PER_LOWER .0000001

#define UPPER_PER_SEC .00232830643653869628  /* 10E7/(2^32) */
#define LOWER_PER_SEC 10000000.


extern t_bnettime time_to_bnettime(time_t stdtime, unsigned int usec)
{
    double     secs=(double)stdtime+(double)usec*SEC_PER_USEC;
    t_bnettime temp;
    
    temp.u = (unsigned int)(UNIX_EPOCH_UPPER+secs*UPPER_PER_SEC);
    temp.l = (unsigned int)(UNIX_EPOCH_LOWER+secs*LOWER_PER_SEC);
    
    return temp;
}


extern time_t bnettime_to_time(t_bnettime bntime)
{
    double secs;
    
    secs = (bntime.u-UNIX_EPOCH_UPPER)*SEC_PER_UPPER+
           (bntime.l-UNIX_EPOCH_LOWER)*SEC_PER_LOWER;
    
    return (time_t)secs;
}


/* return current time as bnettime */
extern t_bnettime bnettime(void)
{
    struct timeval tv;
    
    if (gettimeofday(&tv,NULL)<0)
    {
        eventlog(eventlog_level_error,"bnettime","could not get time (gettimeofday: %s)",strerror(errno));
        return time_to_bnettime(time(NULL),0);
    }
    return time_to_bnettime(tv.tv_sec,tv.tv_usec);
}


/* FIXME: the string functions here should probably go in account_wrap */
extern char const * bnettime_get_str(t_bnettime bntime)
{
    static char temp[1024];
    
    sprintf(temp,"%u %u",bntime.u,bntime.l);
    
    return temp;
}


extern int bnettime_set_str(t_bnettime * bntime, char const * timestr)
{
    if (!bntime)
    {
	eventlog(eventlog_level_error,"bnettime_set_str","got NULL bntime");
	return -1;
    }
    if (!timestr)
    {
	eventlog(eventlog_level_error,"bnettime_set_str","got NULL timestr");
	return -1;
    }
    
    if (sscanf(timestr,"%u %u",&bntime->u,&bntime->l)!=2)
	return -1;
    
    return 0;
}


extern void bnettime_to_bn_long(t_bnettime in, bn_long * out)
{
    if (!out)
    {
	eventlog(eventlog_level_error,"bnettime_to_bn_long","got NULL out");
	return;
    }
    
    bn_long_set_a_b(out,in.u,in.l);
}


extern void bn_long_to_bnettime(bn_long in, t_bnettime * out)
{
    if (!out)
    {
	eventlog(eventlog_level_error,"bn_long_to_bnettime","got NULL out");
	return;
    }
    
    out->u = bn_long_get_a(in);
    out->l = bn_long_get_b(in);
}
