/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/gc_profile.c            */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Fri May 14 06:49:17 1999                          */
/*    Last change :  Fri Nov 14 21:30:25 2003 (serrano)                */
/*    -------------------------------------------------------------    */
/*    The GC profiler. This file is not specific to the Boehm's        */
/*    collector.                                                       */
/*=====================================================================*/
#include <bigloo.h>
#include <string.h>

/*---------------------------------------------------------------------*/
/*    Importations                                                     */
/*---------------------------------------------------------------------*/
extern void GC_COLLECT();
extern void GC_gcollect();

/*---------------------------------------------------------------------*/
/*    The global allocation counter                                    */
/*---------------------------------------------------------------------*/
long GC_words_allocd_byte = 0;

/*---------------------------------------------------------------------*/
/*    MEGABYTE                                                         */
/*---------------------------------------------------------------------*/
#define MEGABYTE( val ) ((double)(val) / (1024.0 * 1024.0))

static void (*bgl_heap_debug_mark_obj_hook)() = 0;
static void (*bgl_heap_debug_mark_str_hook)() = 0;

/*---------------------------------------------------------------------*/
/*    ppair_t ...                                                      */
/*    -------------------------------------------------------------    */
/*    This type is only used for profile stamp recording.              */
/*---------------------------------------------------------------------*/
typedef struct ppair {
   char *ident;
   struct ppair *next;
} *ppair_t;
  
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    GC_profile_stamp ...                                             */
/*    -------------------------------------------------------------    */
/*    The stack of profile stamps.                                     */
/*---------------------------------------------------------------------*/
static ppair_t GC_profile_stamp = 0;
static ppair_t GC_profile_all_stamp = 0;

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bgl_heap_debug_mark_hooks_set ...                                */
/*---------------------------------------------------------------------*/
void
bgl_heap_debug_mark_hooks_set( void (*o)(), void (*b)() ) {
   bgl_heap_debug_mark_obj_hook = o;
   bgl_heap_debug_mark_str_hook = b;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_heap_debug_mark_obj ...                                      */
/*    -------------------------------------------------------------    */
/*    This function takes a typed Bigloo object and increment a        */
/*    counter associated to that type.                                 */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
bgl_heap_debug_mark_obj( obj_t obj ) {
   if( bgl_heap_debug_mark_obj_hook )
      bgl_heap_debug_mark_obj_hook( obj );

   return obj;
}

/*---------------------------------------------------------------------*/
/*    char *                                                           */
/*    bgl_heap_debug_mark_str ...                                      */
/*    -------------------------------------------------------------    */
/*    This function mark a string buffer as allocated.                 */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
char *
bgl_heap_debug_mark_str( char *buf ) {
   if( bgl_heap_debug_mark_str_hook )
      bgl_heap_debug_mark_str_hook( buf );
 
   return buf;
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    GC_profile_push ...                                              */
/*    -------------------------------------------------------------    */
/*    The dummy argument is used to break the Bigloo optimization.     */
/*    It is mandatory that the function introduced inside the          */
/*    profile form is really compiled into a C function. To get        */
/*    that, the function is passed as an argument in                   */
/*    GC_profile_push but not unsed here.                              */
/*    -------------------------------------------------------------    */
/*    When entering a profile, we force a GC.                          */
/*---------------------------------------------------------------------*/
long
GC_profile_push( char *ident, obj_t dummy ) {
   ppair_t new;

   new = malloc( sizeof( struct ppair ) );

   new->ident = strdup( ident );
   new->next  = GC_profile_stamp;

   GC_profile_all_stamp = GC_profile_stamp = new;
   
   return 0;
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    GC_collect_profile_push ...                                      */
/*    -------------------------------------------------------------    */
/*    The same function as GC_profile_push but force a collection.     */
/*---------------------------------------------------------------------*/
long
GC_collect_profile_push( char *ident, obj_t dummy ) {
   GC_profile_push( ident, dummy );
   
   GC_COLLECT();
   
   return 0;
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    GC_profile_pop ...                                               */
/*---------------------------------------------------------------------*/
long
GC_profile_pop() {
   if( GC_profile_stamp ) GC_profile_stamp = GC_profile_stamp->next;

   return 0;
}

/*---------------------------------------------------------------------*/
/*    gc_profile_info_t                                                */
/*---------------------------------------------------------------------*/
struct gc_profile_info_t {
   long          num;
   long          heap;
   long          alloc;
   unsigned long stack;
   unsigned long live;
   ppair_t       stamp;
};

/*---------------------------------------------------------------------*/
/*    get_top_of_stack                                                 */
/*    -------------------------------------------------------------    */
/*    This function is defined in the Call/cc implementation.          */
/*---------------------------------------------------------------------*/
extern unsigned long get_top_of_stack();

/*---------------------------------------------------------------------*/
/*    GC_PROFILE_VECTOR_INCREASE                                       */
/*    -------------------------------------------------------------    */
/*    How many gc_profile_info_t struct shall we add on each           */
/*    increase (when the vector is full).                              */
/*---------------------------------------------------------------------*/
#define GC_PROFILE_VECTOR_INCREASE 100

/*---------------------------------------------------------------------*/
/*    struct gc_profile_info_t *                                       */
/*    gc_profile_vector ...                                            */
/*---------------------------------------------------------------------*/
static struct gc_profile_info_t *gc_profile_vector = 0L;
static long   gc_profile_len = 0L;
static long   gc_profile_num;

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    gc_profile_realloc_vector ...                                    */
/*---------------------------------------------------------------------*/
static void
gc_profile_realloc_vector() {
   gc_profile_len += GC_PROFILE_VECTOR_INCREASE;

   gc_profile_vector = (struct gc_profile_info_t *)
      realloc( gc_profile_vector,
	       gc_profile_len * sizeof( struct gc_profile_info_t ) );
}

/*---------------------------------------------------------------------*/
/*    unsigned long                                                    */
/*    gc_profile_stack_base ...                                        */
/*---------------------------------------------------------------------*/
static unsigned long gc_profile_stack_base = 0;

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    GC_profile_start ...                                             */
/*---------------------------------------------------------------------*/
void GC_profile_start() {
   gc_profile_len = 0;
   gc_profile_num = 0;
   gc_profile_stack_base = get_top_of_stack();
}

/*---------------------------------------------------------------------*/
/*    unsigned long                                                    */
/*    GC_profile_stack_size ...                                        */
/*---------------------------------------------------------------------*/
unsigned long GC_profile_stack_size( unsigned long stack_top ) {
#if( STACK_GROWS_DOWN )
   return ( gc_profile_stack_base - stack_top );
#else 
   return ( stack_top - gc_profile_stack_base );
#endif
}
   
/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    GC_profile ...                                                   */
/*    -------------------------------------------------------------    */
/*    This function is called on every collection.                     */
/*       GC_NUM: the number of the collection                          */
/*       GC_HEAP: the size of the heap                                 */
/*       GC_ALLOCATED: the number of bytes allocated since last GC     */
/*---------------------------------------------------------------------*/
void GC_profile( long gc_num, long gc_heap, long gc_allocated, unsigned long live ) {
   if( gc_num >= gc_profile_len ) {
      gc_profile_realloc_vector();
   }

   gc_profile_vector[ gc_num ].num   = gc_num;
   gc_profile_vector[ gc_num ].heap  = gc_heap;
   gc_profile_vector[ gc_num ].alloc = gc_allocated;
   gc_profile_vector[ gc_num ].stack = get_top_of_stack();
   gc_profile_vector[ gc_num ].live  = live;
   gc_profile_vector[ gc_num ].stamp = GC_profile_stamp;

   gc_profile_num = gc_num;
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    GC_profile_stop ...                                              */
/*---------------------------------------------------------------------*/
long GC_profile_stop() {
#ifdef PROFILE
   extern obj_t bprof_port;
   FILE *file = (FILE *)bprof_port;
   long i;
   ppair_t runner;

   /* We have to force a last GC otherwise we are missing allocations */
   GC_gcollect();
   
   fprintf( file, "\n#a012\n\n" );
   fputs( "(gc \n", file );
   for( i = 0; i <= gc_profile_num; i++ ) {
      fprintf( file, "   (%d %#.2f %#.2f %#.2f %#.2f",
	       gc_profile_vector[ i ].num,
	       MEGABYTE( gc_profile_vector[ i ].heap ),
	       MEGABYTE( gc_profile_vector[ i ].alloc ),
	       MEGABYTE( GC_profile_stack_size( gc_profile_vector[ i ].stack )),
	       MEGABYTE( gc_profile_vector[ i ].live ) );

      for( runner = gc_profile_vector[i].stamp; runner; runner = runner->next )
	 fprintf( file, " %s", runner->ident );

      fputs( ")\n", file );
   }
   fputs( "   )\n", file );

   /* We now free the stamp list */
   runner = GC_profile_all_stamp;
   while( runner ) {
      ppair_t old = runner;
      free( old->ident );
      runner = runner->next;
      free( old );
   }
   GC_profile_all_stamp = 0L;
   
   /* And the profile vector */
   free( gc_profile_vector );
   gc_profile_len = gc_profile_num = 0;

   return 0;
#endif
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    GC_profile_init ...                                              */
/*---------------------------------------------------------------------*/
void
GC_profile_init() {
   static int init = 0;

   if( !init ) {
      init = 1;
      atexit( (void (*)(void))GC_profile_stop );
   }
}
 
/*---------------------------------------------------------------------*/
/*    GC_malloc profiling                                              */
/*    -------------------------------------------------------------    */
/*    It has been found necessary to have a precise GC_MALLOC          */
/*    profiling tool. The [GC_profile_alloc] stores in a table         */
/*    each allocation site and the accumulated allocated size of that  */
/*    site.                                                            */
/*---------------------------------------------------------------------*/
#if defined( BIGLOO_GC_MALLOC_PROFILE )

struct minfo {
   char *fname;
   int line;
   int num;
   long size;
   struct minfo *next;
};

static struct minfo *the_minfo = 0L;

/*---------------------------------------------------------------------*/
/*    struct minfo *                                                   */
/*    find_minfo ...                                                   */
/*---------------------------------------------------------------------*/
static struct minfo *
find_minfo( char *file, int line ) {
   struct minfo *runner = the_minfo;

   while( runner ) {
      if( runner->line == line && !strcmp( runner->fname, file ) )
	 return runner;

      runner = runner->next;
   }

   return 0L;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    GC_profile_alloc ...                                             */
/*---------------------------------------------------------------------*/
void
GC_profile_alloc( size_t size, char *file, int line ) {
   struct minfo *minfo = find_minfo( file, line );

   if( !minfo ) {
      minfo = malloc( sizeof( struct minfo ) );
      minfo->fname = file;
      minfo->line  = line;
      minfo->num = 0;
      minfo->size = 0;
      minfo->next  = the_minfo;

      the_minfo = minfo;
   }

   minfo->num += 1;
   minfo->size += size;
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    compare_minfo ...                                                */
/*---------------------------------------------------------------------*/
int
compare_minfo( struct minfo **m1, struct minfo **m2 ) {
   return (*m1)->size < (*m2)->size;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    GC_profile_dump_alloc_table ...                                  */
/*---------------------------------------------------------------------*/
void
GC_profile_dump_alloc_table() {
   struct minfo *runner = the_minfo;
   struct minfo **tab;
   long len = 0;
   long i = 0;
   long sum = 0;

   while( runner ) len++, runner = runner->next;
   runner = the_minfo;
   
   tab = malloc( sizeof( struct minfo * ) * len );

   while( runner ) {
      tab[ i++ ] = runner;
      runner = runner->next;
   }

   qsort( tab, len, sizeof( struct minfo * ), compare_minfo );

   printf( "*** Global allocated size: %7.2f (%d bytes)\n",
	   GC_words_allocd_byte / (1024. * 1024.),
	   GC_words_allocd_byte );
   
   for( i = 0; i < len; i ++ ) {
      runner = tab[ i ];

      sum += runner->size;
      printf( "Sum: %7.2f (%8d bytes)  Alloc size: %7.2f (%8d bytes)  call: %6d  file: %s  line: %d\n",
	      sum / (1024. * 1024.),
	      sum,
	      runner->size / (1024. * 1024.),
	      runner->size,
	      runner->num,
	      runner->fname,
	      runner->line );
   }
}
/*---------------------------------------------------------------------*/
/*    void *                                                           */
/*    GC_profile_malloc ...                                            */
/*---------------------------------------------------------------------*/
void *GC_profile_malloc( size_t size, char *file, int line ) {
   GC_profile_alloc( size, file, line );
   return GC_malloc( size );
}
   
/*---------------------------------------------------------------------*/
/*    void *                                                           */
/*    GC_profile_malloc_atomic ...                                     */
/*---------------------------------------------------------------------*/
void *GC_profile_malloc_atomic( size_t size, char *file, int line ) {
   GC_profile_alloc( size, file, line );
   return GC_malloc_atomic( size );
}
#endif
