(module jas_produce
   (import jas_classfile)
   (export (produce ::binary-port ::obj) ))

;;
;; Binary output
;;
(define (outchar outchan n)
   (output-char outchan n) )

(define (outbyte outchan n)
   (outchar outchan (integer->char (bit-and n #xFF))) )

(define (outshort outchan n)
   (outbyte outchan (bit-rsh n 8))
   (outbyte outchan (bit-and n #xFF)) )

(define (outint outchan n)
   (outshort outchan (bit-rsh n 16))
   (outshort outchan (bit-and n #xFFFF)) )

(define (produce outchan classfile)
   (with-access::classfile classfile (pool pool-size flags me super interfaces
					   fields methods attributes )
      (outshort outchan #xCAFE)
      (outshort outchan #xBABE)
      (outshort outchan 3)
      (outshort outchan 45)
      (produce-pool outchan pool-size (reverse pool))
      (outshort outchan flags)
      (outshort outchan me)
      (outshort outchan super)
      (produce-interfaces outchan interfaces)
      (produce-fields outchan fields)
      (produce-methods outchan methods)
      (produce-attributes outchan attributes) ))

(define (produce-pool outchan size pool)
   (outshort outchan size)
   (for-each (lambda (cpinfo) (produce-cpinfo outchan cpinfo)) pool) )

(define (produce-cpinfo outchan cpinfo)
   (outbyte outchan (car cpinfo))
   (case (car cpinfo)
      ((1)
       (let ( (str (cdr cpinfo)) )
	  (outshort outchan (utf8-length str))
	  (produce-utf8 outchan str) ))
      ((3 4 5 6 7 8 9 10 11 12)
       (for-each (lambda (n) (outshort outchan n)) (cdr cpinfo)) )
      (else (error "as" "bad cpinfo tag" (car cpinfo))) ))

(define (sref s i)
   (char->integer (string-ref s i)) )

(define (utf8-length s)
   (let ( (n (string-length s)) )
      (define (walk s i r)
	 (if (=fx i n)
	     r
	     (walk s (+fx i 1) (+fx r (utf8-length1 (sref s i)))) ))
      (walk s 0 0) ))

(define (utf8-length1 cn)
   (cond
      ((= cn 0) 2)
      ((< cn #x80) 1)
      ((< cn #x800) 2)
      (else 3) ))

(define (produce-utf8 outchan s)
   (let ( (n (string-length s)) )
      (define (walk s i)
	 (if (=fx i n)
	     'ok
	     (begin (produce-utf8-1 outchan (sref s i))
		    (walk s (+fx i 1)) )))
      (walk s 0) ))

(define (produce-utf8-1 outchan cn)
   (cond
      ((= cn 0)
       (outbyte outchan #xC0)
       (outbyte outchan #x80) )
      ((< cn #x80)
       (outbyte outchan cn) )
      ((< cn #x800)
       (outbyte outchan (bit-or #xC0 (bit-rsh cn 6)))
       (outbyte outchan (bit-or #x80 (bit-and cn #x3F))) )
      (else
       (outbyte outchan (bit-or #xE0 (bit-rsh cn 12)))
       (outbyte outchan (bit-or #x80 (bit-and (bit-rsh cn 6) #x3F)))
       (outbyte outchan (bit-or #x80 (bit-and cn #x3F))) )))

(define (produce-interfaces outchan interfaces)
   (outshort outchan (length interfaces))
   (for-each (lambda (n) (outshort outchan n)) interfaces) )

(define (produce-fields outchan fields)
   (outshort outchan (length fields))
   (for-each (lambda (field) (produce-field outchan field)) fields) )

(define (produce-field outchan field)
   (with-access::field field (flags name pname descriptor attributes)
      (outshort outchan flags)
      (outshort outchan pname)
      (outshort outchan descriptor)
      (produce-attributes outchan attributes) ))

(define (produce-methods outchan methods)
   (outshort outchan (length methods))
   (for-each (lambda (method) (produce-method outchan method)) methods) )

(define (produce-method outchan method)
   (with-access::method method (flags pname descriptor attributes)
      (outshort outchan flags)
      (outshort outchan pname)
      (outshort outchan descriptor)
      (produce-attributes outchan attributes) ))

(define (produce-attributes outchan attributes)
   (outshort outchan (length attributes))
   (for-each (lambda (attribute) (produce-attribute outchan attribute))
	     attributes ))

(define (produce-attribute outchan attribute)
   (outshort outchan (attribute-name attribute))
   (outint outchan (attribute-size attribute))
   (let ( (info (attribute-info attribute)) (type (attribute-type attribute)) )
      (case type
	 ((code)
	  (produce-code outchan info) )
	 ((srcfile)
	  (outshort outchan info) )
	 ((linenumber)
	  (outshort outchan (length info))
	  (for-each (lambda (slot) (outshort outchan (car slot))
			           (outshort outchan (cdr slot)) )
		    info ))
	 ((localvariable)
	  (outshort outchan (length info))
	  (for-each (lambda (slot) (for-each (lambda (n) (outshort outchan n))
					     slot ))
		    info ))
	 (else (error "produce-attribute" "bad attribute type" type)) )))

(define (produce-code outchan code)
   (match-case code
      ((?maxstk ?maxlocal ?bytecode  ?handlers . ?attributes)
       (outshort outchan maxstk)
       (outshort outchan maxlocal)
       (outint outchan (length bytecode))
       (for-each (lambda (b) (outbyte outchan b)) bytecode)
       (outshort outchan (length handlers))
       (for-each (lambda (h)
		    (match-case h
		       ((?beg ?end ?label ?type)
			(outshort outchan beg)
			(outshort outchan end)
			(outshort outchan label)
			(outshort outchan type) )
		       (else (error "produce-code" "bad handler" h)) ))
		handlers )
       (produce-attributes outchan attributes) )
      (else (error "produce-code" "bad code attribute" code)) ))
