/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "versification.h"
#include <config.h>
#include "utilities.h"
#include "bible.h"
#include "books.h"
#include "sqlite_reader.h"
#include "gwrappers.h"
#include "directories.h"


ustring versification_get_filename (ustring name)
{
  replace_text (name, " ", "_");
  name.insert (0, "vrs_");
  name.append (".sql");
  return gw_build_filename (directories_get_templates (), name.casefold ());
}


ustring versification_get_name (ustring path)
{
  path = gw_path_get_basename (path);
  ustring name (path);
  name.erase (0, 4);
  name.erase (name.length() - 4, 4);
  replace_text (name, "_", " ");
  size_t pos = 0;
  while (pos != string::npos) {
    name.replace (pos, 1, upperCase (name.substr (pos, 1)));
    pos = name.find (" ", pos);
    if (pos != string::npos) pos++;
  }
  return name;
}


vector<ustring> versification_get_raw_templates_available ()
{
  vector <ustring> paths;
  ReadFiles rf (PACKAGE_DATA_DIR, "vrs_", ".txt");
  for (unsigned int i = 0; i < rf.files.size(); i++) {
    paths.push_back (gw_build_filename (PACKAGE_DATA_DIR, rf.files[i]));
  }
  ReadFiles rf2 (directories_get_templates_user (), "vrs_", ".txt");
  for (unsigned int i = 0; i < rf2.files.size(); i++) {
    paths.push_back (gw_build_filename (directories_get_templates_user (), rf2.files[i]));
  }
  return paths;
}


void versification_get_ones_available (vector<ustring>& versifications)
{
  versifications.clear();
  ReadFiles rf (directories_get_templates (), "vrs_", ".sql");
  for (unsigned int i = 0; i < rf.files.size(); i++) {
    versifications.push_back (versification_get_name (rf.files[i]));
  }
  sort (versifications.begin(), versifications.end());
}


void versification_get_chapters_verses (const ustring& versification, 
                                        const ustring& book,
                                        vector<unsigned int>& chapters,
                                        vector<unsigned int>& verses)
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(versification_get_filename (versification).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select chapter, verse from verse where book = %d;", books_english_to_id (book));
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    for (unsigned int i = 0; i < sqlitereader.ustring0.size(); i++) {
      chapters.push_back (convert_to_int (sqlitereader.ustring0[i]));
      verses.push_back (convert_to_int (sqlitereader.ustring1[i]));
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
  quick_sort (chapters, verses, 0, chapters.size());
}


void versification_create_book_template (const ustring& versification, const ustring& book, vector<ustring>& booktemplate)
{
  // Clear template and add id and running header.
  booktemplate.clear();
  unsigned index = books_english_to_id (book);
  ustring id = books_id_to_paratext (index);
  booktemplate.push_back ("\\id " + id);
  booktemplate.push_back ("\\h " + book);
  booktemplate.push_back ("\\toc2 " + book);
  // Get the number of chapters in the book and verses in the chapter.
  vector<unsigned int> chapters;
  vector<unsigned int> verses;
  versification_get_chapters_verses (versification, book, chapters, verses);
  // Add \c and \v information for all chapters.
  for (unsigned int i = 0; i < chapters.size(); i++) {
    booktemplate.push_back ("\\c " + convert_to_string (chapters[i]));
    // Some paragraph ought to follow the chapter, so insert it here.
    booktemplate.push_back ("\\p ");
    for (unsigned int verse = 1; verse <= verses[i]; verse++) {
      booktemplate.push_back ("\\v " + convert_to_string (verse));
    }
  }
}


void versification_create_chapter_template (const ustring& versification, const ustring& book, unsigned int chapter, vector<ustring>& chaptertemplate)
{
  chaptertemplate.clear();
  vector<unsigned int> chapters;
  vector<unsigned int> verses;
  versification_get_chapters_verses (versification, book, chapters, verses);
  for (unsigned int i = 0; i < chapters.size(); i++) {
    if (chapters[i] == chapter) {
      chaptertemplate.push_back ("\\c " + convert_to_string (chapter));
      chaptertemplate.push_back ("\\p ");
      for (unsigned int verse = 1; verse <= verses[i]; verse++) {
        chaptertemplate.push_back ("\\v " + convert_to_string (verse));
      }
    }
  }
}


vector <unsigned int> versification_get_chapters (const ustring& versification, const ustring& book)
{
  vector<unsigned int> chapters;
  vector<unsigned int> verses;
  versification_get_chapters_verses (versification, book, chapters, verses);
  return chapters;
}


ustring versification_get_last_verse (const ustring& versification, const ustring& book, unsigned int chapter)
{
  ustring lastverse = "1";
  vector<unsigned int> chapters;
  vector<unsigned int> verses;
  versification_get_chapters_verses (versification, book, chapters, verses);
  for (unsigned int i = 0; i < chapters.size(); i++) {
    if (chapters[i] == chapter) {
      lastverse = convert_to_string (verses[i]);
    }
  }
  return lastverse;
}


void versification_import_textfile (const ustring& name, const ustring& textfile)
// Imports a versification textfile
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try 
  {
    // Read the text.
    ReadText rt (textfile);
    // Open and create the database.
    unlink (versification_get_filename(name).c_str());
    rc = sqlite3_open (versification_get_filename(name).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    char * sql;
    sql = g_strdup_printf ("create table verse (book integer, chapter integer, verse integer);");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sql = g_strdup_printf ("PRAGMA synchronous=OFF;");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    // Go through the lines.
    cout << name << " " << rt.lines.size() << " lines" << endl;
    for (unsigned int i = 0; i < rt.lines.size(); i++) {
      // Skip comments.
      if (rt.lines[i].find ("#") != string::npos)
        continue;
      // Convert the line.
      ustring line = rt.lines[i];
      size_t pos = line.find_last_of (":");
      if (pos == string::npos) throw runtime_error (line);
      int verse = convert_to_int (line.substr (pos + 1, 5));
      line.erase (pos, 6);
      pos = line.find_last_of (" ");
      if (pos == string::npos) throw runtime_error (line);
      int chapter = convert_to_int (line.substr (pos + 1, 5));
      line.erase (pos, 6);
      line = trim (line);
      unsigned int id = book_find_valid (line);
      sql = g_strdup_printf ("insert into verse values (%d, %d, %d);", id, chapter, verse);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc) throw runtime_error (sqlite3_errmsg(db));
      if (id == 0) 
        throw runtime_error ("Unknown book");
    }    
  }
  catch (exception& ex) {
    unlink (versification_get_filename(name).c_str());
    gw_critical (ex.what ());
  }    
  sqlite3_close (db);
}
