------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--       G N A T C H E C K . R U L E S . M E T R I C S . C H E C K S        --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                       Copyright (C) 2008, AdaCore                        --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

--  This package defines the metric-specific rules. These rules are not
--  directltly visible for a gnatcheck user, they are used as an implementation
--  of the Metric rule

with ASIS_UL.Metrics.Definitions; use ASIS_UL.Metrics.Definitions;

package Gnatcheck.Rules.Metrics.Checks is

   type Metric_Value_Checks is (Smaller_Then, Greater_Then);

   type Metric_Checks_Rule_Type is abstract new Internal_Rule_Template with
      record
         Metric_Value_Check : Metric_Value_Checks;
      --  If this field is set to Smaller_Then, then all the cases when
      --  the computed metric value is smaller then a specified limit are
      --  flagged, and the other way around
      end record;

   procedure Analyze_Metric_Parameter
     (Rule : in out Metric_Checks_Rule_Type'Class;
      Par  : String);
   --  Checks if the right metric value check is specified by the actual for
   --  Par (this actual should be a part of the rule parameter that follows the
   --  rule name, it starts either from "<" or from ">") suites the metric
   --  check. If it does - analyzes the rest of Par string and stores the
   --  metric limit. Generates the diagnostic message and disables the check
   --  for the metric if the metric value check does not suite the metric, or
   --  if the parameter cannot be interpreted as the limit value for the
   --  metric.

   procedure Set_Metric_Limit
     (Rule    : in out Metric_Checks_Rule_Type;
      Par     :        String;
      Success :    out Boolean) is abstract;
   --  Par is supposed to be a part of the rule parameter that follows '<' or
   --  '>'. Analyzes the Par string and if it can be interpreted as a limit for
   --  the metric value, stores it in Rule and sets Success ON, otherwise
   --  Success is set OFF.

   type Integer_Metric_Rule_Type is new Metric_Checks_Rule_Type with
      record
         Metric_Limit : Metric_Count;
      end record;
   --  For metrics counted in integers

   procedure Set_Metric_Limit
     (Rule    : in out Integer_Metric_Rule_Type;
      Par     :        String;
      Success :    out Boolean);
   --  Try to treat Par as a value of Metric_Count

   type Real_Metric_Rule_Type is new Metric_Checks_Rule_Type with record
      Metric_Limit : Float;
   end record;
   --  For metrics counted in floats

   procedure Set_Metric_Limit
     (Rule : in out Real_Metric_Rule_Type;
      Par     :        String;
      Success :    out Boolean);

   ---------------------------
   -- Cyclomatic_Complexity --
   ---------------------------

   type Cyclomatic_Complexity_Rule_Type is new Integer_Metric_Rule_Type with
      null record;

   procedure Rule_Check_Pre_Op
     (Rule    : in out Cyclomatic_Complexity_Rule_Type;
      Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Rule_Traversal_State);
   --  If cyclomatic complexity metric is applcable to the argument Element,
   --  checks if the metric value is not greater then is specified for this
   --  metric check. Flags a constract if this check fails.

   procedure Init_Rule (Rule : in out Cyclomatic_Complexity_Rule_Type);

   Cyclomatic_Complexity_Rule : aliased Cyclomatic_Complexity_Rule_Type;

   --------------------------
   -- Essential_Complexity --
   --------------------------

   type Essential_Complexity_Rule_Type is new Integer_Metric_Rule_Type with
      null record;

   procedure Rule_Check_Pre_Op
     (Rule    : in out Essential_Complexity_Rule_Type;
      Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Rule_Traversal_State);
   --  If essential complexity metric is applcable to the argument Element,
   --  checks if the metric value is not greater then is specified. Flags a
   --  constract if this check fails.

   procedure Init_Rule (Rule : in out Essential_Complexity_Rule_Type);

   Essential_Complexity_Rule : aliased Essential_Complexity_Rule_Type;

   -----------
   -- LSLOC --
   -----------

   type LSLOC_Rule_Type is new Integer_Metric_Rule_Type with null record;

   procedure Rule_Check_Pre_Op
     (Rule    : in out LSLOC_Rule_Type;
      Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Rule_Traversal_State);
   --  If LSLOC metric is applcable to the argument Element, checks if the
   --  metric value is not greater then is specified. Flags a constract if this
   --  check fails.

   procedure Init_Rule (Rule : in out LSLOC_Rule_Type);

   LSLOC_Rule : aliased LSLOC_Rule_Type;

end Gnatcheck.Rules.Metrics.Checks;
