#ifndef sourceh
#define sourceh

#include <glib.h>
#include <pthread.h>
#include <libxml/tree.h>

#include <recursive-mutex.h>
#include <notify.h>
#include <sample-type.h>
#include <volume.h>

struct _Source;
typedef struct _Source Source;

#include <throughput.h>
#include <source.h>
#include <block.h>
#include <block-queue.h>
#include <sink.h>
#include <origin.h>

struct _Source
{
  gchar *shortname;
  gchar *name;
  gchar *type; /*static string!*/

  SampleType sample_type;
  Volume volume;

  SourceMode mode;
  RecursiveMutex *mode_mutex;
  guint flags;
  
  pthread_t thread;
  gboolean thread_running;

  Throughput* throughput;

  OriginId origin_id;
  
  GSList *output_block_queues;
  RecursiveMutex *output_block_queues_mutex;

  Sink* direct_sink;

  Notify notify;

  gpointer private_data;
  gboolean (*open) (Source *);
  void (*close) (Source *);
  gboolean (*reopen) (Source *);
  glong (*read) (Source *, guint8*, gulong);
  glong (*direct_read) (Source *s, guint8*, gulong);
  Block* (*read_block) (Source *);
  void (*free_private) (Source *);
  void (*save_private) (Source *s, xmlNodePtr node);
};

// Creates an abstract instance of a source
Source* source_new(gchar *shortname, gchar *name, gboolean ns_reg);

// Adds/removes a block queue to/from the list of connected block queues
void source_add_output_block_queue(Source *s, BlockQueue *b);
void source_remove_output_block_queue(Source *s, BlockQueue *b);

// Convenience function for conjunction manager
void source_output_block_queue_death_proc(gpointer p, BlockQueue*b);

// PRIVATE: Writes a block to all block queues
gboolean source_output_write_block(Source *s, Block *b);

// PRIVATE: Wait until all connected block queues are empty
void source_output_block_queues_wait_until_empty(Source *s);

// PRIVATE: Enables all connected block queues
void source_output_block_queues_enable(Source *s, gboolean b);

// Starts the thread
void source_start(Source *s);

// Stops the thread 
void source_stop(Source *s);

// Stops the thread (if available) and removes all conjunctions
void source_kill(Source *s);

// PRIVATE: Sets the mode of the source
void source_set_mode(Source *s, SourceMode m);

// Return latency in msec
gulong source_get_latency(Source *s);

// Enabled/Disabled direct mode to sink
gboolean source_direct(Source *s, gboolean enable, Sink* sink);

// Sets the volume of the source
void source_set_volume(Source *s, Volume *v);

#endif
