/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __ENCODEROPTS_H__
#define __ENCODEROPTS_H__

#include <qstringlist.h>

/*========================================================*/

#ifdef FLAC
class ArsonFlacEncoderProcess;
#endif

class ArsonEncoderProcess;
class ArsonOggencProcess;
class ArsonBladeProcess;
class ArsonLameProcess;
class ArsonXmlWriter;
class QXmlAttributes;

class ArsonEncoderOpts
{
public:
	ArsonEncoderOpts (void);

	int bitrate (void) const { return m_bitrate; }
	int channels (void) const { return m_channels; }
	int vbr (void) const { return m_vbr; }
	bool is (uint mask) const { return (m_flags & mask) != 0; }
	uint flags (void) const { return m_flags; }
	const QString &manualSwitches (void) const { return m_switches; }
	int manualEncoder (void) const { return m_manual; }

	void setBitrate (int bitrate) { m_bitrate = bitrate; }
	void setChannels (int chans) { m_channels = chans; }
	void setVbr (int vbr) { m_vbr = vbr; }
	void set (uint mask) { m_flags |= mask; }
	void clear (uint mask) { m_flags &= ~mask; }
	void setFlags (uint flags) { m_flags = flags; }
	void setManual (int enc, const QString &switches = QString::null);

	enum {
		chanStereo = 0,
		chanMono,
		chanLeftMono,
		chanRghtMono,
		chanJoint,
		_chan_max
	};

	enum {
		flagCRC = 0x01,
		flagCopyright = 0x02,
		flagCopy = 0x04,
	};

	enum {
		encoderNone = -1,
		encoderAny = 0,
		encoderLAME,
		encoderBlade,
		encoderOgg,
#ifdef FLAC
		encoderFLAC,
#endif
		_encoder_max,
	};

	void load (const QXmlAttributes &attr);
	void save (const QString &name, ArsonXmlWriter &writer);

	void applyTo (ArsonLameProcess &proc);
	void applyTo (ArsonBladeProcess &proc);
	void applyTo (ArsonOggencProcess &proc);
	void applyTo (ArsonEncoderProcess &proc);
#ifdef FLAC
	void applyTo (ArsonFlacEncoderProcess &proc);
#endif

	static const QStringList &warnings (void) { return m_warnings; }
	static void clearWarnings (void) { m_warnings.clear(); }

	const static QString encoder_names[_encoder_max];

private:
	static void addWarning (const QString &warning)
	{ m_warnings.append(warning); }

	bool useManual (int enc, ArsonEncoderProcess &proc) const;

	int m_bitrate;		//	Default 128kbps
	int m_channels;		//	Default stereo
	int m_vbr;			//	Default -1
	int m_manual;
	QString m_switches;
	uint m_flags;

	static QStringList m_warnings;
};

/*========================================================*/

class ArsonEncoderPresets
{
	typedef QMap<QString,ArsonEncoderOpts> ENCODERMAP;
	
public:
	ArsonEncoderPresets (void);

	void addPreset (const QString &name, const ArsonEncoderOpts &opts);
	void delPreset (const QString &name);
	void save (void);

	QStringList names (void) const;
	
	const ArsonEncoderOpts *opts (const QString &name) const;
	ArsonEncoderOpts *opts (const QString &name);

	int count (void) const { return m_presets.count(); }
	
private:
	ENCODERMAP m_presets;
};

/*========================================================*/
#endif	/*	__ENCODEROPTS_H__	*/
