/**********************************************************************
 ** btree_a - a btree class for areas.  Wanted to make this a generic
 **           btree class but couldn't get it to work with linux C++. 
 **           someday may combine this with btree_l when I figure out
 **           how linux C++ generic classes are done.
 ** 
 ** Last reviewed:
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef BTREE_A_C
#define BTREE_A_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "location.h"
#include "btree_a.h"
#include "newfuncts.h"

btree_a::btree_a()
{
   root = NULL; 
   root_list = NULL;
   last_added = NULL; 
   next_in_list = NULL;
}

int btree_a::add(char *key,Area_Dbase *data)
{   b_node_a *newnode;
    b_node_a *tmpnode;
    int      result;

    if ((key == NULL) || (data == NULL))
       return -1;

    // First lets create a new b_node record for it, and copy the key in it,
    // so we will be able to identify it later. The pointers in the newnode
    // record are cleared by the constructor of b_node.

    newnode = new_b_node_a();
    newnode->key = NULL;
    newnode->data = NULL;
    newnode->left = NULL;
    newnode->right = NULL;
    newnode->next = NULL;

    newnode->key = new char[strlen(key) + 1];
    strcpy(newnode->key,key);
    newnode->data = data;
    newnode->next = NULL;

    // Then we try adding it to the tree
    result = add_node(&root,newnode);

    // if the add to tree was successful, add it to the linked list
    if (result > 0)
    {
       if (root_list == NULL)
       {
          root_list = newnode;          
          return result;
       }

       /* first see if last added is not pointing to anything, if so, find
          the end of the list */
       if (last_added == NULL)
       {
          tmpnode = root_list;
          if (tmpnode != NULL)
	  {
             while (tmpnode->next != NULL)
	     {
                tmpnode = tmpnode->next;
             }
          }
       }
       else
          tmpnode = last_added;
    
       /* now tempnode is where we want it to be, add and reassign 
          last_node */ 
       if (tmpnode != NULL)
       {
          if (tmpnode->next != NULL)
             printf("error in btree_a!\n");
          tmpnode->next = newnode;
          last_added = newnode;
       }
    }
    return result;
}

void btree_a::show()
{   show(root,0);
}


int btree_a::add_node(b_node_a **baseptr,b_node_a *newnode)
{   int cmpres = 0;      // result of the string compare

    if (newnode == NULL)
       return -1;

    // First see if we hit rock bottom. If so, we can safely add our new
    // node here.
    if ((*baseptr) == NULL)  
    {  *baseptr = newnode;
       return 1;
    }
    else if ((cmpres = STRCASECMP((*baseptr)->key,newnode->key)) < 0)
    {   // Key is smaller than current one, so lets try inserting it on
        // the left branch.
        return add_node(&(*baseptr)->left,newnode);
    }
    else if (cmpres > 0)
    {   // The other option is that it is greater than the current key, then
        // we traverse down the right branch.
        return add_node(&(*baseptr)->right,newnode);
    }
    // Last but not least, it may already be in there, in which case the
    // cmpres will be zero, so lets tell the user he's being foolish.
    return 0;
}

/* Show shows a nice dump of the tree on the screen.
 */
void btree_a::show(b_node_a *baseptr, int level)
{   int i;

    if (baseptr != NULL)
    {  show(baseptr->left, level + 1);
       
       for (i = 0; i < level; i++)		// Indent to form a tree
         printf("   ");
       printf("%s\n",baseptr->key);

       show(baseptr->right, level + 1); 
    }
}


int btree_a::del(char *key)
{
   b_node_a *tmpnode;
   b_node_a *lastnode;

   tmpnode = root_list;
   if (STRCASECMP(tmpnode->key, key))
   {
      while (tmpnode->next != NULL)
      {
         lastnode = tmpnode;
         tmpnode = tmpnode->next;
         if (!STRCASECMP(tmpnode->key, key))
	 {
            lastnode->next = tmpnode->next;
            break;
         }
      }
   }
   return del_node(&root,key);
}


int btree_a::del_node(b_node_a **baseptr,char *key)
{  int res = 0;

   if ((*baseptr) == NULL)
      return 0;                                          // Not Found
   else if ((res = STRCASECMP((*baseptr)->key,key)) == 0)    // This node is the one
   {   b_node_a *tempptr, *delptr;

       // Ensure last_node pointer is kept up to date
       if (last_added == *baseptr)
          last_added = NULL;
 
       // Store current pointer  so we can free up memory when it is removed.
       delptr = (*baseptr);

       if ((*baseptr)->right == NULL)
           *baseptr = (*baseptr)->left;
       else 
       {   tempptr = (*baseptr)->left;
           *baseptr = (*baseptr)->right;
           (void)add_node(baseptr,tempptr);        // insert the remainder
       }

       if (delptr->data != NULL)
           delete_Area_Dbase(delptr->data);
       delete delptr->key;
       delete_b_node_a(delptr);
       return 1;     
   }
   else if (res < 0)
       return del_node(&(*baseptr)->left,key);
   else
       return del_node(&(*baseptr)->right,key);
   return 0;
}


Area_Dbase *btree_a::find(char *key)
{  return find_node(root,key);
}

 
bool btree_a::can_find(char *key)
{  if (find_node(root,key) == NULL)
      return false;
   return true;
}

Area_Dbase *btree_a::find_node(b_node_a *baseptr, char *key)
{   int cmpres;

    if (baseptr == NULL)
       return NULL;
    else if ((cmpres = STRCASECMP(baseptr->key,key)) == 0)
       return baseptr->data;
    else if (cmpres < 0)
       return find_node(baseptr->left,key);

    return find_node(baseptr->right,key);
}

int btree_a::reset_next_in_list(void)
{
   next_in_list = root_list;
   return 1;
}

Area_Dbase *btree_a::get_next_in_list(void)
{
   b_node_a *tmp_ptr;

   if (next_in_list == NULL)
      return NULL;

   tmp_ptr = next_in_list;
   next_in_list = next_in_list->next;
   return tmp_ptr->data;
}

int btree_a::free_node(b_node_a *the_node)
{
   b_node_a *tmp_node;

   if (the_node == NULL)
      return 1;

   if (the_node->next != NULL)
   {

      /* prevent any functions from searching while we delete */
      tmp_node = the_node->next;
      the_node->next = NULL;

      /* kill all below! */
      free_node(tmp_node);
   }
   delete_Area_Dbase(the_node->data);
   delete the_node->key;
   delete_b_node_a(the_node);
   return 1;
}

btree_a::~btree_a(void)
{
   free_node(root);
   root = NULL;
}

void btree_a::save_state()
{
  next_in_list_saved = next_in_list;
}

void btree_a::restore_state()
{
  next_in_list = next_in_list_saved;
}

#endif
