/**********************************************************************
 * cidr_net_list.h                                           March 2001
 * Horms                                             horms@vergenet.net
 *
 * aggregate
 * CIDR network aggregation and filtering
 * Copyright (C) 2001  Horms
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 *
 **********************************************************************/


#ifndef CIDR_NET_LIST_FRUB
#define CIDR_NET_LIST_FRUB

#define IP_STRING_LENGTH 16
#define IP_BITS 32


#include "cidr_net.h"
#include <unistd.h>


typedef struct cidr_net_list_elem_struct {
  struct cidr_net_list_elem_struct *next;
  struct cidr_net_list_elem_struct *prev;
  cidr_net_t *net;
} cidr_net_list_elem_t;

typedef struct {
  cidr_net_list_elem_t *first;
  cidr_net_list_elem_t *last;
  cidr_net_list_elem_t **recent;
  size_t norecent;
  size_t recent_offset;
} cidr_net_list_t;


/**********************************************************************
 * cidr_net_list_create
 * Create a new, empty list
 * pre: norecent: number of elements for recent list.
 *                If negative DEFAULT_NORECENT is used
 * post: list structure is alocated, and values initialised to NULL
 * return: pointer to list
 *         NULL on error
 **********************************************************************/

cidr_net_list_t *cidr_net_list_create(int norecent);


/**********************************************************************
 * cidr_net_list_destroy
 * Destroy a list and all the data contained in the list
 * pre: l: list
 * post: all elements of l are destroyed
 **********************************************************************/

void cidr_net_list_destroy(cidr_net_list_t *l);


/**********************************************************************
 * cidr_net_list_display
 * Print to stdout  all the elements in a list, in the order they are 
 * stored elemetns are printed as per cidr_net_display()
 * pre: l: list to display
 *      str: preallocated string to "render" network address to in
 *          dotted quad noation.
 *      minimim_prefix: Don't display networks with a prefix less than
 *                      this value. Valid values 0-32. 
 *                      -1 for no mimimium prefix.
 *      maximim_prefix: Don't display networks with a prefix greater than
 *                      this value. Valid values 0-32. 
 *                      -1 for no maximium prefix.
 *      flag: passed tp cidr_net_display
 * post: networks in list displayed to stdout
 **********************************************************************/

int cidr_net_list_display(
  cidr_net_list_t *l, 
  prefix_t minimum_prefix,
  prefix_t maximum_prefix,
  cidr_flag_t flag
);
 

/**********************************************************************
 * cidr_net_list_insert
 * Insert element into a list
 * pre: l: list to insert value into
 *      n: network to insert
 * post: network is inserted into the list in order
 *       NULL if l is NULL
 *       l, unchanged if n is null
 **********************************************************************/

cidr_net_list_t *cidr_net_list_insert(cidr_net_list_t *l, cidr_net_t *n);


/**********************************************************************
 * cidr_net_list_elem_remove
 * remove an element from a list
 * pre: l: list to remove elment from
 *      e: element to remove
 * post: element is removed from list
 *       If this was the last element of the list then l->recent is
 *       set to the previous (now last) elemet of the list.
 *       Otherwise l->recent is set to the next element in the list
 *       No change is made to the list if l or e are NULL
 * return: l
 *         NULL if l or e are NULL
 **********************************************************************/

cidr_net_list_t *cidr_net_list_elem_remove(
  cidr_net_list_t *l,
  cidr_net_list_elem_t *e
);


/**********************************************************************
 * cidr_net_list_insert_raw
 * Insert raw network and prefix information into a list
 * Puts this data into a cidr_net structure and then inserts
 * this structure into the list, as per cidr_net_list_insert
 * pre: l: list to add network to
 *      a: base address of network
 *      p: prefix of network
 **********************************************************************/

cidr_net_list_t *cidr_net_list_insert_raw(
  cidr_net_list_t *l, 
  ip_address_t a,
  prefix_t p
);


/**********************************************************************
 * cidr_net_list_elem_create
 * Create a new element, and seed it
 * pre: e: unalocated pointer to elelemt to create
 *      prev: previous element in list
 *      next: next element in list
 *      net: net to store
 * post: e is initialised and values are seeded
 * return: pointer to e
 *         NULL on error
 **********************************************************************/

cidr_net_list_elem_t *cidr_net_list_elem_create(
  cidr_net_list_elem_t *prev,
  cidr_net_list_elem_t *next,
  cidr_net_t *net
);


/**********************************************************************
 * cidr_net_list_elem_create_empty
 * Create a new element, and seed it with NULL values
 * pre: none
 * post: e is initialised and values are seeded with NULL
 * return: pointer to e
 *         NULL on error
 **********************************************************************/

cidr_net_list_elem_t *cidr_net_list_elem_create_empty(void);


/**********************************************************************
 * cidr_net_list_elem_assign
 * Assign values to a list element
 * pre: e: alocated pointer to elelemt to seed
 *      prev: previous element in list
 *      next: next element in list
 *      net: network to store in element
 * post: values are assigned to e
 * return: pointer to e
 *         NULL if e was NULL
 **********************************************************************/

cidr_net_list_elem_t *cidr_net_list_elem_assign(
  cidr_net_list_elem_t *e,
  cidr_net_list_elem_t *prev,
  cidr_net_list_elem_t *next,
  cidr_net_t *net
);


/**********************************************************************
 * cidr_net_list_elem_unassign
 * Assign NULL values to a list element.
 * As the name suggets this can be used to unasign values, possibly
 * so you can destroy the list element without freeing its contents.
 * pre: e: alocated pointer to elelemt to seed
 * post: values are assigned to e
 * return: pointer to e
 *         NULL if e was NULL
 **********************************************************************/

cidr_net_list_elem_t *cidr_net_list_elem_unassign(cidr_net_list_elem_t *e);


/**********************************************************************
 * cidr_net_list_elem_destroy
 * Assign values to a list element
 * pre: e: pointer to elelemt to destroy
 * post: e and values in e are dstroyed
 *       nothing if e is NULL
 **********************************************************************/

void cidr_net_list_elem_destroy(cidr_net_list_elem_t *e);


/**********************************************************************
 * cidr_net_list_aggregate
 * Aggregate networks - join adjacent small networks to form larger networks
 * Resulting list will contain as few networks as possible
 * pre: l: list of networks to aggregate
 * post: l contains as few networks as possible
 * return: l
 **********************************************************************/

cidr_net_list_t *cidr_net_list_aggregate(cidr_net_list_t *l);
    

/**********************************************************************
 * cidr_net_list_assimilate
 * If a network is contrained in another network remove it
 * pre: l: list of networks
 * post: networks inside of networks are removed
 **********************************************************************/

cidr_net_list_t *cidr_net_list_assimilate(cidr_net_list_t *l);


/**********************************************************************
 * cidr_net_list_range
 * Insert a range of addresses, broken down into cidr networks into
 * a list of cidr networks
 * pre: ip1: first ip in range
 *      ip2: last ip in range
 * post: list of cidr networks 
 *       NULL list on error
 **********************************************************************/

cidr_net_list_t *cidr_net_list_insert_range(
  ip_address_t ip1, 
  ip_address_t ip2, 
  cidr_net_list_t *l
);

#endif
