/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <vtimezone.h>
#include <toolbox.h>

#define IDSTRING "VTimeZone: "

extern Toolbox tb;

#define DEBUG_VTZ
//#define DEBUG_VTZ_1


/**
 *		The default constructor. creates an VTimeZone object based on a
 *	MetaObject.
 */

VTimeZone::VTimeZone(MetaObject *object, int year)
{
	if(year==-1)
	{
		QDateTime tmp;
		queryYear=tmp.currentDateTime().date().year();
	}
	else
		queryYear=year;
	if(object==NULL)
	{
	    printf(IDSTRING"ERROR: void object.\n");
	    errorCode=VTZ_ERROR_INVALID;
	    return;
	}
	//																	#warning TODO: set autodelete ...
//	namesList.setAutoDelete(true);
	errorCode=warningCode=0;
	setupTZID(object);
	setupLastModified(object);
	parse(object);
}

/**
 * Default destructor. does nothing => TO BE DELETED!
 */
VTimeZone::~VTimeZone()
{
	namesList.clear();
}

/**
 *	Method for setting up the TZID field
 */
void VTimeZone::setupTZID(MetaObject *object)
{
	QValueList<int> matches=object->query("TZID");
	if(matches.count()!=1)
	{
		errorCode=VTZ_ERROR_TZID;
		return;
	}
	tzId=object->getLine(matches[0]).getContent().simplifyWhiteSpace();
}

/**
 * Method for setting up LAST-MODIFIED
 */
void VTimeZone::setupLastModified(MetaObject *object)
{
	QValueList<int> matches=object->query("LAST-MODIFIED");
#ifdef DEBUG_VTZ_1
	printf(IDSTRING"LAST-MODIFIED count: %d\n",matches.count());
#endif
	if(matches.count()>1)
	{
		errorCode=VTZ_ERROR_LAST_MODIFIED;
		return;
	}
	if(matches.count()==0)
	{
		warningCode=VTZ_WARNING_NO_LAST_MODIFIED;
		QDateTime tmp;
		lastModified=QDateTime(tmp.currentDateTime());
		return;
	}
	Toolbox::TimeZone tz;
	lastModified=tb.getDateTime(object->getLine(matches[0]).getContent(), tz);
}

/**
 * Method for converting the error code into normal text
 */
QString VTimeZone::getErrorText()
{
	QString result;
	switch(errorCode)
	{
		case VTZ_OK:
			result="Ok.";
			break;
		case VTZ_ERROR_TZID:
			result="Multiple \'TZIDs\'.";
			break;
		case VTZ_ERROR_LAST_MODIFIED:
			result="Multiple \'LAST-MODIFIED\'.";
			break;
		case VTZ_ERROR_VTC_ERROR:
			result="TimeZone component has an error.";
			break;
		case VTZ_EXPIRED_COMPONENT:
			result="TimeZone component is expired.";
			break;
	}
	return result;
}


/**
 * Method for converting the warning code into text.
 */
QString VTimeZone::getWarningText()
{
	QString result;
	switch(warningCode)
	{
		case VTZ_OK:
			result="Ok.";
			break;
		case VTZ_WARNING_NO_LAST_MODIFIED:
			result="No \'LAST-MODIFIED\' field. Current date set.";
			break;
	}
	return result;
}


/**
 * Method for parsing the MetaObject and build the QDict and the list
 */
void VTimeZone::parse(MetaObject *object)
{
	// get all the slices of the timezone...
	for(int index=0; index<object->getChildrenNo();index ++)
	{
		// for each small object:
		#define o							((*object)[index])
#ifdef DEBUG_VTZ
		printf(IDSTRING"Object type: %s\n", (const char *)o->getName());
#endif
		VTimeZoneComponent vtc(o);
		if(vtc.getErrorCode()!=0)
		{
			errorCode=VTZ_ERROR_VTC_ERROR;
			return;
		}
		// set the year for valid query
		vtc.setCurrentStart(queryYear);
		if(vtc.isValid())
		{
#ifdef DEBUG_VTZ
			printf(IDSTRING"             Still valid...\n");
#endif
			TZData data;
			data=vtc.getCurrentStart();
			QString *UID= new QString(vtc.generateUID());
			QStringList names= vtc.getNames();
			list.insert(*UID,data.start, data.offsetGMT, vtc.isLastRequired());
			if(list.getError()==true)
			{
				printf(IDSTRING" Invalid date location/occurence\n");
				errorCode=VTZ_ERROR_VTC_ERROR;
				return;
			}
			for(int i=0;i<names.count();i++)
				namesList.insert(names[i],UID);
#ifdef DEBUG_VTZ
	#ifdef DEBUG_VID
			list.show();
	#endif
			printf(IDSTRING"             Error? %s\n",(list.getError())?"YES":"no");
			printf(IDSTRING"             Date: %s\n",
					(const char *)data.start.toString());
			printf(IDSTRING"             has to be last? %s\n",
					(vtc.isLastRequired())?"YES":"no");
#endif
		} // if vtc is valid
		else
		{
			printf(IDSTRING"             Expired...\n");
			errorCode=VTZ_EXPIRED_COMPONENT;
			return;
		}
	} //for
#ifdef DEBUG_VTZ
	printf(IDSTRING"tzList has %d elements\n", list.count());
	printf(IDSTRING"namesList has %d elements\n", namesList.count());
#endif
}

/**
 * Operator for retreiving the difference to ZULU based on a QDateTime
 *	reference. (in the reference year really).
 */
long VTimeZone::operator[](QDateTime reference)
{
	return list[reference];
}

/**
 * Operator for retreiving the ZULU difference based on a timezone name.
 * If one knows the date is a CET or EDT or whatever, one can retreive
 *	the offset based on that.
 */
long VTimeZone::operator[](QString name)
{
	if(namesList[name]==NULL) return -1;
	return list[*(namesList[name])];
}

/**
 * Method for checking if the timezone structure contains a given name as
 *	TZID or TZNAME
 */
bool VTimeZone::containsName(QString name, bool extended=false)
{
	if(namesList[name]==NULL)
	{
		if(extended && name==tzId)
			return true;
		return false;
	}
	return true;
}

QStringList VTimeZone::getTzNames()
{
	QStringList result;
	QDictIterator<QString> it(namesList);
	while ( it.current() ) {
		result.append(it.currentKey());
		++it;
	}
	return result;
}

void VTimeZone::show(void)
{
	printf(IDSTRING"---------- %p\n", this);
	printf("\tqueryYear:      %d\n", queryYear);
	printf("\terrorCode:      %d\n", errorCode);
	printf("\twarningCode:    %d\n", warningCode);
	list.show();
	QDictIterator<QString> it(namesList);
	printf("\tTZNAMEs: ");
	while ( it.current() ) {
		printf("%s ", (const char *)it.currentKey());
		++it;
	}
	printf("\n");
	printf("\tTZID: %s\n", (const char *)tzId);
	printf("\tLAST-MODIFIED: %s\n", (const char *)lastModified.toString());
}

main()
{
	MetaObject mo(
		"BEGIN:VTIMEZONE\n"
			"TZID:US-Eastern\n"
			"Begin:Standard\n"
				"DTSTART:19951026T020000\n"
				"TZOFFSETFROM:-0400\n"
				"TZOFFSETTO:-0500\n"
				"TZNAME:EST\n"
				"RDATE:19961026T020000\n"
			"End:Standard\n"
			"Begin:Daylight\n"
				"DTSTART:19950405T020000\n"
				"TZOFFSETFROM:-0500\n"
				"TZOFFSETTO:-0400\n"
				"RRULE:FREQ=YEARLY;BYDAY=1SU;BYMONTH=4\n"
				"TZNAME:EDT\n"
				"TZNAME:EDT1\n"
				"TZNAME:EDT2\n"
				"TZNAME:EDT3\n"
			"End:Daylight\n"
		"END:VTIMEZONE\n");
	VTimeZone vtz(&mo, 1996);
	printf("\nTimeZone Error   : \"%s\"\n", (const char *)vtz.getErrorText());
	printf("TimeZone Warning : \"%s\"\n", (const char *)vtz.getWarningText());
	printf("TimeZone name    : \"%s\"\n", (const char *)vtz.getTZID());
	printf("TimeZone modified: \"%s\"\n",
			(const char *)vtz.getLastModified().toString());
	printf("Timezone offset: %d\n",vtz["EDT"]);
	printf("Timezone offset: %d\n",vtz["EST"]);
	printf("Timezone offset: %d\n",vtz["EDTU"]);
	printf("Contains <EDT> %d\n",vtz.containsName("EDT"));
	printf("Contains <EDTU> %d\n",vtz.containsName("EDTU"));
	printf("\n\n\n\n");
	vtz.show();
}





