/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#ifndef HAS_METAOBJECT
#define HAS_METAOBJECT

#include <metaline.h>
#include <metaquality.h>
#include <qvaluelist.h>

class MetaObject;

typedef QList<MetaLine> MetaLineList;
typedef QList<MetaObject> MetaObjectList;
typedef QList<int> intList;

/*
*	MetaObject 
* @short Class for creating, modifying a generic object.
*/
class MetaObject
{
  public:
		/**
		* Default constructor. Initializes a void object.
		*/
	  MetaObject();
		/**
		* Normal constructor. Creates an object out of a string.
		* The object has to begin with "BEGIN:object_name" and end with 
		* "END:object_name".
		*	@param qs		input string containing the object
		*	@param dp		The parent (by default is NULL)
		*/
	  MetaObject(QString qs, MetaObject *dp=NULL);
		/**
		* Copy constructor. Initializes new object with the contents of the old one.
		* @param	mo	a reference to the source object.
		*/
	  MetaObject(MetaObject &mo);
		/**
		* = operator. Copies the content of the right hand parameter.
		* @param	mo	a reference to the source object.
		*/
	  MetaObject &operator=(MetaObject &mo);
		/**
		* Destructor. 
		*/
	  ~MetaObject();
		/**
		* This function is designed for internal use only (copy constructor etc.).
		* @return it returns a reference to the list containing all the object's 
		* lines.
		*/
	  MetaLineList &getLines() { return lines;}
		/**
		* This function is designed for internal use only (copy constructor etc.).
		* @return it returns a reference to the list containing all the object's 
		* childrens (for nested objects).
		*/
	  MetaObjectList &getChildren() { return children;}
		/**
		* This function is designed for internal use only (copy constructor etc.).
		* @return it returns a reference to the list containing the object's name.
		*/
	  QString &getObjectName() { return objectName;}
		/**
		* This method is used to identify a line by its index.
		* @param	idx	the index of the line we want.
		* @return the line matched with the index.
		*/
	  MetaLine &getLine(int idx);
		/**
		* This method is used to return the object's parent (if any).
		* @return the line object's parent (or NULL if none is specified with the
		* constructor).
		*/
	  MetaObject *getDirectParent() { return directParent;};
		/**
		* This operator is used to identify a line by its name.
		* @param	qs	the name of the line we want.
		* @return the line matched with the name or a void line (all the fields 
		* are void).
		*/
	  MetaLine &operator[](QString qs);
		/**
		* This operator is used to retreive the number of children.
		* @return the number of children.
		*/
		int getChildrenNo() { return children.count(); }
		/**
		* This operator is used to identify a nested object by its index in the
		* list. It is provided for having a discrimination between lines and nested
		* objects.
		* @param	index	the index of the object we want.
		* @return the object matched with the name or a void object (all the fields 
		* are void).
		*/
	  MetaObject *operator[](int index);
		
		/**
		* The query method is provided for checking a line for some properties. It
		* can be used to wrap a function for returning the right line when more
		* lines have the same name, but different properties
		* @param	name	line's name,
		* @param	includedQuality the qualities required to be included in the 
		*		line,
		* @param	excludedQuality	the qualities which are NOT to be included in the
		* 	line,
		* @return a list (QValueList) with all the lines numbers complying to the
		* given criteria.
		*/
		QValueList<int> query(QString name=QString::null, 
				QString includedQuality=QString::null, 
				QString excludedQuality=QString::null);
		/**
		* The query method is provided for checking a line for some properties. It
		* can be used to wrap a function for returning the right line when more
		* lines have the same name, but different properties
		* @param	name	line's name,
		* @param	includedQuality the qualities required to be included in the 
		*		line,
		* @param	excludedQuality	the qualities which are NOT to be included in the
		* 	line,
		* @param	includedContent the content which HAS to be contained by the
		* 	line's content,
		* @param	excludedContent the content which HAS to be NOT contained by the
		* 	line's content.
		* @param	results the list containing the indexes of the lines complying
		* with the tested criteria.
		* @return the number of lines matching the critera or -1 if an error has
		* occured
		*/
		int extendedQuery(QString name=QString::null, 
				QString includedQuality=QString::null, 
				QString excludedQuality=QString::null, 
				QString includedContent=QString::null, 
				QString excludedContent=QString::null, 
				intList *results=NULL);
		/**
		* This method is used for deleting a nested object along with all its
		* sons.
		* @param	idx	the object's index
		* NOTE: it is advisable to check if the object's idx before changing it.
		* If the object with the index does not exist, then no action is taken.
		*/
		void deleteNestedObject(int idx);
		/**
		* This method is used for adding (by appending) new qualities to a line.
		* @param	name	the line's name,
		* @param	quality	the line's new qualities.
		* NOTE: it is advisable to check if the line exists before changing it.
		* If the line does not exist, then no action is taken.
		*/
		void appendLineQuality(QString name, QString quality);
		/**
		* This method is used for adding (by inserting) new content text to a line.
		* @param	name	the line's name,
		* @param	content	the line's new content to be inserted.
		* @param	pos	the position where we want to insert the new text (-1 means 
		* that the text will be inserted at the end).
		* NOTE: it is advisable to check if the line exists before changing it.
		* If the line does not exist, then no action is taken.
		*/
		void insertLineContent(QString name="", QString content="",int pos=-1);
		/**
		* This method is used for adding (by replacing) new qualities to a line.
		* @param	name	the line's name,
		* @param	quality	the line's new qualities.
		* NOTE: it is advisable to check if the line exists before changing it. This
		* method is useful when changing the "TYPE=..." qualities. If the line does
		* not exist, then no action is taken.
		*/
		void replaceLineQuality(QString name="", QString quality="");
		/**
		* This method is used for adding (by replacing) new content text to a line.
		* @param	name	the line's name,
		* @param	content	the line's new content to be inserted.
		* NOTE: it is advisable to check if the line exists before changing it.
		* If the line does not exist, then no action is taken.
		*/
		void replaceLineContent(QString name="", QString content="");
		/**
		* This method is used for adding or replacing a line.
		* @param	lineno	the line's index number in the list.
		* @param	lineContent	the line's new content (including the name).
		*/
		void replaceLine(int lineno, QString lineContent);
		/**
		* This method is used for deleting a line.
		* @param	name	the line's name.
		* @return TRUE if it found a line to erase, FALSE otherwise.
		*/
		bool eraseLine(QString name="");
		/**
		* Use this method to append a line to the lines list
		* @param	line	the new line (full content!).
		*/
		void appendLine(QString line);
		/**
		* This method is used for generating back the text from the object. It
		* generates all the stand-alone lines first and afterwards the objects.
		* @return	a string containing the object.
		*/
		QString recompose();
		/**
		* This method is used for providing the object's name.
		* @return	a string containing the object's name.
		*/
		QString getName() { return objectName; }
		/**
		* This method is used for changing the object's name.
		* @param newName	a string containing the object's new name.
		*/
		void setName(QString newName) { objectName=newName; }
		
		
  private:
    // private data
		QString text;	// used by the recompose function
  	QString objectName;
	  MetaLineList lines;
	  MetaObjectList children;
    MetaObject *directParent;
    // private methods
		/**
		* This method is used for replacing "\r\n" with "\n".
		* @param source	the original folded source.
		*/
	  void replaceCRLF(QString &source);
		/**
		* This method is used for generating the object.
		* @param qs	the object's source (text).
		*/
	  void parse(QString &qs);
	  QString &getPrivateLine(QString &source);
};

#endif








