/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <kconfig.h>

#include <qfile.h>
#include <qfiledialog.h>
#include <qtextstream.h>
#include <qmessagebox.h>

#include <maillistview.h>
#include <mailer.h>
#include <mainwindow.h>
#include <clipboard.h>
#include <contentswidget.h>
#include <qstringlist.h>
#include <maillabel.h>
#include <vfolder.h>
#include <templateview.h>
#include <indexmailitem.h>
#include <brokeriface.h>
#include <clientrmi.h>
#include <pixtoolbox.h>
#include <messagepropertiesdialog.h>
#include <composerwindow.h>
#include <datahandlerimpl.h>

extern KConfig *GlobalConfig;

MailListView::MailListView(VFolder *folder, QWidget *parent, const char *name, int t) :
	EListView(parent, name),TemplateView(folder),DataHandlerImpl(),
m_isDragging(false)
{
	// save type
	type=t;
	
	// set listview properties
	setAllColumnsShowFocus(true);
	setShowSortIndicator(true);
	setSelectionMode(QListView::Extended);
  setHScrollBarMode(QScrollView::AlwaysOff);
	
	// add columns and menu options
	switch(type)
	{
		case mail:
			addColumn("", EListDims::Fixed, 8); // priority
			addColumn("", EListDims::Fixed, 20); // attachments
			addColumn("", EListDims::Fixed, 20); // message icon
			addColumn("From", EListDims::Percent, 25);
			addColumn("Subject", EListDims::Percent, 35);
			addColumn("Date", EListDims::Percent, 20);
			addColumn("Status", EListDims::Percent, 10);
			addColumn("", EListDims::Fixed, 15); // flag
			addColumn("Size", EListDims::Percent, 10);
			setSorting(5, false);
			break;
		case sent:
			addColumn("", EListDims::Fixed, 8); // priority
			addColumn("", EListDims::Fixed, 20); // attachments
			addColumn("", EListDims::Fixed, 20); // message icon
			addColumn("Recipient", EListDims::Percent, 25);
			addColumn("Subject", EListDims::Percent, 35);
			addColumn("Date", EListDims::Percent, 20);
			addColumn("Status", EListDims::Percent, 10);
			addColumn("Size", EListDims::Percent, 10);
			setSorting(5, false);
			break;
		case drafts:
			addColumn("", EListDims::Fixed, 8); // priority
			addColumn("", EListDims::Fixed, 20); // attachments
			addColumn("", EListDims::Fixed, 20); // message icon
			addColumn("Recipient", EListDims::Percent, 40);
			addColumn("Subject", EListDims::Percent, 50);
			addColumn("Size", EListDims::Percent, 10);
			break;
		case outgoing:
			addColumn("", EListDims::Fixed, 8); // priority
			addColumn("", EListDims::Fixed, 20); // attachments
			addColumn("", EListDims::Fixed, 20); // message icon
			addColumn("Recipient", EListDims::Percent, 30);
			addColumn("Subject", EListDims::Percent, 40);
			addColumn("Scheduling", EListDims::Percent, 20);
			addColumn("Size", EListDims::Percent, 10);
			break;
	}
	
	// scan for messages
	if(folder)
	{
		QList<VObject>& msgList=folder->objectList();
   	for(QListIterator<VObject> it(msgList); it.current(); ++it)
		{
		  if( folder->isSimpleObject(it.current()->name()) )
		  {
  			MailObject *obj=(MailObject *)(it.current());
  			IndexMailItem *mview=new IndexMailItem(this, obj);
  			obj->setView(mview);
		  }
		}
	}
	
	// make new popup menu
	listpopup=new QPopupMenu();
	
	listpopup->insertItem(BarIcon("open"), "Open", this, SLOT(slotOpenSelection()));
	listpopup->insertItem(BarIcon("reply"), "Reply", this, SLOT(reply()));
	listpopup->insertItem(BarIcon("replyall"), "Reply to all", this, SLOT(replyAll()));
	listpopup->insertItem(BarIcon("forward"), "Forward", this, SLOT(forward()));
	listpopup->insertSeparator();
	listpopup->insertItem(BarIcon("cut"), "Cut", this, SLOT(slotCutSelection()));
	listpopup->insertItem(BarIcon("copy"), "Copy", this, SLOT(slotCopySelection()));
	listpopup->insertItem(BarIcon("paste"), "Paste", this, SLOT(slotPaste()));
	listpopup->insertItem(BarIcon("delete"), "Delete", this, SLOT(slotDeleteSelection()));
	listpopup->insertSeparator();
	listpopup->insertItem(BarIcon("save"), "Save", this, SLOT(slotSaveSelection()));
	listpopup->insertSeparator();
	listpopup->insertItem(BarIcon("flag"), "Toggle flag", this, SLOT(toggleFlag()));
	listpopup->insertItem(BarIcon("mark"), "Mark as unread", this, SLOT(markUnread()));
	listpopup->insertSeparator();
	listpopup->insertItem("Properties", this, SLOT(slotEditProperties()));
	
	connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)), this, SLOT(showPopup(QListViewItem *, const QPoint &, int)));
			
	// connect stuff
  // connect((QObject *)header(), SIGNAL(sizeChange(int, int, int)), (QObject *)this, SLOT(triggerUpdate()));
}

MailListView::~MailListView()
{
}

void MailListView::showPopup(QListViewItem *, const QPoint &pos, int)
{
	listpopup->popup(pos);
}
	
QList<DataWidget> MailListView::getSelection()
{
	QList<DataWidget> selection;
	IndexMailItem *item=(IndexMailItem *)firstChild();
	
	if(!item) printf("maillistview: cannot find the first child\n");
	
	while(item)
	{
		if(item->isSelected()) selection.append(item);
		item=(IndexMailItem *)item->itemBelow();
	}
	
	return selection;
}

bool MailListView::isSelectionMultiple()
{
	int selCount=0;
	IndexMailItem *item=(IndexMailItem *)firstChild();
	while(item && selCount<2)
	{
		if(item->isSelected()) selCount++;
		item=(IndexMailItem *)item->itemBelow();
	}
	
	if(selCount==2) return true;
	return false;
}

bool MailListView::isSelectionEmpty()
{
	IndexMailItem *item=(IndexMailItem *)firstChild();
	while(item)
	{
		if(item->isSelected()) return false;
		item=(IndexMailItem *)item->itemBelow();
	}
	
	return true;
}

IndexMailItem *MailListView::firstSelected()
{
	IndexMailItem *item=(IndexMailItem *)firstChild();
	while(item)
	{
		if(item->isSelected()) return item;
		item=(IndexMailItem *)item->itemBelow();
	}
	
	return 0;
}

void MailListView::deleteSelection()
{
	GlobalConfig->setGroup("Misc mail");
	if(GlobalConfig->readEntry("Permanent delete")=="Yes" || path()=="/Magellan/Mail/Deleted Messages")
	{
		DataHandlerImpl::deleteSelection();
	}
	else
	{
		// move messages to Deleted Mail
		QList<DataWidget> selection=getSelection();
		int i;
		
		bool multiDelete=selection.count()>1;
		if(multiDelete) ClientRMI::ref()->setAutoExpunge(false);
		for(i=0;i<(int)selection.count()-1;i++)
			BrokerInterface::thisInstance()->move(((IndexMailItem *)selection.at(i))->path(), "/Magellan/Mail/Deleted Messages");
		
		if(multiDelete) ClientRMI::ref()->setAutoExpunge(true);
		if( selection.count() )
		  BrokerInterface::thisInstance()->move(((IndexMailItem *)selection.at(i))->path(), "/Magellan/Mail/Deleted Messages");
	}
}

QStringList MailListView::selectedPaths()
{
	QStringList pathList;
	QList<DataWidget> selection=getSelection();
	
	for (unsigned int i = 0; i < selection.count(); i++)
	{
		pathList.append(static_cast<IndexMailItem *>(selection.at(i))->path());
	}
	
	return pathList;
}
	
void MailListView::selectAll()
{
	QListView::selectAll(true);
}

void MailListView::selectNone()
{
	QListView::selectAll(false);
}

void MailListView::markUnread()
{
	QList<DataWidget> selection=getSelection();
	for (unsigned int i=0; i < selection.count(); i++)
	{
		QString address=((IndexMailItem *)selection.at(i))->path();

		MailObject *obj=static_cast<MailObject *>(ClientVFS::thisInstance()->_fact_ref(address, "mail"));
		if (obj && obj->status()=="Read")
		{
			obj->sync();
			obj->setStatus("New");
			obj->save();
			obj->updateViews();
		}
	}
}

void MailListView::markAllRead()
{
	for (unsigned int i = 0; i < childViews.count(); i++)
	{
		QString address=static_cast<IndexMailItem *>(childViews.at(i))->path();
		
		MailObject *obj=static_cast<MailObject *>(ClientVFS::thisInstance()->_fact_ref(address, "mail"));
		if (obj && obj->status()=="New")
		{
			obj->sync();
			obj->setStatus("Read");
			obj->save();
			obj->updateViews();
		}
	}
}

void MailListView::markAllUnread()
{
	for(unsigned int i=0; i < childViews.count(); i++)
	{
		QString address=((IndexMailItem *)childViews.at(i))->path();
		
		MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(address, "mail");
		if(obj && obj->status()=="Read")
		{
			obj->sync();
			obj->setStatus("New");
			obj->save();
			obj->updateViews();
		}
	}
}

void MailListView::markRead()
{
	QList<DataWidget> selection=getSelection();
	for(unsigned int i=0; i < selection.count(); i++)
	{
		QString address=((IndexMailItem *)selection.at(i))->path();
		
		MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(address, "mail");
		if(obj && obj->status()=="New")
		{
			obj->sync();
			obj->setStatus("Read");
			obj->save();
			obj->updateViews();
		}
	}
}

void MailListView::toggleFlag()
{
	QList<DataWidget> selection=getSelection();
	for (unsigned int i = 0; i < selection.count(); i++)
	{
		QString address=((IndexMailItem *)selection.at(i))->path();
		
		MailObject *obj=(MailObject *)ClientVFS::thisInstance()->_fact_ref(address, "mail");
		if(obj)
		{
			obj->sync();
			obj->setFlag(!obj->flag());
			obj->save();
			obj->updateViews();
		}
	}
}

void MailListView::reply()
{
	IndexMailItem *item=firstSelected();
	if (item)
	{
		ComposerWindow::openComposer(item->path(), ComposerWindow::Reply);
	}
	else
	{
		printf("maillistview: no item selected\n");
	}
}

void MailListView::replyAll()
{
	IndexMailItem *item=firstSelected();
	if(item) ComposerWindow::openComposer(item->path(), ComposerWindow::ReplyAll);
}

void MailListView::forward()
{
	QStringList pathList=selectedPaths();
		
	if(pathList.count()>1) ComposerWindow::openComposer(pathList, ComposerWindow::Forward);
	else ComposerWindow::openComposer(pathList[0], ComposerWindow::Forward);
}

void MailListView::forwardQuoted()
{
	IndexMailItem *selected=firstSelected();
	if(selected) ComposerWindow::openComposer(selected->path(), ComposerWindow::ForwardQuoted);
}

void MailListView::forwardInline()
{
	IndexMailItem *selected=firstSelected();
	if(selected) ComposerWindow::openComposer(selected->path(), ComposerWindow::ForwardInline);
}

void MailListView::forwardAsAttachment()
{
	QStringList pathList=selectedPaths();
		
	if(pathList.count()>1) ComposerWindow::openComposer(pathList, ComposerWindow::Forward);
	else ComposerWindow::openComposer(pathList[0], ComposerWindow::ForwardAsAttachment);
}

void MailListView::editProperties()
{
	QStringList pathList=selectedPaths();
	
	if(pathList.count())
	{
		MessagePropertiesDialog *propertiesDialog=new MessagePropertiesDialog(pathList, MainWindow::ref());
		propertiesDialog->exec(); // TODO: Check the result maybe?
		delete propertiesDialog;
	}
}

void MailListView::openSelection()
{
	if(((VFolder *)getVItem())->viewType()=="outbox" || ((VFolder *)getVItem())->viewType()=="drafts")
	{
		IndexMailItem *selected=firstSelected();
		if (selected)
			ComposerWindow::openComposer(selected->path(), ComposerWindow::EditMessage);
	}
}

void MailListView::saveSelection()
{
	IndexMailItem *selected=firstSelected();
	if(selected) selected->save();
}

void MailListView::slotEditProperties()
{
	editProperties();
}

void MailListView::slotDeleteSelection()
{
	deleteSelection();
}

void MailListView::slotCutSelection()
{
	cutSelection();
}

void MailListView::slotCopySelection()
{
	copySelection();
}

void MailListView::slotPaste()
{
	paste();
}

void MailListView::slotOpenSelection()
{
	openSelection();
}

void MailListView::slotSaveSelection()
{
	saveSelection();
}

void MailListView::contentsMouseMoveEvent(QMouseEvent *e)
{
	if(!m_isDragging) {
		QListView::contentsMouseMoveEvent(e);
	} else {
		dragStart(viewport());
		m_isDragging = false;
	}
}

void MailListView::contentsMousePressEvent(QMouseEvent *e)
{
	if(e->button() == LeftButton)
		m_isDragging = true;

	QListView::contentsMousePressEvent(e);
}

void MailListView::contentsMouseReleaseEvent(QMouseEvent *e)
{
	if(e->button() == LeftButton)
		m_isDragging = false;

	QListView::contentsMouseReleaseEvent(e);
}

