/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */
#include <countriesclass.h>
#include <qmessagebox.h>

#define IDSTRING "CountriesClass: "

CountriesClass *CountriesClass::instCountries=0;

CountriesClass::CountriesClass(QString filename)
{
	load(filename);
}

void CountriesClass::load(QString name)
{
	QFile f(name);
	QString line;
	add(",");
	if ( f.open(IO_ReadOnly) )
	{
		QTextStream t( &f );
		while ( !t.eof() ) 
		{ 
			line=t.readLine();
			switch(((const char *)line)[0])
			{
				case ';':
#ifdef DEBUG_CC
					printf(IDSTRING"Comment...\n");
#endif
					break;
				case '*':
					line.remove(0,1);
					add(line, false);
					break;
				default:
					add(line);
			}
		}
		f.close();
	}
	else
	{
		QMessageBox::warning(0,"Loading countries...", QString("Could not find the file ")+name+
			".\nFailed loading countries (database is empty).", QMessageBox::Ignore, 0);
	}
}

void CountriesClass::save(QString name)
{
	QFile f(name);
	if ( f.open(IO_WriteOnly) )
	{
		QTextStream t( &f );
		QString s;
		for(int i=0;i<countries(false).count();i++)
			t << "*" << countries(false)[i] << "," << domains(false)[i] << endl;
		for(int i=0;i<countries(true).count();i++)
			t << countries(true)[i] << "," << domains(true)[i] << endl;
		f.close();
	}
	else
	{
		QMessageBox::warning(0,"Saving countries...", QString("Could not open the file ")+name+
			".\nFailed saving countries (old database).", QMessageBox::Ignore, 0);
	}
}


QString CountriesClass::country(QString dom, int extended)
{
	int idx;
	if(extended & CountriesClass::Mapped)
	{
		idx = domains(CountriesClass::Mapped).findIndex(dom.lower().stripWhiteSpace());
		if(idx!=-1)
			return countries(CountriesClass::Mapped)[idx];
	}
	if(extended & CountriesClass::Unique)
	{
		idx = domains(CountriesClass::Unique).findIndex(dom.lower().stripWhiteSpace());
		if(idx!=-1)
			return countries(CountriesClass::Unique)[idx];
	}
	if(extended & CountriesClass::Unmapped)
	{
		idx = domains(CountriesClass::Unmapped).findIndex(dom.lower().stripWhiteSpace());
		if(idx!=-1)
			return countries(CountriesClass::Unmapped)[idx];
	}
	return QString("");
}

QString CountriesClass::domain(QString co, int extended)
{
	QStringList localCountryList;
	int idx;
	// First look for the extended (unmapped) countries/domains
	if(extended && CountriesClass::Unique)
	{
		localCountryList = countries(CountriesClass::Unique).grep(co, false);
		if(localCountryList.count()!=0)
		{
			idx=countries(CountriesClass::Unique).findIndex(localCountryList[0]);
			if(idx!=-1)
				return domains(CountriesClass::Unique)[idx];
		}
	}
	if(extended && CountriesClass::Mapped)
	{
		localCountryList = countries(CountriesClass::Mapped).grep(co, false);
		if(localCountryList.count()!=0)
		{
			idx=countries(CountriesClass::Mapped).findIndex(localCountryList[0]);
			if(idx!=-1)
				return domains(CountriesClass::Mapped)[idx];
		}
	}
	if(extended && CountriesClass::Unmapped)
	{
		localCountryList = countries(CountriesClass::Unmapped).grep(co, false);
		if(localCountryList.count()!=0)
		{
			idx=countries(CountriesClass::Unmapped).findIndex(localCountryList[0]);
			if(idx!=-1)
				return domains(CountriesClass::Unmapped)[idx];
		}
	}
	return QString("");
}

bool CountriesClass::isCountry(QString co, int extended)
{
	bool result=false;
	if(extended & CountriesClass::Unique)
	{
		result |= countries(CountriesClass::Unique).grep(co, false).count() != 0;
		if(result)
			return true;
	}
	if(extended & CountriesClass::Mapped)
	{
		result |= countries(CountriesClass::Mapped).grep(co, false).count() != 0;
		if(result)
			return true;
	}
	if(extended & CountriesClass::Unmapped)
		result |= countries(CountriesClass::Unmapped).grep(co, false).count() != 0;
	return result;
}

bool CountriesClass::isDomain(QString dom, int extended)
{
	bool result=false;
	if(extended & CountriesClass::Unique)
	{
		result |= domains(CountriesClass::Unique).grep(dom, false).count() != 0;
		if(result)
			return true;
	}
	if(extended & CountriesClass::Mapped)
	{
		result |= domains(CountriesClass::Mapped).grep(dom, false).count() != 0;
		if(result)
			return true;
	}
	if(extended & CountriesClass::Unmapped)
		result |= domains(CountriesClass::Unmapped).grep(dom, false).count() != 0;
	return result;
}

QStringList &CountriesClass::countries(int mapped)
{
	switch(mapped)
	{
		case Mapped:
			return _mappedCountries;
		case Unmapped:
			return _unmappedCountries;
		case Unique:
		default:
			return _uniqueCountries;
	}
}

QStringList &CountriesClass::domains(int mapped)
{
	switch(mapped)
	{
		case Mapped:
			return _mappedDomains;
		case Unmapped:
			return _unmappedDomains;
		case Unique:
		default:
			return _uniqueDomains;
	}
}

void CountriesClass::add(QString line, bool mapped)
{
	if(line.stripWhiteSpace().isEmpty())
		return;
	QStringList items=QStringList::split(',', line);
	if(mapped)
	{
		_mappedDomains.append(items[1].lower());
		_mappedCountries.append(items[0]);
		if(_uniqueCountries.findIndex(items[0])==-1)
		{
			_uniqueDomains.append(items[1].lower());
			_uniqueCountries.append(items[0]);
		}
	}
	else
	{
		_unmappedDomains.append(items[1].lower());
		_unmappedCountries.append(items[0]);
	}
}

QStringList CountriesClass::allCountries()
{
	return _unmappedCountries+_mappedCountries;
}

QStringList CountriesClass::allDomains()
{
	return _unmappedDomains+_mappedDomains;
}

CountriesClass *CountriesClass::thisInstance(QString fname)
{
	if(!instCountries)
		instCountries = new CountriesClass(fname);
	return instCountries;
}
