/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <ebutton.h>
#include <qcolor.h>
#include <qfont.h>
#include <qframe.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qimage.h>
#include <qevent.h>
#include <stdio.h>
#include <qapplication.h>
#include <kiconeffect.h>
#include <kimageeffect.h>

/****** You have to define one of those below to build the "disabled" pixmap ******/
//#define PIXMAP_DITHERED
//#define PIXMAP_COPY
#define PIXMAP_FX

#ifdef PIXMAP_FX
	#define DarkColor		QColor(70,70,70)
	#define LightColor	QColor(230,230,230)
#endif

EButton::EButton(QWidget *parent, const char *name, int index) : QFrame(parent, name)
{
	set_defs();
	if(index!=-1) setIndex(index);

	if (parent->inherits("QToolBar"))
	  setToolBarStyle();
  else
  	set_dims();
}

EButton::EButton(const QPixmap &npix, QWidget *parent, const char *name, int index) : QFrame(parent, name)
{
	QPixmap tpix;
	set_defs();
	if(index!=-1) setIndex(index);
	normal_pix=focus_pix=clicked_pix=npix;
#ifdef PIXMAP_DITERED
	if(!npix.isNull())
	{
		tpix.convertFromImage(npix.convertToImage(), MonoOnly | ThresholdDither);
//		tpix.convertFromImage(npix.convertToImage(), MonoOnly | DiffuseDither);
		disabled_pix.convertFromImage(tpix.convertToImage(), ColorOnly);
		if(npix.mask())
			disabled_pix.setMask(*npix.mask());
	}
#endif
#ifdef PIXMAP_COPY
	disabled_pix=npix
#endif
#ifdef PIXMAP_FX
	QImage img=npix.convertToImage();
//	KIconEffect::toGray(img, 1.0);
	disabled_pix.convertFromImage(KImageEffect::flatten(img, DarkColor, LightColor));
	if(npix.mask())
		disabled_pix.setMask(*npix.mask());
#endif
	hasPixmap=true;

	if (parent->inherits("QToolBar"))
	  setToolBarStyle();
  else
  	set_dims();
}

EButton::EButton(const QString &text, QWidget *parent, const char *name, int index) : QFrame(parent, name)
{
	set_defs();
	if(index!=-1) setIndex(index);
	label=text;
	
	if (parent->inherits("QToolBar"))
	  setToolBarStyle();
  else
  	set_dims();
}

EButton::~EButton()
{
}

void EButton::set_defs()
{
	buttonIndex=-1;
	label="";
	tcolor=QColor("black");
	tfont=QFont("helvetica", 12, QFont::Normal);
	spacing=5;
	cmargin=4;
	nstyle=QFrame::NoFrame;
	fstyle=QFrame::WinPanel|QFrame::Raised;
	cstyle=QFrame::WinPanel|QFrame::Sunken;
	disabled=is_pressed=mouse_on=FALSE;
	hasText=hasPixmap=FALSE;
  nwidth=fwidth=cwidth=lineWidth();
  ncolor=fcolor=ccolor=backgroundColor();
  buddy=0;
	fixedHeight=false;
	useDefaultStyle=true;
}	

void EButton::setContentMargin(int cm)
{
	cmargin=cm;
	set_dims();
}

void EButton::set_dims()
{
	QFontMetrics fm(tfont);
	int h=0, w=0;
	
	QPixmap pix;
	if(hasPixmap)
		if(disabled) pix=disabled_pix;
		else if(is_pressed) pix=clicked_pix;
		else if(mouse_on) pix=focus_pix;
		else pix=normal_pix;
	
	w=cmargin;
	if (hasPixmap) {
	  w+=pix.width();
	  if (hasText)
	    w+=spacing+fm.width(label);
	} else {
	  if (hasText)
	    w+=fm.width(label);
	}
	w+=cmargin;
	
	if(fixedHeight==false)
	{
		if(hasText && hasPixmap)
		{
			h=QMAX(fm.height(), pix.height());
		}
		else
			if(hasText)
			{
				h=fm.height();
			}
			else
				if(hasPixmap)
				{	
					h=pix.height();
				}
		h+=cmargin*2;
		setMinimumSize(w,h);
		resize(w,h);
  }
  else
  {
    setMinimumWidth(w);
    resize(w,height());
  }
	
  repaint();
}
	
bool EButton::isDisabled()
{
	return disabled;
}

void EButton::setDisabled(bool d)
{
	disabled=d;
  repaint();
}

void EButton::setNormalPixmap(const QPixmap &npix)
{
	QPixmap tpix;
	normal_pix=npix;
#ifdef PIXMAP_DITHERED
	if(!npix.isNull())
	{
//		tpix.convertFromImage(npix.convertToImage(), MonoOnly | ThresholdDither);
		tpix.convertFromImage(npix.convertToImage(), MonoOnly | DiffuseDither);
		disabled_pix.convertFromImage(tpix.convertToImage(), ColorOnly);
		if(npix.mask())
			disabled_pix.setMask(*npix.mask());
	}
#endif
#ifdef PIXMAP_COPY
	disabled_pix=npix;
#endif
#ifdef PIXMAP_FX
	QImage img=npix.convertToImage();
//	KIconEffect::toGray(img, 1.0);
	disabled_pix.convertFromImage(KImageEffect::flatten(img, DarkColor, LightColor));
	if(npix.mask())
		disabled_pix.setMask(*npix.mask());
#endif
	hasPixmap=TRUE;
	set_dims();
}

void EButton::setFocusPixmap(const QPixmap &fpix)
{
	focus_pix=fpix;
	set_dims();
}

void EButton::setClickedPixmap(const QPixmap &cpix)
{
	clicked_pix=cpix;
	set_dims();
}

void EButton::setDisabledPixmap(const QPixmap &dpix)
{
	disabled_pix=dpix;
	set_dims();
}

void EButton::setText(const QString &text)
{
	label=text;
	hasText=TRUE;
	set_dims();
}

void EButton::setTextFont(const QFont &f)
{
	tfont=f;
	set_dims();
}

void EButton::setTextColor(const QColor &c)
{
	tcolor=c;
}

void EButton::setSpacing(int sp)
{
	spacing=sp;
	set_dims();
}

void EButton::setNormalFrameStyle(int ns)
{
	nstyle=ns;
  setFrameStyle(ns);
}

void EButton::setFocusFrameStyle(int fs)
{
	fstyle=fs;
}

void EButton::setClickedFrameStyle(int cs)
{
	cstyle=cs;
}

void EButton::drawContents(QPainter *p)
{
	QFontMetrics fm(tfont);
	p->setClipRect(contentsRect());
	
	QPixmap pix;
	if(hasPixmap)
		if(disabled) pix=disabled_pix;
		else if(is_pressed) pix=clicked_pix;
		else if(mouse_on) pix=focus_pix;
		else pix=normal_pix;
	
	int offset=cmargin;
	
	if(hasPixmap)
  {
    if(useDefaultStyle && !disabled)
			p->drawPixmap(offset, (height()-pix.height())/2, pix);
		else
	    qApp->style().drawItem(p, offset, (height()-pix.height())/2, pix.width(), pix.height(), 0, colorGroup(), disabled, &pix, QString(""));

    offset+=pix.width();
	  if (hasText)
	    offset+=spacing;
  }
	
	p->setPen(tcolor);
	p->setFont(tfont);
	if(hasText)
  {
    if(useDefaultStyle && !disabled)
			p->drawText(offset,0,fm.width(label), height(), AlignVCenter | AlignLeft, label);
//    	p->drawText(offset, (height() + fm.height()/2 )/2, label);
    else
			qApp->style().drawItem(p, offset, (height()-fm.height())/2+1 , fm.width(label),
        fm.height(), AlignVCenter | AlignLeft, colorGroup(), disabled, 0, label);
  }
}

void EButton::enterEvent(QEvent *e)
{
	mouse_on=TRUE;
  if(disabled) return;
#ifdef DEBUG_KFBUTTON
  printf("EButton: enter event, buddy:%p\n", buddy);
#endif
  if(buddy)
  {
#ifdef DEBUG_KFBUTTON
    printf("EButton: setting buddy focus..\n");
#endif
    buddy->showFocused();
  }
  showFocused();
}

void EButton::showFocused()
{
#ifdef DEBUG_KFBUTTON
  printf("EButton: showing focus..  this:%p\n", this);
#endif
	if(is_pressed) 
  {
    setFrameStyle(cstyle);
    setLineWidth(cwidth);
		if(!useDefaultStyle)
	    setBackgroundColor(ccolor);
  }
	else
  {
    setFrameStyle(fstyle);
    setLineWidth(fwidth);
		if(!useDefaultStyle)
	    setBackgroundColor(fcolor);
  }
	repaint();
}

void EButton::leaveEvent(QEvent *e)
{
  mouse_on=FALSE;
#ifdef DEBUG_KFBUTTON
  printf("EButton: leave event, buddy:%p\n", buddy);
#endif
  if(buddy)
  {
#ifdef DEBUG_KFBUTTON
    printf("EButton: setting buddy to normal..\n");
#endif
    buddy->showNormal();
  }
  showNormal();
}

void EButton::showNormal()
{
#ifdef DEGUG_KFBUTTON
  printf("EButton: showing normal..  this:%p\n", this);
#endif
	setFrameStyle(nstyle);
  setLineWidth(nwidth);
	if(!useDefaultStyle)
    setBackgroundColor(ncolor);
	repaint();
}

void EButton::mousePressEvent(QMouseEvent *e)
{
	is_pressed=TRUE;
  if(disabled) return;
	setFrameStyle(cstyle);
  setLineWidth(cwidth);
	if(!useDefaultStyle)
  	setBackgroundColor(ccolor);
	repaint();
	emit pressed();
}

void EButton::mouseReleaseEvent(QMouseEvent *e)
{
	is_pressed=FALSE;
  if(disabled) return;
	if(mouse_on) 
  {
    setFrameStyle(fstyle);
    setLineWidth(fwidth);
		if(!useDefaultStyle)
	    setBackgroundColor(fcolor);
  }
	else 
  {
    setFrameStyle(nstyle);
    setLineWidth(nwidth);
		if(!useDefaultStyle)
	    setBackgroundColor(ncolor);
  }
	repaint();
	emit released();
	if(e->x()>0&&e->x()<width()&&e->y()>0&&e->y()<height())
	{
		emit clicked();
#ifdef KFBUTTON_DEBUG
		printf("KFB: index = %d\n", buttonIndex);
#endif
		if(buttonIndex>=0) emit indexClicked(buttonIndex);
	}
}

void EButton::setToolBarStyle(int)
{
  setTextFont(QFont("helvetica", 10));
  setSpacing(3);
  setLineWidth(1);
  setMidLineWidth(0);
  setFocusFrameStyle(QFrame::Panel|QFrame::Raised);
  setClickedFrameStyle(QFrame::Panel|QFrame::Sunken);
  fixedHeight=true;
  set_dims();
}

void EButton::setArrowStyle()
{
  setSpacing(0);
  setLineWidth(1);
  setMidLineWidth(0);
  setContentMargin(1);
  setFocusFrameStyle(QFrame::Panel|QFrame::Raised);
  setClickedFrameStyle(QFrame::Panel|QFrame::Sunken);
  fixedHeight=true;
  set_dims();
  repaint();
}

void EButton::setCollectionStyle()
{
  setSpacing(0);
  setLineWidth(1);
  setMidLineWidth(1);
  setNormalFrameStyle(QFrame::NoFrame);
  setFocusFrameStyle(QFrame::Panel|QFrame::Raised);
  setClickedFrameStyle(QFrame::Panel|QFrame::Sunken);
  set_dims();
  repaint();
}

void EButton::setNormalFrameWidth(int w)
{
  nwidth=w;
  setLineWidth(w);
}

void EButton::setFocusFrameWidth(int w)
{
  fwidth=w;
}

void EButton::setClickedFrameWidth(int w)
{
  cwidth=w;
}

void EButton::setNormalBgColor(const QColor &c)
{
  ncolor=c;
  setBackgroundColor(c);
}

void EButton::setFocusBgColor(const QColor &c)
{
  fcolor=c;
}

void EButton::setClickedBgColor(const QColor &c)
{
  ccolor=c;
}

void EButton::setFocusBuddy(EButton *b)
{
  buddy=b;
#ifdef DEBUG_KFBUTTON
  printf("Focus buddy set to %p, this:%p\n", buddy, this);
#endif
}

void EButton::setIndex(int index)
{
	buttonIndex=index;
}

int EButton::getIndex()
{
	return buttonIndex;
}

QPixmap EButton::getPixmap(PixmapTypes pt)
{
	switch (pt)
	{
		case Normal:
			return normal_pix;
		case Focus:
			return focus_pix;
		case Clicked:
			return clicked_pix;
		case Disabled:
			return disabled_pix;
	}
	return QPixmap();
}
