/*
 * ratDbMessage.c --
 *
 *	This file contains code which implements dbase messages.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * Postilion software and its included text and images
 * Copyright (C) 1998 Nic Bernstein
 *
 * The full text of the legal notices is contained in the files called
 * COPYING and COPYRIGHT.TkRat, included with this distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ratFolder.h"

/*
 * The ClientData for each message entity
 */
typedef struct DbMessageInfo {
    char *buffer;
    MESSAGE *messagePtr;
} DbMessageInfo;

/*
 * The ClientData for each bodypart entity
 */
typedef struct DbBodyInfo {
    char *text;
} DbBodyInfo;

/*
 * The number of message entities created. This is used to create new
 * unique command names.
 */
static int numDbMessages = 0;

static RatGetHeadersProc Db_GetHeadersProc;
static RatGetEnvelopeProc Db_GetEnvelopeProc;
static RatInfoProc Db_GetInfoProc;
static RatCreateBodyProc Db_CreateBodyProc;
static RatFetchTextProc Db_FetchTextProc;
static RatEnvelopeProc Db_EnvelopeProc;
static RatSetIndexProc Db_SetIndexProc;
static RatMsgDeleteProc Db_MsgDeleteProc;
static RatMakeChildrenProc Db_MakeChildrenProc;
static RatFetchBodyProc Db_FetchBodyProc;
static RatBodyDeleteProc Db_BodyDeleteProc;


/*
 *----------------------------------------------------------------------
 *
 * RatDbMessagesInit --
 *
 *      Initializes the given MessageProcInfo entry for a dbase message
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The given MessageProcInfo is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatDbMessagesInit(MessageProcInfo *messageProcInfoPtr)
{
    messageProcInfoPtr->getHeadersProc = Db_GetHeadersProc;
    messageProcInfoPtr->getEnvelopeProc = Db_GetEnvelopeProc;
    messageProcInfoPtr->getInfoProc = Db_GetInfoProc;
    messageProcInfoPtr->createBodyProc = Db_CreateBodyProc;
    messageProcInfoPtr->fetchTextProc = Db_FetchTextProc;
    messageProcInfoPtr->envelopeProc = Db_EnvelopeProc;
    messageProcInfoPtr->setIndexProc = Db_SetIndexProc;
    messageProcInfoPtr->msgDeleteProc = Db_MsgDeleteProc;
    messageProcInfoPtr->makeChildrenProc = Db_MakeChildrenProc;
    messageProcInfoPtr->fetchBodyProc = Db_FetchBodyProc;
    messageProcInfoPtr->bodyDeleteProc = Db_BodyDeleteProc;
}


/*
 *----------------------------------------------------------------------
 *
 * RatDbMessageCreate --
 *
 *      Creates a dbase message entity
 *
 * Results:
 *	The name of the new message entity.
 *
 * Side effects:
 *	None.
 *	deleted.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatDbMessageCreate(Tcl_Interp *interp, RatFolderInfoPtr infoPtr, int index)
{
    DbMessageInfo *msgDbPtr=(DbMessageInfo*)ckalloc(sizeof(DbMessageInfo));
    MessageInfo *msgPtr=(MessageInfo*)ckalloc(sizeof(MessageInfo));
    char *name = (char*)ckalloc(16);

    msgPtr->folderInfoPtr = infoPtr;
    msgPtr->name = name;
    msgPtr->type = RAT_DBASE_MESSAGE;
    msgPtr->bodyInfoPtr = NULL;
    msgPtr->msgNo = index;
    msgPtr->fromMe = RAT_ISME_UNKOWN;
    msgPtr->toMe = RAT_ISME_UNKOWN;
    msgPtr->clientData = (ClientData)msgDbPtr;
    msgDbPtr->messagePtr = RatDbGetMessage(interp, index, &msgDbPtr->buffer);
    sprintf(name, "RatDbMsg%d", numDbMessages++);
    Tcl_CreateCommand(interp, name, RatMessageCmd, (ClientData) msgPtr, NULL);
    return name;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_GetHeadersProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_GetHeadersProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    return RatDbGetHeaders(interp, msgPtr->msgNo);
}


/*
 *----------------------------------------------------------------------
 *
 * Db_GetEnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_GetEnvelopeProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    static char buf[1024];
    RatDbEntry *entryPtr = RatDbGetEntry(msgPtr->msgNo);
    struct tm *tmPtr;
    time_t date;

    date = atoi(entryPtr->content[DATE]);
    tmPtr = gmtime(&date);
    sprintf(buf, "From %s %s %s %2d %02d:%02d GMT %04d\n",
	    entryPtr->content[FROM], dayName[tmPtr->tm_wday],
	    monthName[tmPtr->tm_mon], tmPtr->tm_mday, tmPtr->tm_hour,
	    tmPtr->tm_min, 1900+tmPtr->tm_year);
    return buf;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_CreateBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static BodyInfo*
Db_CreateBodyProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    DbMessageInfo *dbMsgPtr = (DbMessageInfo*)msgPtr->clientData;
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)malloc(sizeof(DbBodyInfo));
    msgPtr->bodyInfoPtr = CreateBodyInfo(msgPtr);

    msgPtr->bodyInfoPtr->bodyPtr = dbMsgPtr->messagePtr->body;
    msgPtr->bodyInfoPtr->clientData = (ClientData)dbBodyInfoPtr;
    dbBodyInfoPtr->text =
	dbMsgPtr->messagePtr->text.text.data+dbMsgPtr->messagePtr->text.offset;
    return msgPtr->bodyInfoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_FetchTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_FetchTextProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    return RatDbGetText(interp, msgPtr->msgNo);
}


/*
 *----------------------------------------------------------------------
 *
 * Db_EnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static ENVELOPE*
Db_EnvelopeProc(MessageInfo *msgPtr)
{
    return ((DbMessageInfo*)msgPtr->clientData)->messagePtr->env;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_SetIndexProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_SetIndexProc(MessageInfo *msgPtr, int index)
{
     return;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_MsgDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_MsgDeleteProc(MessageInfo *msgPtr)
{
    DbMessageInfo *dbMsgPtr = (DbMessageInfo*)msgPtr->clientData;
    mail_free_body(&dbMsgPtr->messagePtr->body);
    mail_free_envelope(&dbMsgPtr->messagePtr->env);
    ckfree(dbMsgPtr->messagePtr);
    ckfree(dbMsgPtr->buffer);
    ckfree(dbMsgPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Db_MakeChildrenProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_MakeChildrenProc(Tcl_Interp *interp, BodyInfo *bodyInfoPtr)
{
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)bodyInfoPtr->clientData;
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    BodyInfo *partInfoPtr, **partInfoPtrPtr;
    DbBodyInfo *dbPartInfoPtr;
    PART *partPtr;

    if (!bodyInfoPtr->firstbornPtr) {
	partInfoPtrPtr = &bodyInfoPtr->firstbornPtr;
	for (partPtr = bodyPtr->nested.part; partPtr;
		partPtr = partPtr->next) {
	    partInfoPtr = CreateBodyInfo(bodyInfoPtr->msgPtr);
	    dbPartInfoPtr = (DbBodyInfo*)malloc(sizeof(DbBodyInfo));
	    *partInfoPtrPtr = partInfoPtr;
	    partInfoPtr->bodyPtr = &partPtr->body;
	    partInfoPtrPtr = &partInfoPtr->nextPtr;
	    partInfoPtr->clientData = (ClientData)dbPartInfoPtr;
	    dbPartInfoPtr->text = dbBodyInfoPtr->text+
				  partPtr->body.contents.offset;
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Db_FetchBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_FetchBodyProc(BodyInfo *bodyInfoPtr, unsigned long *lengthPtr)
{
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)bodyInfoPtr->clientData;

    if (bodyInfoPtr->decodedTextPtr) {
	*lengthPtr = Tcl_DStringLength(bodyInfoPtr->decodedTextPtr);
	return Tcl_DStringValue(bodyInfoPtr->decodedTextPtr);
    }
    *lengthPtr = bodyInfoPtr->bodyPtr->contents.text.size;
    return dbBodyInfoPtr->text;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_BodyDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_BodyDeleteProc(BodyInfo *bodyInfoPtr)
{
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)bodyInfoPtr->clientData;
    ckfree(dbBodyInfoPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Db_GetInfoProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

char*
Db_GetInfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int notused)
{
    MessageInfo *msgPtr = (MessageInfo*)clientData;
    MESSAGE *messagePtr = ((DbMessageInfo*)msgPtr->clientData)->messagePtr;
    static Tcl_DString ds;
    static int initialized = 0;

    if (!initialized) {
	Tcl_DStringInit(&ds);
	initialized = 1;
    }

    switch (type) {
	case RAT_FOLDER_SUBJECT:	/* fallthrough */
	case RAT_FOLDER_NAME:		/* fallthrough */
	case RAT_FOLDER_MAIL_REAL:	/* fallthrough */
	case RAT_FOLDER_MAIL:		/* fallthrough */
	case RAT_FOLDER_NAME_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_SIZE:		/* fallthrough */
	case RAT_FOLDER_SIZE_F:		/* fallthrough */
	case RAT_FOLDER_TYPE:		/* fallthrough */
	case RAT_FOLDER_PARAMETERS:	/* fallthrough */
	case RAT_FOLDER_DATE_F:		/* fallthrough */
	case RAT_FOLDER_DATE_N:		/* fallthrough */
	case RAT_FOLDER_DATE_IMAP4:	/* fallthrough */
	case RAT_FOLDER_TO:		/* fallthrough */
	case RAT_FOLDER_FROM:		/* fallthrough */
	case RAT_FOLDER_STATUS:		/* fallthrough */
	case RAT_FOLDER_FLAGS:		/* fallthrough */
	case RAT_FOLDER_INDEX:
	    return Db_InfoProc(interp, (ClientData)msgPtr->folderInfoPtr, type,
		    -msgPtr->msgNo);
	case RAT_FOLDER_SENDER:		/* fallthrough */
	case RAT_FOLDER_CC:		/* fallthrough */
	case RAT_FOLDER_REPLY_TO:
	    Tcl_DStringSetLength(&ds, 2048);
	    if (type == RAT_FOLDER_SENDER) {
		rfc822_write_address(Tcl_DStringValue(&ds),
			messagePtr->env->sender);
	    } else if (type == RAT_FOLDER_CC) {
		rfc822_write_address(Tcl_DStringValue(&ds),
			messagePtr->env->cc);
	    } else {
		rfc822_write_address(Tcl_DStringValue(&ds),
			messagePtr->env->reply_to);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_END:
	    break;
    }
    return NULL;
}
