/**
   \file session.hh
   The class embedding a whole SCalc session.
   copyright (c) 2006 by Vincent Fourmond: 
  
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details (in the COPYING file).
  
*/

/** 
    \mainpage SCalc (simple|symbolic) calculator library

    SCalc is a library that provides elements for symbolic
    calculus and (relatively) fast evaluation. It is meant to be
    a simple calculus language, with support for variables and
    user-defined functions. Typical entries that the parser
    understands look like:
    
    \code
    1 + 1
    x = 2
    1 + x
    sin(x)
    f : x -> x**2
    f(3)
    f(x)
    \endcode
   

    Its main entry points are SCalc::Session, an object that holds a
    complete session, keeping track of defined variables, SCalc::ParserResult,
    the return value for an evaluation and its children, SCalc::Expression,
    SCalc::FuncDef and SCalc::SyntaxError.
    
*/

namespace SCalc {

  class ParserResult;
  class Expression;
  class FuncDef;

  /// The version string of the library
  std::string version();

      
  /** The very class that keeps tracks of one session, such as declared
      function and variables. This is your main entry point to SCalc.
      Typical example:

      \code
      SCalc::Session sess;
      SCalc::ParserResult r = sess.eval("1 + 1");
      cout << "1 + 1 = " << r->to_expression()->evaluate() << endl;
      \endcode

      The main function you'll be interested in is eval() to evaluate a
      string. It returns a ParserResult object which you can further
      interrogate and use.

      Each Session holds its own set of variables and function
      definitions. It is probably a very bad idea to try to share
      function definitions across several different Session objects.
      
  */
  /// A class representing a whole session.
  class Session {
  protected:
    /// The variables defined
    std::vector<std::string> variables;
    /// The cross reference to the variables
    std::map<std::string,int> variables_numbers;

    /// (default) values of variables, that is values
    /// that were assigned using a "a = 1.2" syntax.
    std::map<int, double> values;

    /// The currently defined functions. It works this way:
    /// The integer is the rank of the corresponding FuncDef
    /// pointe in the functions vector. This is done so that
    /// user defined functions can override to some extends the ones
    /// that were previously defined...
    std::map<std::string, int> functions_numbers;

    /// The function vector:
    std::vector<FuncDef *> functions;
  public:
    /// Creates a new Session object and register common
    /// mathematical functions.
    Session();

    /// Destroys a Session object and frees associated function
    /// definitions.
    ~Session();

    /**
       \brief Evaluates a string

       eval() returns a ParserResult object, which needs to be
       checked and cast into the right object, using
       ParserResult::is_func_def(), 
       ParserResult::is_expression() or ParserResult::is_syntax_error().

       You are responsible for freeing the object if you don't need it
       but you \b must check that is it not in use (such as function
       definitions) using ParserResult::can_delete():
       \code
       SCalc::ParserResult * r = sess.eval(some_string);
       // do something here
       if(r->can_delete())
         delete r;
       \endcode
    */
    ParserResult * eval(const char *);

    /**
       \brief Evaluates a string and frees the result if possible

       This function evaluates a string and frees the result if
       that is possible. Of course, as the function returns void,
       it is not currently possible to tell fi anything went fine,
       so you might as well not want to use it. It is used
       by the test suite to setup functions. 
    */
    void eval_and_free(const char *);

    /**
       \brief  Registers a variable name and returns its number
      
       Registers a new variable name, and returns its index in the
       array returned by varnames(). If the variable already exists,
       just return its number: no second registration is done.
    */
    int register_varname(const std::string &str);

    int register_varname(const char * str);

    /**
       \brief Returns the name of variable number i
    */
    const char * varname(int i);
    
    /**
      \brief Returns an array of variable names

      varnames() returns an array of variable names, in the
      position expected by SCalc::Expression::evaluate(). 
      
      \see register_varname(const char*)
    */
    std::vector<std::string> varnames() {
      std::vector<std::string> d = variables;
      return d;
    };
    /**
       \brief Returns the number of currently defined variables.
    */
    int nb_vars_defined() {return variables.size();};

    /// \name Variables with 'default values'
    /// 
    /// A set of functions dealing with 'default values'
    /// of variables, that is variables which were assigned
    /// using the "a = 2" syntax, or manually using set_var().

    /*@{*/

    /// Sets the (default) value of a variable
    ///
    /// \todo There should be a way to get the value of a variable.
    int set_var(int, double);

    int set_var(const char * var, double val) {
      return set_var(register_varname(var), val);
    };

    int set_var(const std::string & var, double val) {
      return set_var(register_varname(var), val);
    };

    /// Unset a given variable. Returns 0 if the variable wasn't
    /// previously set.
    int unset_var(int);

    /// Unset a variable of the given name. 
    int unset_var(std::string varname);
    int unset_var(const char * varname) { 
      return unset_var(std::string(varname));
    };

    /// Returns the number of variables that have a value
    int nb_vars_set() {return values.size();};
    
    /// Returns the set of the numbers of the variables which have
    /// a default value.
    std::set<int> vars_set();

    /// Fills the target array with the default values. Be careful:
    /// the argument has to have room for *at least* nb_vars_defined !
    void fill_default(double * );

    /// Tells if we have enough default variables to evaluate this
    /// expression without any other context.
    int evaluable(Expression * expr);

    /*@}*/


    /// Registers a function. Fails if a function is already
    /// registered under that name, or if the function is anonymous.
    /// All the registered functions are *deleted* when the Session is
    /// destroyed.
    int register_func_def(FuncDef *);

    /// Replaces a function definition. The former occupant of the
    /// place is simply deleted. We will need to ensure at some point
    /// that no trailing references are left. If the previous definition
    /// doesn't exist, simply calls register_func_def().
    int replace_func_def(FuncDef *);


    /// Get the corresponding function number (or -1 if not defined).
    int get_func(std::string);

    /// Get the corresponding function number (or -1 if not defined).
    int get_func(const char * name)
    { return get_func(std::string(name));};

    /// Get the function definition of the given name.
    FuncDef * get_func_def(std::string);

    /// Overloaded function for convenience.
    FuncDef * get_func_def(const char * name) 
    { return get_func_def(std::string(name));};

    /// Returns the number of arguments of a function
    int nb_args_func(std::string);
    /// Overloaded for convenience
    int nb_args_func(const char * name)
    { return nb_args_func(std::string(name));};

    /// The number of defined functions.
    int nb_funcs() { return functions.size();};

    /// The function names
    std::vector<std::string> func_names();

    /// Creates a new Expression representing a simple constant
    /// value. Typical use:
    /// \code
    /// // Multiply expr by two
    /// Expression * newExpression = Expression::mul(session.constant(2),expr);
    /// \endcode
    Expression * constant(double value);

  };

};
