/*
 * @BEGIN LICENSE
 *
 * Psi4: an open-source quantum chemistry software package
 *
 * Copyright (c) 2007-2016 The Psi4 Developers.
 *
 * The copyrights for code used from other parties are included in
 * the corresponding files.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * @END LICENSE
 */

#include <libmints/vector.h>
#include "VWN5_Cfunctional.h"
#include "utility.h"
#include <cmath>

using namespace psi;

namespace psi {

VWN5_CFunctional::VWN5_CFunctional()
{
    name_ = "VWN5_C";
    description_ = "    VWN5 Correlation Functional\n";
    citation_ = "    S.H. Vosko, L. Wilk, and M. Nusair, Can. J. Phys., 58, 1200-1211, 1980\n";
    alpha_ = 1.0;
    omega_ = 0.0;
    lrc_ = false;
    gga_ = false;
    meta_ = false;
    parameters_["c"] =   6.2035049089939986E-01;
    parameters_["d2fz0"] =   1.7099209341613653E+00;
    parameters_["EcP_1"] =   3.1090699999999999E-02;
    parameters_["EcP_2"] =  -1.0498000000000000E-01;
    parameters_["EcP_3"] =   3.7274400000000001E+00;
    parameters_["EcP_4"] =   1.2935200000000000E+01;
    parameters_["EcF_1"] =   1.5545349999999999E-02;
    parameters_["EcF_2"] =  -3.2500000000000001E-01;
    parameters_["EcF_3"] =   7.0604199999999997E+00;
    parameters_["EcF_4"] =   1.8057800000000000E+01;
    parameters_["Ac_1"] =  -1.6886863940389629E-02;
    parameters_["Ac_2"] =  -4.7584000000000003E-03;
    parameters_["Ac_3"] =   1.1310700000000000E+00;
    parameters_["Ac_4"] =   1.3004500000000000E+01;
    parameters_["two_13"] =   1.2599210498948732E+00;
}
VWN5_CFunctional::~VWN5_CFunctional()
{
}
void VWN5_CFunctional::compute_functional(const std::map<std::string,SharedVector>& in, const std::map<std::string,SharedVector>& out, int npoints, int deriv, double alpha)
{
    double c = parameters_["c"];
    double d2fz0 = parameters_["d2fz0"];
    double EcP_1 = parameters_["EcP_1"];
    double EcP_2 = parameters_["EcP_2"];
    double EcP_3 = parameters_["EcP_3"];
    double EcP_4 = parameters_["EcP_4"];
    double EcF_1 = parameters_["EcF_1"];
    double EcF_2 = parameters_["EcF_2"];
    double EcF_3 = parameters_["EcF_3"];
    double EcF_4 = parameters_["EcF_4"];
    double Ac_1 = parameters_["Ac_1"];
    double Ac_2 = parameters_["Ac_2"];
    double Ac_3 = parameters_["Ac_3"];
    double Ac_4 = parameters_["Ac_4"];
    double two_13 = parameters_["two_13"];

    // Overall scale factor
    double scale = alpha_ * alpha;

    // => Input variables <= //

    double* rho_ap = NULL;
    double* rho_bp = NULL;
    double* gamma_aap = NULL;
    double* gamma_abp = NULL;
    double* gamma_bbp = NULL;
    double* tau_ap = NULL;
    double* tau_bp = NULL;

    if (true) {
        rho_ap = in.find("RHO_A")->second->pointer();
        rho_bp = in.find("RHO_B")->second->pointer();
    }
    if (gga_) {  
        gamma_aap = in.find("GAMMA_AA")->second->pointer();
        gamma_abp = in.find("GAMMA_AB")->second->pointer();
        gamma_bbp = in.find("GAMMA_BB")->second->pointer();
    } 
    if (meta_)  {
        tau_ap = in.find("TAU_A")->second->pointer();
        tau_bp = in.find("TAU_B")->second->pointer();
    }

    // => Outut variables <= //

    double* v = NULL;

    double* v_rho_a = NULL;
    double* v_rho_b = NULL;
    double* v_gamma_aa = NULL;
    double* v_gamma_ab = NULL;
    double* v_gamma_bb = NULL;
    double* v_tau_a = NULL;
    double* v_tau_b = NULL;
     
    double* v_rho_a_rho_a = NULL;
    double* v_rho_a_rho_b = NULL;
    double* v_rho_b_rho_b = NULL;
    double* v_gamma_aa_gamma_aa = NULL;
    double* v_gamma_aa_gamma_ab = NULL;
    double* v_gamma_aa_gamma_bb = NULL;
    double* v_gamma_ab_gamma_ab = NULL;
    double* v_gamma_ab_gamma_bb = NULL;
    double* v_gamma_bb_gamma_bb = NULL;
    double* v_tau_a_tau_a = NULL;
    double* v_tau_a_tau_b = NULL;
    double* v_tau_b_tau_b = NULL;
    double* v_rho_a_gamma_aa = NULL;
    double* v_rho_a_gamma_ab = NULL;
    double* v_rho_a_gamma_bb = NULL;
    double* v_rho_b_gamma_aa = NULL;
    double* v_rho_b_gamma_ab = NULL;
    double* v_rho_b_gamma_bb = NULL;
    double* v_rho_a_tau_a = NULL;
    double* v_rho_a_tau_b = NULL;
    double* v_rho_b_tau_a = NULL;
    double* v_rho_b_tau_b = NULL;
    double* v_gamma_aa_tau_a = NULL;
    double* v_gamma_aa_tau_b = NULL;
    double* v_gamma_ab_tau_a = NULL;
    double* v_gamma_ab_tau_b = NULL;
    double* v_gamma_bb_tau_a = NULL;
    double* v_gamma_bb_tau_b = NULL;

    if (deriv >= 0) {
        v = out.find("V")->second->pointer();
    } 
    if (deriv >= 1) {
        if (true) {
            v_rho_a = out.find("V_RHO_A")->second->pointer();
            v_rho_b = out.find("V_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa = out.find("V_GAMMA_AA")->second->pointer();
            v_gamma_ab = out.find("V_GAMMA_AB")->second->pointer();
            v_gamma_bb = out.find("V_GAMMA_BB")->second->pointer();
        }
        if (meta_) {    
            v_tau_a = out.find("V_TAU_A")->second->pointer();
            v_tau_b = out.find("V_TAU_B")->second->pointer();
        }
    }
    if (deriv >= 2) {
        if (true) {
            v_rho_a_rho_a = out.find("V_RHO_A_RHO_A")->second->pointer();
            v_rho_a_rho_b = out.find("V_RHO_A_RHO_B")->second->pointer();
            v_rho_b_rho_b = out.find("V_RHO_B_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa_gamma_aa = out.find("V_GAMMA_AA_GAMMA_AA")->second->pointer();
            v_gamma_aa_gamma_ab = out.find("V_GAMMA_AA_GAMMA_AB")->second->pointer();
            v_gamma_aa_gamma_bb = out.find("V_GAMMA_AA_GAMMA_BB")->second->pointer();
            v_gamma_ab_gamma_ab = out.find("V_GAMMA_AB_GAMMA_AB")->second->pointer();
            v_gamma_ab_gamma_bb = out.find("V_GAMMA_AB_GAMMA_BB")->second->pointer();
            v_gamma_bb_gamma_bb = out.find("V_GAMMA_BB_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_tau_a_tau_a = out.find("V_TAU_A_TAU_A")->second->pointer();
            v_tau_a_tau_b = out.find("V_TAU_A_TAU_B")->second->pointer();
            v_tau_b_tau_b = out.find("V_TAU_B_TAU_B")->second->pointer();
        }
        if (gga_) {
            v_rho_a_gamma_aa = out.find("V_RHO_A_GAMMA_AA")->second->pointer();
            v_rho_a_gamma_ab = out.find("V_RHO_A_GAMMA_AB")->second->pointer();
            v_rho_a_gamma_bb = out.find("V_RHO_A_GAMMA_BB")->second->pointer();
            v_rho_b_gamma_aa = out.find("V_RHO_B_GAMMA_AA")->second->pointer();
            v_rho_b_gamma_ab = out.find("V_RHO_B_GAMMA_AB")->second->pointer();
            v_rho_b_gamma_bb = out.find("V_RHO_B_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_rho_a_tau_a = out.find("V_RHO_A_TAU_A")->second->pointer();
            v_rho_a_tau_b = out.find("V_RHO_A_TAU_B")->second->pointer();
            v_rho_b_tau_a = out.find("V_RHO_B_TAU_A")->second->pointer();
            v_rho_b_tau_b = out.find("V_RHO_B_TAU_B")->second->pointer();
        }
        if (gga_ && meta_) {
            v_gamma_aa_tau_a = out.find("V_GAMMA_AA_TAU_A")->second->pointer();
            v_gamma_aa_tau_b = out.find("V_GAMMA_AA_TAU_B")->second->pointer();
            v_gamma_ab_tau_a = out.find("V_GAMMA_AB_TAU_A")->second->pointer();
            v_gamma_ab_tau_b = out.find("V_GAMMA_AB_TAU_B")->second->pointer();
            v_gamma_bb_tau_a = out.find("V_GAMMA_BB_TAU_A")->second->pointer();
            v_gamma_bb_tau_b = out.find("V_GAMMA_BB_TAU_B")->second->pointer();
        }
    }

    // => Loop over points <= //

    for (int Q = 0; Q < npoints; Q++) {

        // Input variables 
        double rho_a;
        double rho_b;
        double gamma_aa;
        double gamma_ab;
        double gamma_bb;
        double tau_a;
        double tau_b;

        if (true) {
            rho_a = rho_ap[Q];
            rho_b = rho_bp[Q];
        }        
        if (gga_) {
            gamma_aa = gamma_aap[Q];
            gamma_ab = gamma_abp[Q];
            gamma_bb = gamma_bbp[Q];
        }        
        if (meta_) {
            tau_a = tau_ap[Q];
            tau_b = tau_bp[Q];
        }        

        // Definitions (asymptotics to prevent numerical problems)
        if (rho_a < lsda_cutoff_ && rho_b < lsda_cutoff_) {
            continue;
        } else if (rho_a < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t10421 = rho_a+rho_b;
                double t10422 = 1.0/pow(t10421,1.0/3.0);
                double t10423 = c*t10422;
                double t10424 = sqrt(t10423);
                double t10425 = EcP_4*4.0;
                double t10426 = EcP_3*EcP_3;
                double t10427 = t10425-t10426;
                double t10428 = EcP_2-t10424;
                double t10429 = EcP_3*t10424;
                double t10430 = EcP_4+t10423+t10429;
                double t10431 = 1.0/t10430;
                double t10432 = sqrt(t10427);
                double t10433 = t10424*2.0;
                double t10434 = EcP_3+t10433;
                double t10435 = 1.0/t10434;
                double t10436 = t10432*t10435;
                double t10437 = atan(t10436);
                double t10438 = 1.0/sqrt(t10427);
                double t10439 = EcF_4*4.0;
                double t10440 = EcF_3*EcF_3;
                double t10441 = -t10440+t10439;
                double t10442 = EcF_2-t10424;
                double t10443 = EcF_3*t10424;
                double t10444 = EcF_4+t10423+t10443;
                double t10445 = 1.0/t10444;
                double t10446 = sqrt(t10441);
                double t10447 = EcF_3+t10433;
                double t10448 = 1.0/t10447;
                double t10449 = t10446*t10448;
                double t10450 = atan(t10449);
                double t10451 = 1.0/sqrt(t10441);
                double t10452 = c*t10422*t10431;
                double t10453 = log(t10452);
                double t10454 = EcP_3*t10437*t10438*2.0;
                double t10455 = t10428*t10428;
                double t10456 = t10431*t10455;
                double t10457 = log(t10456);
                double t10458 = EcP_2*4.0;
                double t10459 = EcP_3*2.0;
                double t10460 = t10458+t10459;
                double t10461 = t10460*t10437*t10438;
                double t10462 = t10461+t10457;
                double t10463 = EcP_2*EcP_2;
                double t10464 = EcP_2*EcP_3;
                double t10465 = EcP_4+t10463+t10464;
                double t10466 = 1.0/t10465;
                double t10467 = t10453+t10454-EcP_2*EcP_3*t10462*t10466;
                double t10468 = EcP_1*t10467;
                v[Q] += scale * (t10421*(t10468-((pow(2.0,1.0/3.0)*2.0-2.0)*(t10468-EcF_1*(log(c*t10422*t10445)+EcF_3*t10450*t10451*2.0-(EcF_2*EcF_3*(log((t10442*t10442)*t10445)+t10450*t10451*(EcF_2*4.0+EcF_3*2.0)))/(EcF_4+EcF_2*EcF_2+EcF_2*EcF_3))))/(two_13*2.0-2.0)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t10470 = rho_a+rho_b;
                double t10471 = 1.0/pow(t10470,1.0/3.0);
                double t10472 = c*t10471;
                double t10473 = sqrt(t10472);
                double t10474 = EcP_4*4.0;
                double t10475 = EcP_3*EcP_3;
                double t10476 = t10474-t10475;
                double t10477 = EcP_2-t10473;
                double t10478 = EcP_3*t10473;
                double t10479 = EcP_4+t10472+t10478;
                double t10480 = 1.0/t10479;
                double t10481 = sqrt(t10476);
                double t10482 = t10473*2.0;
                double t10483 = EcP_3+t10482;
                double t10484 = 1.0/t10483;
                double t10485 = t10481*t10484;
                double t10486 = atan(t10485);
                double t10487 = 1.0/sqrt(t10476);
                double t10488 = 1.0/pow(t10470,4.0/3.0);
                double t10489 = c*t10488*(1.0/3.0);
                double t10490 = 1.0/sqrt(t10472);
                double t10491 = EcP_3*c*t10490*t10488*(1.0/6.0);
                double t10492 = t10491+t10489;
                double t10493 = t10477*t10477;
                double t10494 = 1.0/(t10479*t10479);
                double t10495 = EcP_2*4.0;
                double t10496 = EcP_3*2.0;
                double t10497 = t10495+t10496;
                double t10498 = 1.0/(t10483*t10483);
                double t10499 = EcP_2*EcP_2;
                double t10500 = EcP_2*EcP_3;
                double t10501 = EcP_4+t10500+t10499;
                double t10502 = 1.0/t10501;
                double t10503 = t10476*t10498;
                double t10504 = t10503+1.0;
                double t10505 = 1.0/t10504;
                double t10506 = 1.0/c;
                double t10507 = EcF_3*t10473;
                double t10508 = EcF_4+t10507+t10472;
                double t10509 = pow(t10470,1.0/3.0);
                double t10510 = EcF_3*c*t10490*t10488*(1.0/6.0);
                double t10511 = t10510+t10489;
                double t10512 = EcF_2-t10473;
                double t10513 = 1.0/(t10508*t10508);
                double t10514 = 1.0/t10508;
                double t10515 = EcF_3+t10482;
                double t10516 = 1.0/(t10515*t10515);
                double t10517 = EcF_4*4.0;
                double t10518 = EcF_3*EcF_3;
                double t10519 = t10517-t10518;
                double t10520 = t10516*t10519;
                double t10521 = t10520+1.0;
                double t10522 = 1.0/t10521;
                double t10523 = c*t10480*t10488*(1.0/3.0);
                double t10524 = t10523-c*t10471*t10492*t10494;
                double t10525 = t10506*t10524*t10509*t10479;
                double t10526 = t10492*t10493*t10494;
                double t10527 = c*t10480*t10490*t10477*t10488*(1.0/3.0);
                double t10528 = t10526+t10527;
                double t10529 = 1.0/(t10477*t10477);
                double t10530 = t10528*t10529*t10479;
                double t10531 = c*t10505*t10490*t10488*t10497*t10498*(1.0/3.0);
                double t10532 = t10530+t10531;
                double t10533 = EcP_2*EcP_3*t10502*t10532;
                double t10534 = t10533+t10525-EcP_3*c*t10505*t10490*t10488*t10498*(2.0/3.0);
                double t10535 = EcP_1*t10534;
                double t10536 = pow(2.0,1.0/3.0);
                double t10537 = t10536*2.0;
                double t10538 = t10537-2.0;
                double t10539 = two_13*2.0;
                double t10540 = t10539-2.0;
                double t10541 = 1.0/t10540;
                double t10542 = t10512*t10512;
                double t10543 = sqrt(t10519);
                double t10544 = 1.0/t10515;
                double t10545 = t10543*t10544;
                double t10546 = atan(t10545);
                double t10547 = 1.0/sqrt(t10519);
                double t10548 = EcF_2*4.0;
                double t10549 = EcF_3*2.0;
                double t10550 = t10548+t10549;
                double t10551 = EcF_2*EcF_2;
                double t10552 = EcF_2*EcF_3;
                double t10553 = EcF_4+t10551+t10552;
                double t10554 = 1.0/t10553;
                double t10555 = c*t10471*t10480;
                double t10556 = log(t10555);
                double t10557 = EcP_3*t10486*t10487*2.0;
                double t10558 = t10480*t10493;
                double t10559 = log(t10558);
                double t10560 = t10486*t10487*t10497;
                double t10561 = t10560+t10559;
                double t10562 = t10556+t10557-EcP_2*EcP_3*t10502*t10561;
                double t10563 = EcP_1*t10562;
                v_rho_a[Q] += scale * (t10563-t10470*(t10535-t10541*t10538*(t10535-EcF_1*(EcF_2*EcF_3*t10554*(1.0/(t10512*t10512)*t10508*(t10511*t10513*t10542+c*t10512*t10514*t10490*t10488*(1.0/3.0))+c*t10522*t10550*t10516*t10490*t10488*(1.0/3.0))+t10506*t10508*t10509*(c*t10514*t10488*(1.0/3.0)-c*t10511*t10513*t10471)-EcF_3*c*t10522*t10516*t10490*t10488*(2.0/3.0))))-t10541*t10538*(t10563-EcF_1*(log(c*t10514*t10471)+EcF_3*t10546*t10547*2.0-EcF_2*EcF_3*t10554*(log(t10514*t10542)+t10550*t10546*t10547))));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t10565 = rho_a+rho_b;
                double t10566 = 1.0/pow(t10565,1.0/3.0);
                double t10567 = c*t10566;
                double t10568 = sqrt(t10567);
                double t10569 = EcP_4*4.0;
                double t10570 = EcP_3*EcP_3;
                double t10571 = -t10570+t10569;
                double t10572 = EcP_2-t10568;
                double t10573 = EcP_3*t10568;
                double t10574 = EcP_4+t10573+t10567;
                double t10575 = 1.0/t10574;
                double t10576 = sqrt(t10571);
                double t10577 = t10568*2.0;
                double t10578 = EcP_3+t10577;
                double t10579 = 1.0/t10578;
                double t10580 = t10576*t10579;
                double t10581 = atan(t10580);
                double t10582 = 1.0/sqrt(t10571);
                double t10583 = 1.0/pow(t10565,4.0/3.0);
                double t10584 = c*t10583*(1.0/3.0);
                double t10585 = 1.0/sqrt(t10567);
                double t10586 = EcP_3*c*t10583*t10585*(1.0/6.0);
                double t10587 = t10584+t10586;
                double t10588 = t10572*t10572;
                double t10589 = 1.0/(t10574*t10574);
                double t10590 = EcP_2*4.0;
                double t10591 = EcP_3*2.0;
                double t10592 = t10590+t10591;
                double t10593 = 1.0/(t10578*t10578);
                double t10594 = EcP_2*EcP_2;
                double t10595 = EcP_2*EcP_3;
                double t10596 = EcP_4+t10594+t10595;
                double t10597 = 1.0/t10596;
                double t10598 = t10571*t10593;
                double t10599 = t10598+1.0;
                double t10600 = 1.0/t10599;
                double t10601 = 1.0/c;
                double t10602 = EcF_3*t10568;
                double t10603 = EcF_4+t10602+t10567;
                double t10604 = pow(t10565,1.0/3.0);
                double t10605 = EcF_3*c*t10583*t10585*(1.0/6.0);
                double t10606 = t10605+t10584;
                double t10607 = EcF_2-t10568;
                double t10608 = 1.0/(t10603*t10603);
                double t10609 = 1.0/t10603;
                double t10610 = EcF_3+t10577;
                double t10611 = 1.0/(t10610*t10610);
                double t10612 = EcF_4*4.0;
                double t10613 = EcF_3*EcF_3;
                double t10614 = t10612-t10613;
                double t10615 = t10611*t10614;
                double t10616 = t10615+1.0;
                double t10617 = 1.0/t10616;
                double t10618 = c*t10583*t10575*(1.0/3.0);
                double t10619 = t10618-c*t10566*t10587*t10589;
                double t10620 = t10601*t10604*t10574*t10619;
                double t10621 = t10587*t10588*t10589;
                double t10622 = c*t10572*t10583*t10575*t10585*(1.0/3.0);
                double t10623 = t10621+t10622;
                double t10624 = 1.0/(t10572*t10572);
                double t10625 = t10623*t10624*t10574;
                double t10626 = c*t10600*t10583*t10592*t10593*t10585*(1.0/3.0);
                double t10627 = t10625+t10626;
                double t10628 = EcP_2*EcP_3*t10627*t10597;
                double t10629 = t10620+t10628-EcP_3*c*t10600*t10583*t10593*t10585*(2.0/3.0);
                double t10630 = EcP_1*t10629;
                double t10631 = pow(2.0,1.0/3.0);
                double t10632 = t10631*2.0;
                double t10633 = t10632-2.0;
                double t10634 = two_13*2.0;
                double t10635 = t10634-2.0;
                double t10636 = 1.0/t10635;
                double t10637 = t10607*t10607;
                double t10638 = sqrt(t10614);
                double t10639 = 1.0/t10610;
                double t10640 = t10638*t10639;
                double t10641 = atan(t10640);
                double t10642 = 1.0/sqrt(t10614);
                double t10643 = EcF_2*4.0;
                double t10644 = EcF_3*2.0;
                double t10645 = t10643+t10644;
                double t10646 = EcF_2*EcF_2;
                double t10647 = EcF_2*EcF_3;
                double t10648 = EcF_4+t10646+t10647;
                double t10649 = 1.0/t10648;
                double t10650 = c*t10566*t10575;
                double t10651 = log(t10650);
                double t10652 = EcP_3*t10581*t10582*2.0;
                double t10653 = t10575*t10588;
                double t10654 = log(t10653);
                double t10655 = t10581*t10582*t10592;
                double t10656 = t10654+t10655;
                double t10657 = t10651+t10652-EcP_2*EcP_3*t10656*t10597;
                double t10658 = EcP_1*t10657;
                v_rho_b[Q] += scale * (t10658-t10565*(t10630-t10633*t10636*(t10630-EcF_1*(EcF_2*EcF_3*t10649*(t10603*1.0/(t10607*t10607)*(t10606*t10608*t10637+c*t10607*t10609*t10583*t10585*(1.0/3.0))+c*t10611*t10617*t10645*t10583*t10585*(1.0/3.0))+t10601*t10603*t10604*(c*t10609*t10583*(1.0/3.0)-c*t10606*t10608*t10566)-EcF_3*c*t10611*t10617*t10583*t10585*(2.0/3.0))))-t10633*t10636*(t10658-EcF_1*(log(c*t10609*t10566)+EcF_3*t10641*t10642*2.0-EcF_2*EcF_3*t10649*(log(t10609*t10637)+t10641*t10642*t10645))));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t10665 = rho_a+rho_b;
                double t10666 = 1.0/pow(t10665,1.0/3.0);
                double t10667 = c*t10666;
                double t10668 = 1.0/pow(t10665,4.0/3.0);
                double t10669 = sqrt(t10667);
                double t10670 = EcP_3*t10669;
                double t10671 = EcP_4+t10670+t10667;
                double t10672 = c*t10668*(1.0/3.0);
                double t10673 = 1.0/sqrt(t10667);
                double t10674 = EcP_3*c*t10673*t10668*(1.0/6.0);
                double t10675 = t10672+t10674;
                double t10676 = EcP_2-t10669;
                double t10677 = 1.0/(t10671*t10671);
                double t10678 = 1.0/t10671;
                double t10679 = t10669*2.0;
                double t10680 = EcP_3+t10679;
                double t10681 = 1.0/(t10680*t10680);
                double t10682 = EcP_4*4.0;
                double t10683 = EcP_3*EcP_3;
                double t10684 = t10682-t10683;
                double t10685 = t10681*t10684;
                double t10686 = t10685+1.0;
                double t10687 = 1.0/t10686;
                double t10688 = 1.0/c;
                double t10689 = c*t10668*t10678*(1.0/3.0);
                double t10692 = c*t10666*t10675*t10677;
                double t10690 = -t10692+t10689;
                double t10691 = pow(t10665,1.0/3.0);
                double t10693 = 1.0/pow(t10665,7.0/3.0);
                double t10694 = EcP_2*EcP_2;
                double t10695 = EcP_2*EcP_3;
                double t10696 = EcP_4+t10694+t10695;
                double t10697 = 1.0/t10696;
                double t10698 = t10676*t10676;
                double t10699 = t10675*t10677*t10698;
                double t10700 = c*t10673*t10676*t10668*t10678*(1.0/3.0);
                double t10701 = t10700+t10699;
                double t10702 = 1.0/(t10676*t10676);
                double t10703 = c*t10693*(4.0/9.0);
                double t10704 = EcP_3*c*t10673*t10693*(2.0/9.0);
                double t10705 = c*c;
                double t10706 = 1.0/pow(t10665,8.0/3.0);
                double t10707 = 1.0/pow(t10667,3.0/2.0);
                double t10752 = EcP_3*t10705*t10706*t10707*(1.0/3.6E1);
                double t10708 = t10703+t10704-t10752;
                double t10709 = t10675*t10675;
                double t10710 = 1.0/(t10671*t10671*t10671);
                double t10711 = EcP_2*4.0;
                double t10712 = EcP_3*2.0;
                double t10713 = t10711+t10712;
                double t10714 = 1.0/(t10680*t10680*t10680);
                double t10715 = 1.0/(t10680*t10680*t10680*t10680*t10680);
                double t10716 = 1.0/(t10686*t10686);
                double t10717 = EcF_3*t10669;
                double t10718 = EcF_4+t10717+t10667;
                double t10719 = EcF_3*c*t10673*t10668*(1.0/6.0);
                double t10720 = t10672+t10719;
                double t10721 = 1.0/t10718;
                double t10722 = c*t10721*t10668*(1.0/3.0);
                double t10723 = 1.0/(t10718*t10718);
                double t10778 = c*t10720*t10723*t10666;
                double t10724 = t10722-t10778;
                double t10725 = 1.0/pow(t10665,2.0/3.0);
                double t10726 = EcF_2-t10669;
                double t10727 = 1.0/(t10726*t10726);
                double t10728 = t10726*t10726;
                double t10729 = EcF_3*c*t10673*t10693*(2.0/9.0);
                double t10730 = t10703+t10729-EcF_3*t10705*t10706*t10707*(1.0/3.6E1);
                double t10731 = t10720*t10720;
                double t10732 = 1.0/(t10718*t10718*t10718);
                double t10733 = EcF_3+t10679;
                double t10734 = EcF_2*4.0;
                double t10735 = EcF_3*2.0;
                double t10736 = t10734+t10735;
                double t10737 = 1.0/(t10733*t10733);
                double t10738 = EcF_4*4.0;
                double t10739 = EcF_3*EcF_3;
                double t10740 = t10738-t10739;
                double t10741 = t10740*t10737;
                double t10742 = t10741+1.0;
                double t10743 = 1.0/t10742;
                double t10744 = t10720*t10723*t10728;
                double t10745 = c*t10721*t10726*t10673*t10668*(1.0/3.0);
                double t10746 = t10744+t10745;
                double t10747 = 1.0/(t10733*t10733*t10733);
                double t10748 = 1.0/(t10733*t10733*t10733*t10733*t10733);
                double t10749 = 1.0/(t10742*t10742);
                double t10750 = t10690*t10691*t10675*t10688;
                double t10751 = c*t10693*t10678*(4.0/9.0);
                double t10753 = c*t10710*t10709*t10666*2.0;
                double t10754 = t10751+t10753-c*t10708*t10666*t10677-c*t10675*t10668*t10677*(2.0/3.0);
                double t10755 = t10671*t10691*t10754*t10688;
                double t10756 = t10701*t10702*t10675;
                double t10757 = c*t10693*t10678*(1.0/1.8E1);
                double t10758 = t10710*t10709*t10698*2.0;
                double t10759 = t10705*t10706*t10707*t10676*t10678*(1.0/1.8E1);
                double t10760 = c*t10673*t10675*t10676*t10668*t10677*(2.0/3.0);
                double t10761 = t10760+t10757+t10758+t10759-t10708*t10677*t10698-c*t10673*t10693*t10676*t10678*(4.0/9.0);
                double t10762 = c*t10713*t10681*t10673*t10693*t10687*(4.0/9.0);
                double t10763 = 1.0/(t10676*t10676*t10676);
                double t10764 = c*t10701*t10671*t10673*t10763*t10668*(1.0/3.0);
                double t10765 = c*t10713*t10715*t10716*t10684*t10693*(2.0/9.0);
                double t10766 = t10762+t10764+t10756+t10765-t10702*t10671*t10761-c*t10713*t10714*t10693*t10687*(2.0/9.0)-t10713*t10705*t10706*t10707*t10681*t10687*(1.0/1.8E1);
                double t10767 = EcP_2*EcP_3*t10766*t10697;
                double t10768 = EcP_3*c*t10714*t10693*t10687*(4.0/9.0);
                double t10769 = EcP_3*t10705*t10706*t10707*t10681*t10687*(1.0/9.0);
                double t10770 = t10750+t10755+t10767+t10768+t10769-t10671*t10725*t10690*t10688*(1.0/3.0)-EcP_3*c*t10715*t10716*t10684*t10693*(4.0/9.0)-EcP_3*c*t10681*t10673*t10693*t10687*(8.0/9.0);
                double t10771 = EcP_1*t10770;
                double t10772 = pow(2.0,1.0/3.0);
                double t10773 = t10772*2.0;
                double t10774 = t10773-2.0;
                double t10775 = two_13*2.0;
                double t10776 = t10775-2.0;
                double t10777 = 1.0/t10776;
                double t10779 = EcF_2*EcF_2;
                double t10780 = EcF_2*EcF_3;
                double t10781 = EcF_4+t10780+t10779;
                double t10782 = 1.0/t10781;
                double t10783 = t10671*t10690*t10691*t10688;
                double t10784 = t10701*t10702*t10671;
                double t10785 = c*t10713*t10681*t10673*t10668*t10687*(1.0/3.0);
                double t10786 = t10784+t10785;
                double t10787 = EcP_2*EcP_3*t10786*t10697;
                double t10788 = t10783+t10787-EcP_3*c*t10681*t10673*t10668*t10687*(2.0/3.0);
                v_rho_a_rho_a[Q] += scale * (EcP_1*t10788*-2.0+t10665*(t10771-t10774*t10777*(t10771-EcF_1*(t10691*t10718*t10688*(c*t10721*t10693*(4.0/9.0)-c*t10730*t10723*t10666-c*t10720*t10723*t10668*(2.0/3.0)+c*t10731*t10732*t10666*2.0)+t10720*t10724*t10691*t10688-t10724*t10725*t10718*t10688*(1.0/3.0)+EcF_2*EcF_3*t10782*(-t10718*t10727*(c*t10721*t10693*(1.0/1.8E1)-t10730*t10723*t10728+t10731*t10732*t10728*2.0-c*t10721*t10726*t10673*t10693*(4.0/9.0)+t10721*t10705*t10706*t10707*t10726*(1.0/1.8E1)+c*t10720*t10723*t10726*t10673*t10668*(2.0/3.0))+t10720*t10727*t10746-c*t10743*t10736*t10693*t10747*(2.0/9.0)+c*t10743*t10673*t10736*t10737*t10693*(4.0/9.0)+c*t10740*t10736*t10693*t10748*t10749*(2.0/9.0)-t10705*t10706*t10707*t10743*t10736*t10737*(1.0/1.8E1)+c*1.0/(t10726*t10726*t10726)*t10673*t10718*t10746*t10668*(1.0/3.0))+EcF_3*c*t10743*t10693*t10747*(4.0/9.0)-EcF_3*c*t10743*t10673*t10737*t10693*(8.0/9.0)-EcF_3*c*t10740*t10693*t10748*t10749*(4.0/9.0)+EcF_3*t10705*t10706*t10707*t10743*t10737*(1.0/9.0))))+t10774*t10777*(EcP_1*t10788-EcF_1*(t10724*t10691*t10718*t10688+EcF_2*EcF_3*t10782*(t10718*t10727*t10746+c*t10743*t10673*t10736*t10737*t10668*(1.0/3.0))-EcF_3*c*t10743*t10673*t10737*t10668*(2.0/3.0)))*2.0);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t10790 = rho_a+rho_b;
                double t10791 = 1.0/pow(t10790,1.0/3.0);
                double t10792 = c*t10791;
                double t10793 = 1.0/pow(t10790,4.0/3.0);
                double t10794 = sqrt(t10792);
                double t10795 = EcP_3*t10794;
                double t10796 = EcP_4+t10792+t10795;
                double t10797 = c*t10793*(1.0/3.0);
                double t10798 = 1.0/sqrt(t10792);
                double t10799 = EcP_3*c*t10793*t10798*(1.0/6.0);
                double t10800 = t10797+t10799;
                double t10801 = EcP_2-t10794;
                double t10802 = 1.0/(t10796*t10796);
                double t10803 = 1.0/t10796;
                double t10804 = t10794*2.0;
                double t10805 = EcP_3+t10804;
                double t10806 = 1.0/(t10805*t10805);
                double t10807 = EcP_4*4.0;
                double t10808 = EcP_3*EcP_3;
                double t10809 = t10807-t10808;
                double t10810 = t10806*t10809;
                double t10811 = t10810+1.0;
                double t10812 = 1.0/t10811;
                double t10813 = 1.0/c;
                double t10814 = c*t10803*t10793*(1.0/3.0);
                double t10817 = c*t10800*t10802*t10791;
                double t10815 = t10814-t10817;
                double t10816 = pow(t10790,1.0/3.0);
                double t10818 = 1.0/pow(t10790,7.0/3.0);
                double t10819 = EcP_2*EcP_2;
                double t10820 = EcP_2*EcP_3;
                double t10821 = EcP_4+t10820+t10819;
                double t10822 = 1.0/t10821;
                double t10823 = t10801*t10801;
                double t10824 = t10800*t10802*t10823;
                double t10825 = c*t10801*t10803*t10793*t10798*(1.0/3.0);
                double t10826 = t10824+t10825;
                double t10827 = 1.0/(t10801*t10801);
                double t10828 = c*t10818*(4.0/9.0);
                double t10829 = EcP_3*c*t10818*t10798*(2.0/9.0);
                double t10830 = c*c;
                double t10831 = 1.0/pow(t10790,8.0/3.0);
                double t10832 = 1.0/pow(t10792,3.0/2.0);
                double t10877 = EcP_3*t10830*t10831*t10832*(1.0/3.6E1);
                double t10833 = t10828+t10829-t10877;
                double t10834 = t10800*t10800;
                double t10835 = 1.0/(t10796*t10796*t10796);
                double t10836 = EcP_2*4.0;
                double t10837 = EcP_3*2.0;
                double t10838 = t10836+t10837;
                double t10839 = 1.0/(t10805*t10805*t10805);
                double t10840 = 1.0/(t10805*t10805*t10805*t10805*t10805);
                double t10841 = 1.0/(t10811*t10811);
                double t10842 = EcF_3*t10794;
                double t10843 = EcF_4+t10842+t10792;
                double t10844 = EcF_3*c*t10793*t10798*(1.0/6.0);
                double t10845 = t10844+t10797;
                double t10846 = 1.0/t10843;
                double t10847 = c*t10846*t10793*(1.0/3.0);
                double t10848 = 1.0/(t10843*t10843);
                double t10903 = c*t10791*t10845*t10848;
                double t10849 = -t10903+t10847;
                double t10850 = 1.0/pow(t10790,2.0/3.0);
                double t10851 = EcF_2-t10794;
                double t10852 = 1.0/(t10851*t10851);
                double t10853 = t10851*t10851;
                double t10854 = EcF_3*c*t10818*t10798*(2.0/9.0);
                double t10855 = t10854+t10828-EcF_3*t10830*t10831*t10832*(1.0/3.6E1);
                double t10856 = t10845*t10845;
                double t10857 = 1.0/(t10843*t10843*t10843);
                double t10858 = EcF_3+t10804;
                double t10859 = EcF_2*4.0;
                double t10860 = EcF_3*2.0;
                double t10861 = t10860+t10859;
                double t10862 = 1.0/(t10858*t10858);
                double t10863 = EcF_4*4.0;
                double t10864 = EcF_3*EcF_3;
                double t10865 = t10863-t10864;
                double t10866 = t10862*t10865;
                double t10867 = t10866+1.0;
                double t10868 = 1.0/t10867;
                double t10869 = t10853*t10845*t10848;
                double t10870 = c*t10851*t10846*t10793*t10798*(1.0/3.0);
                double t10871 = t10870+t10869;
                double t10872 = 1.0/(t10858*t10858*t10858);
                double t10873 = 1.0/(t10858*t10858*t10858*t10858*t10858);
                double t10874 = 1.0/(t10867*t10867);
                double t10875 = t10800*t10813*t10815*t10816;
                double t10876 = c*t10803*t10818*(4.0/9.0);
                double t10878 = c*t10834*t10835*t10791*2.0;
                double t10879 = t10876+t10878-c*t10800*t10802*t10793*(2.0/3.0)-c*t10802*t10833*t10791;
                double t10880 = t10813*t10816*t10796*t10879;
                double t10881 = t10800*t10826*t10827;
                double t10882 = c*t10803*t10818*(1.0/1.8E1);
                double t10883 = t10823*t10834*t10835*2.0;
                double t10884 = t10801*t10803*t10830*t10831*t10832*(1.0/1.8E1);
                double t10885 = c*t10800*t10801*t10802*t10793*t10798*(2.0/3.0);
                double t10886 = t10882+t10883+t10884+t10885-t10802*t10823*t10833-c*t10801*t10803*t10818*t10798*(4.0/9.0);
                double t10887 = c*t10812*t10806*t10818*t10838*t10798*(4.0/9.0);
                double t10888 = 1.0/(t10801*t10801*t10801);
                double t10889 = c*t10826*t10793*t10796*t10798*t10888*(1.0/3.0);
                double t10890 = c*t10840*t10841*t10809*t10818*t10838*(2.0/9.0);
                double t10891 = t10881+t10890+t10887+t10889-t10827*t10796*t10886-c*t10812*t10818*t10838*t10839*(2.0/9.0)-t10812*t10830*t10831*t10832*t10806*t10838*(1.0/1.8E1);
                double t10892 = EcP_2*EcP_3*t10822*t10891;
                double t10893 = EcP_3*c*t10812*t10818*t10839*(4.0/9.0);
                double t10894 = EcP_3*t10812*t10830*t10831*t10832*t10806*(1.0/9.0);
                double t10895 = t10880+t10892+t10875+t10893+t10894-t10813*t10850*t10815*t10796*(1.0/3.0)-EcP_3*c*t10840*t10841*t10809*t10818*(4.0/9.0)-EcP_3*c*t10812*t10806*t10818*t10798*(8.0/9.0);
                double t10896 = EcP_1*t10895;
                double t10897 = pow(2.0,1.0/3.0);
                double t10898 = t10897*2.0;
                double t10899 = t10898-2.0;
                double t10900 = two_13*2.0;
                double t10901 = t10900-2.0;
                double t10902 = 1.0/t10901;
                double t10904 = EcF_2*EcF_2;
                double t10905 = EcF_2*EcF_3;
                double t10906 = EcF_4+t10904+t10905;
                double t10907 = 1.0/t10906;
                double t10908 = t10813*t10815*t10816*t10796;
                double t10909 = t10826*t10827*t10796;
                double t10910 = c*t10812*t10806*t10793*t10838*t10798*(1.0/3.0);
                double t10911 = t10910+t10909;
                double t10912 = EcP_2*EcP_3*t10911*t10822;
                double t10913 = t10912+t10908-EcP_3*c*t10812*t10806*t10793*t10798*(2.0/3.0);
                v_rho_a_rho_b[Q] += scale * (EcP_1*t10913*-2.0+t10790*(t10896-t10902*t10899*(t10896-EcF_1*(t10813*t10816*t10843*(c*t10818*t10846*(4.0/9.0)-c*t10791*t10855*t10848+c*t10791*t10856*t10857*2.0-c*t10845*t10793*t10848*(2.0/3.0))-t10813*t10850*t10843*t10849*(1.0/3.0)+t10813*t10816*t10845*t10849+EcF_2*EcF_3*t10907*(-t10843*t10852*(c*t10818*t10846*(1.0/1.8E1)-t10853*t10855*t10848+t10853*t10856*t10857*2.0-c*t10851*t10818*t10846*t10798*(4.0/9.0)+t10830*t10831*t10832*t10851*t10846*(1.0/1.8E1)+c*t10851*t10845*t10793*t10848*t10798*(2.0/3.0))+t10852*t10871*t10845-c*t10861*t10818*t10872*t10868*(2.0/9.0)+c*t10861*t10818*t10873*t10865*t10874*(2.0/9.0)+c*t10861*t10862*t10818*t10868*t10798*(4.0/9.0)-t10830*t10831*t10832*t10861*t10862*t10868*(1.0/1.8E1)+c*1.0/(t10851*t10851*t10851)*t10843*t10871*t10793*t10798*(1.0/3.0))+EcF_3*c*t10818*t10872*t10868*(4.0/9.0)-EcF_3*c*t10818*t10873*t10865*t10874*(4.0/9.0)-EcF_3*c*t10862*t10818*t10868*t10798*(8.0/9.0)+EcF_3*t10830*t10831*t10832*t10862*t10868*(1.0/9.0))))+t10902*t10899*(EcP_1*t10913-EcF_1*(t10813*t10816*t10843*t10849+EcF_2*EcF_3*t10907*(t10843*t10852*t10871+c*t10861*t10862*t10793*t10868*t10798*(1.0/3.0))-EcF_3*c*t10862*t10793*t10868*t10798*(2.0/3.0)))*2.0);
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t10915 = rho_a+rho_b;
                double t10916 = 1.0/pow(t10915,1.0/3.0);
                double t10917 = c*t10916;
                double t10918 = 1.0/pow(t10915,4.0/3.0);
                double t10919 = sqrt(t10917);
                double t10920 = EcP_3*t10919;
                double t10921 = EcP_4+t10920+t10917;
                double t10922 = c*t10918*(1.0/3.0);
                double t10923 = 1.0/sqrt(t10917);
                double t10924 = EcP_3*c*t10923*t10918*(1.0/6.0);
                double t10925 = t10922+t10924;
                double t10926 = EcP_2-t10919;
                double t10927 = 1.0/(t10921*t10921);
                double t10928 = 1.0/t10921;
                double t10929 = t10919*2.0;
                double t10930 = EcP_3+t10929;
                double t10931 = 1.0/(t10930*t10930);
                double t10932 = EcP_4*4.0;
                double t10933 = EcP_3*EcP_3;
                double t10934 = t10932-t10933;
                double t10935 = t10931*t10934;
                double t10936 = t10935+1.0;
                double t10937 = 1.0/t10936;
                double t10938 = 1.0/c;
                double t10939 = c*t10918*t10928*(1.0/3.0);
                double t10942 = c*t10916*t10925*t10927;
                double t10940 = -t10942+t10939;
                double t10941 = pow(t10915,1.0/3.0);
                double t10943 = 1.0/pow(t10915,7.0/3.0);
                double t10944 = EcP_2*EcP_2;
                double t10945 = EcP_2*EcP_3;
                double t10946 = EcP_4+t10944+t10945;
                double t10947 = 1.0/t10946;
                double t10948 = t10926*t10926;
                double t10949 = t10925*t10927*t10948;
                double t10950 = c*t10923*t10926*t10918*t10928*(1.0/3.0);
                double t10951 = t10950+t10949;
                double t10952 = 1.0/(t10926*t10926);
                double t10953 = c*t10943*(4.0/9.0);
                double t10954 = EcP_3*c*t10923*t10943*(2.0/9.0);
                double t10955 = c*c;
                double t10956 = 1.0/pow(t10915,8.0/3.0);
                double t10957 = 1.0/pow(t10917,3.0/2.0);
                double t11002 = EcP_3*t10955*t10956*t10957*(1.0/3.6E1);
                double t10958 = t10953+t10954-t11002;
                double t10959 = t10925*t10925;
                double t10960 = 1.0/(t10921*t10921*t10921);
                double t10961 = EcP_2*4.0;
                double t10962 = EcP_3*2.0;
                double t10963 = t10961+t10962;
                double t10964 = 1.0/(t10930*t10930*t10930);
                double t10965 = 1.0/(t10930*t10930*t10930*t10930*t10930);
                double t10966 = 1.0/(t10936*t10936);
                double t10967 = EcF_3*t10919;
                double t10968 = EcF_4+t10917+t10967;
                double t10969 = EcF_3*c*t10923*t10918*(1.0/6.0);
                double t10970 = t10922+t10969;
                double t10971 = 1.0/t10968;
                double t10972 = c*t10971*t10918*(1.0/3.0);
                double t10973 = 1.0/(t10968*t10968);
                double t11028 = c*t10916*t10970*t10973;
                double t10974 = t10972-t11028;
                double t10975 = 1.0/pow(t10915,2.0/3.0);
                double t10976 = EcF_2-t10919;
                double t10977 = 1.0/(t10976*t10976);
                double t10978 = t10976*t10976;
                double t10979 = EcF_3*c*t10923*t10943*(2.0/9.0);
                double t10980 = t10953+t10979-EcF_3*t10955*t10956*t10957*(1.0/3.6E1);
                double t10981 = t10970*t10970;
                double t10982 = 1.0/(t10968*t10968*t10968);
                double t10983 = EcF_3+t10929;
                double t10984 = EcF_2*4.0;
                double t10985 = EcF_3*2.0;
                double t10986 = t10984+t10985;
                double t10987 = 1.0/(t10983*t10983);
                double t10988 = EcF_4*4.0;
                double t10989 = EcF_3*EcF_3;
                double t10990 = t10988-t10989;
                double t10991 = t10990*t10987;
                double t10992 = t10991+1.0;
                double t10993 = 1.0/t10992;
                double t10994 = t10970*t10973*t10978;
                double t10995 = c*t10923*t10971*t10918*t10976*(1.0/3.0);
                double t10996 = t10994+t10995;
                double t10997 = 1.0/(t10983*t10983*t10983);
                double t10998 = 1.0/(t10983*t10983*t10983*t10983*t10983);
                double t10999 = 1.0/(t10992*t10992);
                double t11000 = t10940*t10941*t10925*t10938;
                double t11001 = c*t10943*t10928*(4.0/9.0);
                double t11003 = c*t10960*t10916*t10959*2.0;
                double t11004 = t11001+t11003-c*t10925*t10918*t10927*(2.0/3.0)-c*t10916*t10927*t10958;
                double t11005 = t10921*t10941*t10938*t11004;
                double t11006 = t10951*t10925*t10952;
                double t11007 = c*t10943*t10928*(1.0/1.8E1);
                double t11008 = t10960*t10948*t10959*2.0;
                double t11009 = t10926*t10928*t10955*t10956*t10957*(1.0/1.8E1);
                double t11010 = c*t10923*t10925*t10926*t10918*t10927*(2.0/3.0);
                double t11011 = t11010+t11007+t11008+t11009-t10927*t10948*t10958-c*t10923*t10943*t10926*t10928*(4.0/9.0);
                double t11012 = c*t10931*t10923*t10943*t10963*t10937*(4.0/9.0);
                double t11013 = 1.0/(t10926*t10926*t10926);
                double t11014 = c*t10921*t10923*t10951*t10918*t11013*(1.0/3.0);
                double t11015 = c*t10934*t10943*t10963*t10965*t10966*(2.0/9.0);
                double t11016 = t11012+t11014+t11006+t11015-t10921*t10952*t11011-c*t10943*t10963*t10937*t10964*(2.0/9.0)-t10931*t10963*t10937*t10955*t10956*t10957*(1.0/1.8E1);
                double t11017 = EcP_2*EcP_3*t10947*t11016;
                double t11018 = EcP_3*c*t10943*t10937*t10964*(4.0/9.0);
                double t11019 = EcP_3*t10931*t10937*t10955*t10956*t10957*(1.0/9.0);
                double t11020 = t11000+t11005+t11017+t11018+t11019-t10921*t10940*t10938*t10975*(1.0/3.0)-EcP_3*c*t10931*t10923*t10943*t10937*(8.0/9.0)-EcP_3*c*t10934*t10943*t10965*t10966*(4.0/9.0);
                double t11021 = EcP_1*t11020;
                double t11022 = pow(2.0,1.0/3.0);
                double t11023 = t11022*2.0;
                double t11024 = t11023-2.0;
                double t11025 = two_13*2.0;
                double t11026 = t11025-2.0;
                double t11027 = 1.0/t11026;
                double t11029 = EcF_2*EcF_2;
                double t11030 = EcF_2*EcF_3;
                double t11031 = EcF_4+t11030+t11029;
                double t11032 = 1.0/t11031;
                double t11033 = t10921*t10940*t10941*t10938;
                double t11034 = t10921*t10951*t10952;
                double t11035 = c*t10931*t10923*t10918*t10963*t10937*(1.0/3.0);
                double t11036 = t11034+t11035;
                double t11037 = EcP_2*EcP_3*t10947*t11036;
                double t11038 = t11033+t11037-EcP_3*c*t10931*t10923*t10918*t10937*(2.0/3.0);
                v_rho_b_rho_b[Q] += scale * (EcP_1*t11038*-2.0+t10915*(t11021-t11024*t11027*(t11021-EcF_1*(t10941*t10938*t10968*(c*t10943*t10971*(4.0/9.0)-c*t10916*t10980*t10973+c*t10916*t10981*t10982*2.0-c*t10970*t10918*t10973*(2.0/3.0))+t10941*t10970*t10938*t10974-t10938*t10974*t10975*t10968*(1.0/3.0)+EcF_2*EcF_3*t11032*(-t10968*t10977*(c*t10943*t10971*(1.0/1.8E1)-t10980*t10973*t10978+t10981*t10982*t10978*2.0-c*t10923*t10943*t10971*t10976*(4.0/9.0)+t10971*t10955*t10956*t10957*t10976*(1.0/1.8E1)+c*t10923*t10970*t10918*t10973*t10976*(2.0/3.0))+t10970*t10977*t10996-c*t10943*t10993*t10986*t10997*(2.0/9.0)+c*t10923*t10943*t10993*t10986*t10987*(4.0/9.0)+c*t10943*t10990*t10986*t10998*t10999*(2.0/9.0)-t10955*t10956*t10957*t10993*t10986*t10987*(1.0/1.8E1)+c*t10923*t10918*1.0/(t10976*t10976*t10976)*t10968*t10996*(1.0/3.0))+EcF_3*c*t10943*t10993*t10997*(4.0/9.0)-EcF_3*c*t10923*t10943*t10993*t10987*(8.0/9.0)-EcF_3*c*t10943*t10990*t10998*t10999*(4.0/9.0)+EcF_3*t10955*t10956*t10957*t10993*t10987*(1.0/9.0))))+t11024*t11027*(EcP_1*t11038-EcF_1*(t10941*t10938*t10974*t10968+EcF_2*EcF_3*t11032*(t10968*t10977*t10996+c*t10923*t10918*t10993*t10986*t10987*(1.0/3.0))-EcF_3*c*t10923*t10918*t10993*t10987*(2.0/3.0)))*2.0);
            }
            
        } else if (rho_b < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t11065 = rho_a+rho_b;
                double t11066 = 1.0/pow(t11065,1.0/3.0);
                double t11067 = c*t11066;
                double t11068 = sqrt(t11067);
                double t11069 = EcP_4*4.0;
                double t11070 = EcP_3*EcP_3;
                double t11071 = -t11070+t11069;
                double t11072 = EcP_2-t11068;
                double t11073 = EcP_3*t11068;
                double t11074 = EcP_4+t11073+t11067;
                double t11075 = 1.0/t11074;
                double t11076 = sqrt(t11071);
                double t11077 = t11068*2.0;
                double t11078 = EcP_3+t11077;
                double t11079 = 1.0/t11078;
                double t11080 = t11076*t11079;
                double t11081 = atan(t11080);
                double t11082 = 1.0/sqrt(t11071);
                double t11083 = EcF_4*4.0;
                double t11084 = EcF_3*EcF_3;
                double t11085 = t11083-t11084;
                double t11086 = EcF_2-t11068;
                double t11087 = EcF_3*t11068;
                double t11088 = EcF_4+t11067+t11087;
                double t11089 = 1.0/t11088;
                double t11090 = sqrt(t11085);
                double t11091 = EcF_3+t11077;
                double t11092 = 1.0/t11091;
                double t11093 = t11090*t11092;
                double t11094 = atan(t11093);
                double t11095 = 1.0/sqrt(t11085);
                double t11096 = c*t11066*t11075;
                double t11097 = log(t11096);
                double t11098 = EcP_3*t11081*t11082*2.0;
                double t11099 = t11072*t11072;
                double t11100 = t11075*t11099;
                double t11101 = log(t11100);
                double t11102 = EcP_2*4.0;
                double t11103 = EcP_3*2.0;
                double t11104 = t11102+t11103;
                double t11105 = t11104*t11081*t11082;
                double t11106 = t11101+t11105;
                double t11107 = EcP_2*EcP_2;
                double t11108 = EcP_2*EcP_3;
                double t11109 = EcP_4+t11107+t11108;
                double t11110 = 1.0/t11109;
                double t11111 = t11097+t11098-EcP_2*EcP_3*t11110*t11106;
                double t11112 = EcP_1*t11111;
                v[Q] += scale * (t11065*(t11112-((pow(2.0,1.0/3.0)*2.0-2.0)*(t11112-EcF_1*(log(c*t11066*t11089)+EcF_3*t11094*t11095*2.0-(EcF_2*EcF_3*(log((t11086*t11086)*t11089)+t11094*t11095*(EcF_2*4.0+EcF_3*2.0)))/(EcF_4+EcF_2*EcF_2+EcF_2*EcF_3))))/(two_13*2.0-2.0)));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t11114 = rho_a+rho_b;
                double t11115 = 1.0/pow(t11114,1.0/3.0);
                double t11116 = c*t11115;
                double t11117 = sqrt(t11116);
                double t11118 = EcP_4*4.0;
                double t11119 = EcP_3*EcP_3;
                double t11120 = t11118-t11119;
                double t11121 = EcP_2-t11117;
                double t11122 = EcP_3*t11117;
                double t11123 = EcP_4+t11122+t11116;
                double t11124 = 1.0/t11123;
                double t11125 = sqrt(t11120);
                double t11126 = t11117*2.0;
                double t11127 = EcP_3+t11126;
                double t11128 = 1.0/t11127;
                double t11129 = t11125*t11128;
                double t11130 = atan(t11129);
                double t11131 = 1.0/sqrt(t11120);
                double t11132 = 1.0/pow(t11114,4.0/3.0);
                double t11133 = c*t11132*(1.0/3.0);
                double t11134 = 1.0/sqrt(t11116);
                double t11135 = EcP_3*c*t11132*t11134*(1.0/6.0);
                double t11136 = t11133+t11135;
                double t11137 = t11121*t11121;
                double t11138 = 1.0/(t11123*t11123);
                double t11139 = EcP_2*4.0;
                double t11140 = EcP_3*2.0;
                double t11141 = t11140+t11139;
                double t11142 = 1.0/(t11127*t11127);
                double t11143 = EcP_2*EcP_2;
                double t11144 = EcP_2*EcP_3;
                double t11145 = EcP_4+t11143+t11144;
                double t11146 = 1.0/t11145;
                double t11147 = t11120*t11142;
                double t11148 = t11147+1.0;
                double t11149 = 1.0/t11148;
                double t11150 = 1.0/c;
                double t11151 = EcF_3*t11117;
                double t11152 = EcF_4+t11151+t11116;
                double t11153 = pow(t11114,1.0/3.0);
                double t11154 = EcF_3*c*t11132*t11134*(1.0/6.0);
                double t11155 = t11133+t11154;
                double t11156 = EcF_2-t11117;
                double t11157 = 1.0/(t11152*t11152);
                double t11158 = 1.0/t11152;
                double t11159 = EcF_3+t11126;
                double t11160 = 1.0/(t11159*t11159);
                double t11161 = EcF_4*4.0;
                double t11162 = EcF_3*EcF_3;
                double t11163 = t11161-t11162;
                double t11164 = t11160*t11163;
                double t11165 = t11164+1.0;
                double t11166 = 1.0/t11165;
                double t11167 = c*t11132*t11124*(1.0/3.0);
                double t11168 = t11167-c*t11115*t11136*t11138;
                double t11169 = t11123*t11150*t11153*t11168;
                double t11170 = t11136*t11137*t11138;
                double t11171 = c*t11121*t11132*t11124*t11134*(1.0/3.0);
                double t11172 = t11170+t11171;
                double t11173 = 1.0/(t11121*t11121);
                double t11174 = t11123*t11172*t11173;
                double t11175 = c*t11132*t11141*t11142*t11134*t11149*(1.0/3.0);
                double t11176 = t11174+t11175;
                double t11177 = EcP_2*EcP_3*t11146*t11176;
                double t11178 = t11177+t11169-EcP_3*c*t11132*t11142*t11134*t11149*(2.0/3.0);
                double t11179 = EcP_1*t11178;
                double t11180 = pow(2.0,1.0/3.0);
                double t11181 = t11180*2.0;
                double t11182 = t11181-2.0;
                double t11183 = two_13*2.0;
                double t11184 = t11183-2.0;
                double t11185 = 1.0/t11184;
                double t11186 = t11156*t11156;
                double t11187 = sqrt(t11163);
                double t11188 = 1.0/t11159;
                double t11189 = t11187*t11188;
                double t11190 = atan(t11189);
                double t11191 = 1.0/sqrt(t11163);
                double t11192 = EcF_2*4.0;
                double t11193 = EcF_3*2.0;
                double t11194 = t11192+t11193;
                double t11195 = EcF_2*EcF_2;
                double t11196 = EcF_2*EcF_3;
                double t11197 = EcF_4+t11195+t11196;
                double t11198 = 1.0/t11197;
                double t11199 = c*t11115*t11124;
                double t11200 = log(t11199);
                double t11201 = EcP_3*t11130*t11131*2.0;
                double t11202 = t11124*t11137;
                double t11203 = log(t11202);
                double t11204 = t11130*t11131*t11141;
                double t11205 = t11203+t11204;
                double t11206 = t11200+t11201-EcP_2*EcP_3*t11205*t11146;
                double t11207 = EcP_1*t11206;
                v_rho_a[Q] += scale * (t11207-t11114*(t11179-t11182*t11185*(t11179-EcF_1*(EcF_2*EcF_3*t11198*(t11152*1.0/(t11156*t11156)*(t11155*t11157*t11186+c*t11132*t11134*t11156*t11158*(1.0/3.0))+c*t11132*t11160*t11134*t11166*t11194*(1.0/3.0))+t11150*t11152*t11153*(c*t11132*t11158*(1.0/3.0)-c*t11115*t11155*t11157)-EcF_3*c*t11132*t11160*t11134*t11166*(2.0/3.0))))-t11182*t11185*(t11207-EcF_1*(log(c*t11115*t11158)+EcF_3*t11190*t11191*2.0-EcF_2*EcF_3*t11198*(log(t11158*t11186)+t11190*t11191*t11194))));
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t11209 = rho_a+rho_b;
                double t11210 = 1.0/pow(t11209,1.0/3.0);
                double t11211 = c*t11210;
                double t11212 = sqrt(t11211);
                double t11213 = EcP_4*4.0;
                double t11214 = EcP_3*EcP_3;
                double t11215 = t11213-t11214;
                double t11216 = EcP_2-t11212;
                double t11217 = EcP_3*t11212;
                double t11218 = EcP_4+t11211+t11217;
                double t11219 = 1.0/t11218;
                double t11220 = sqrt(t11215);
                double t11221 = t11212*2.0;
                double t11222 = EcP_3+t11221;
                double t11223 = 1.0/t11222;
                double t11224 = t11220*t11223;
                double t11225 = atan(t11224);
                double t11226 = 1.0/sqrt(t11215);
                double t11227 = 1.0/pow(t11209,4.0/3.0);
                double t11228 = c*t11227*(1.0/3.0);
                double t11229 = 1.0/sqrt(t11211);
                double t11230 = EcP_3*c*t11227*t11229*(1.0/6.0);
                double t11231 = t11230+t11228;
                double t11232 = t11216*t11216;
                double t11233 = 1.0/(t11218*t11218);
                double t11234 = EcP_2*4.0;
                double t11235 = EcP_3*2.0;
                double t11236 = t11234+t11235;
                double t11237 = 1.0/(t11222*t11222);
                double t11238 = EcP_2*EcP_2;
                double t11239 = EcP_2*EcP_3;
                double t11240 = EcP_4+t11238+t11239;
                double t11241 = 1.0/t11240;
                double t11242 = t11215*t11237;
                double t11243 = t11242+1.0;
                double t11244 = 1.0/t11243;
                double t11245 = 1.0/c;
                double t11246 = EcF_3*t11212;
                double t11247 = EcF_4+t11211+t11246;
                double t11248 = pow(t11209,1.0/3.0);
                double t11249 = EcF_3*c*t11227*t11229*(1.0/6.0);
                double t11250 = t11228+t11249;
                double t11251 = EcF_2-t11212;
                double t11252 = 1.0/(t11247*t11247);
                double t11253 = 1.0/t11247;
                double t11254 = EcF_3+t11221;
                double t11255 = 1.0/(t11254*t11254);
                double t11256 = EcF_4*4.0;
                double t11257 = EcF_3*EcF_3;
                double t11258 = t11256-t11257;
                double t11259 = t11255*t11258;
                double t11260 = t11259+1.0;
                double t11261 = 1.0/t11260;
                double t11262 = c*t11227*t11219*(1.0/3.0);
                double t11263 = t11262-c*t11210*t11231*t11233;
                double t11264 = t11218*t11245*t11263*t11248;
                double t11265 = t11231*t11232*t11233;
                double t11266 = c*t11216*t11227*t11219*t11229*(1.0/3.0);
                double t11267 = t11265+t11266;
                double t11268 = 1.0/(t11216*t11216);
                double t11269 = t11218*t11267*t11268;
                double t11270 = c*t11244*t11227*t11236*t11237*t11229*(1.0/3.0);
                double t11271 = t11270+t11269;
                double t11272 = EcP_2*EcP_3*t11241*t11271;
                double t11273 = t11272+t11264-EcP_3*c*t11244*t11227*t11237*t11229*(2.0/3.0);
                double t11274 = EcP_1*t11273;
                double t11275 = pow(2.0,1.0/3.0);
                double t11276 = t11275*2.0;
                double t11277 = t11276-2.0;
                double t11278 = two_13*2.0;
                double t11279 = t11278-2.0;
                double t11280 = 1.0/t11279;
                double t11281 = t11251*t11251;
                double t11282 = sqrt(t11258);
                double t11283 = 1.0/t11254;
                double t11284 = t11282*t11283;
                double t11285 = atan(t11284);
                double t11286 = 1.0/sqrt(t11258);
                double t11287 = EcF_2*4.0;
                double t11288 = EcF_3*2.0;
                double t11289 = t11287+t11288;
                double t11290 = EcF_2*EcF_2;
                double t11291 = EcF_2*EcF_3;
                double t11292 = EcF_4+t11290+t11291;
                double t11293 = 1.0/t11292;
                double t11294 = c*t11210*t11219;
                double t11295 = log(t11294);
                double t11296 = EcP_3*t11225*t11226*2.0;
                double t11297 = t11232*t11219;
                double t11298 = log(t11297);
                double t11299 = t11225*t11226*t11236;
                double t11300 = t11298+t11299;
                double t11301 = t11295+t11296-EcP_2*EcP_3*t11300*t11241;
                double t11302 = EcP_1*t11301;
                v_rho_b[Q] += scale * (t11302-t11209*(t11274-t11280*t11277*(t11274-EcF_1*(EcF_2*EcF_3*t11293*(1.0/(t11251*t11251)*t11247*(t11250*t11252*t11281+c*t11251*t11253*t11227*t11229*(1.0/3.0))+c*t11261*t11227*t11255*t11229*t11289*(1.0/3.0))+t11245*t11247*t11248*(c*t11253*t11227*(1.0/3.0)-c*t11210*t11250*t11252)-EcF_3*c*t11261*t11227*t11255*t11229*(2.0/3.0))))-t11280*t11277*(t11302-EcF_1*(log(c*t11210*t11253)+EcF_3*t11285*t11286*2.0-EcF_2*EcF_3*t11293*(log(t11253*t11281)+t11285*t11286*t11289))));
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t11309 = rho_a+rho_b;
                double t11310 = 1.0/pow(t11309,1.0/3.0);
                double t11311 = c*t11310;
                double t11312 = 1.0/pow(t11309,4.0/3.0);
                double t11313 = sqrt(t11311);
                double t11314 = EcP_3*t11313;
                double t11315 = EcP_4+t11311+t11314;
                double t11316 = c*t11312*(1.0/3.0);
                double t11317 = 1.0/sqrt(t11311);
                double t11318 = EcP_3*c*t11312*t11317*(1.0/6.0);
                double t11319 = t11316+t11318;
                double t11320 = EcP_2-t11313;
                double t11321 = 1.0/(t11315*t11315);
                double t11322 = 1.0/t11315;
                double t11323 = t11313*2.0;
                double t11324 = EcP_3+t11323;
                double t11325 = 1.0/(t11324*t11324);
                double t11326 = EcP_4*4.0;
                double t11327 = EcP_3*EcP_3;
                double t11328 = t11326-t11327;
                double t11329 = t11325*t11328;
                double t11330 = t11329+1.0;
                double t11331 = 1.0/t11330;
                double t11332 = 1.0/c;
                double t11333 = c*t11312*t11322*(1.0/3.0);
                double t11336 = c*t11310*t11321*t11319;
                double t11334 = t11333-t11336;
                double t11335 = pow(t11309,1.0/3.0);
                double t11337 = 1.0/pow(t11309,7.0/3.0);
                double t11338 = EcP_2*EcP_2;
                double t11339 = EcP_2*EcP_3;
                double t11340 = EcP_4+t11338+t11339;
                double t11341 = 1.0/t11340;
                double t11342 = t11320*t11320;
                double t11343 = t11321*t11342*t11319;
                double t11344 = c*t11320*t11312*t11322*t11317*(1.0/3.0);
                double t11345 = t11343+t11344;
                double t11346 = 1.0/(t11320*t11320);
                double t11347 = c*t11337*(4.0/9.0);
                double t11348 = EcP_3*c*t11317*t11337*(2.0/9.0);
                double t11349 = c*c;
                double t11350 = 1.0/pow(t11309,8.0/3.0);
                double t11351 = 1.0/pow(t11311,3.0/2.0);
                double t11396 = EcP_3*t11350*t11351*t11349*(1.0/3.6E1);
                double t11352 = t11347+t11348-t11396;
                double t11353 = t11319*t11319;
                double t11354 = 1.0/(t11315*t11315*t11315);
                double t11355 = EcP_2*4.0;
                double t11356 = EcP_3*2.0;
                double t11357 = t11355+t11356;
                double t11358 = 1.0/(t11324*t11324*t11324);
                double t11359 = 1.0/(t11324*t11324*t11324*t11324*t11324);
                double t11360 = 1.0/(t11330*t11330);
                double t11361 = EcF_3*t11313;
                double t11362 = EcF_4+t11311+t11361;
                double t11363 = EcF_3*c*t11312*t11317*(1.0/6.0);
                double t11364 = t11316+t11363;
                double t11365 = 1.0/t11362;
                double t11366 = c*t11312*t11365*(1.0/3.0);
                double t11367 = 1.0/(t11362*t11362);
                double t11422 = c*t11310*t11364*t11367;
                double t11368 = -t11422+t11366;
                double t11369 = 1.0/pow(t11309,2.0/3.0);
                double t11370 = EcF_2-t11313;
                double t11371 = 1.0/(t11370*t11370);
                double t11372 = t11370*t11370;
                double t11373 = EcF_3*c*t11317*t11337*(2.0/9.0);
                double t11374 = t11373+t11347-EcF_3*t11350*t11351*t11349*(1.0/3.6E1);
                double t11375 = t11364*t11364;
                double t11376 = 1.0/(t11362*t11362*t11362);
                double t11377 = EcF_3+t11323;
                double t11378 = EcF_2*4.0;
                double t11379 = EcF_3*2.0;
                double t11380 = t11378+t11379;
                double t11381 = 1.0/(t11377*t11377);
                double t11382 = EcF_4*4.0;
                double t11383 = EcF_3*EcF_3;
                double t11384 = t11382-t11383;
                double t11385 = t11381*t11384;
                double t11386 = t11385+1.0;
                double t11387 = 1.0/t11386;
                double t11388 = t11372*t11364*t11367;
                double t11389 = c*t11312*t11370*t11317*t11365*(1.0/3.0);
                double t11390 = t11388+t11389;
                double t11391 = 1.0/(t11377*t11377*t11377);
                double t11392 = 1.0/(t11377*t11377*t11377*t11377*t11377);
                double t11393 = 1.0/(t11386*t11386);
                double t11394 = t11332*t11334*t11335*t11319;
                double t11395 = c*t11322*t11337*(4.0/9.0);
                double t11397 = c*t11310*t11353*t11354*2.0;
                double t11398 = t11395+t11397-c*t11310*t11321*t11352-c*t11312*t11321*t11319*(2.0/3.0);
                double t11399 = t11332*t11315*t11335*t11398;
                double t11400 = t11345*t11319*t11346;
                double t11401 = c*t11322*t11337*(1.0/1.8E1);
                double t11402 = t11342*t11353*t11354*2.0;
                double t11403 = t11320*t11322*t11350*t11351*t11349*(1.0/1.8E1);
                double t11404 = c*t11320*t11312*t11321*t11317*t11319*(2.0/3.0);
                double t11405 = t11401+t11402+t11403+t11404-t11321*t11342*t11352-c*t11320*t11322*t11317*t11337*(4.0/9.0);
                double t11406 = c*t11331*t11325*t11317*t11337*t11357*(4.0/9.0);
                double t11407 = 1.0/(t11320*t11320*t11320);
                double t11408 = c*t11312*t11315*t11317*t11407*t11345*(1.0/3.0);
                double t11409 = c*t11360*t11328*t11337*t11357*t11359*(2.0/9.0);
                double t11410 = t11400+t11406+t11408+t11409-t11315*t11405*t11346-c*t11331*t11337*t11357*t11358*(2.0/9.0)-t11331*t11350*t11351*t11325*t11357*t11349*(1.0/1.8E1);
                double t11411 = EcP_2*EcP_3*t11410*t11341;
                double t11412 = EcP_3*c*t11331*t11337*t11358*(4.0/9.0);
                double t11413 = EcP_3*t11331*t11350*t11351*t11325*t11349*(1.0/9.0);
                double t11414 = t11411+t11412+t11413+t11394+t11399-t11332*t11315*t11334*t11369*(1.0/3.0)-EcP_3*c*t11331*t11325*t11317*t11337*(8.0/9.0)-EcP_3*c*t11360*t11328*t11337*t11359*(4.0/9.0);
                double t11415 = EcP_1*t11414;
                double t11416 = pow(2.0,1.0/3.0);
                double t11417 = t11416*2.0;
                double t11418 = t11417-2.0;
                double t11419 = two_13*2.0;
                double t11420 = t11419-2.0;
                double t11421 = 1.0/t11420;
                double t11423 = EcF_2*EcF_2;
                double t11424 = EcF_2*EcF_3;
                double t11425 = EcF_4+t11423+t11424;
                double t11426 = 1.0/t11425;
                double t11427 = t11332*t11315*t11334*t11335;
                double t11428 = t11315*t11345*t11346;
                double t11429 = c*t11312*t11331*t11325*t11317*t11357*(1.0/3.0);
                double t11430 = t11428+t11429;
                double t11431 = EcP_2*EcP_3*t11430*t11341;
                double t11432 = t11431+t11427-EcP_3*c*t11312*t11331*t11325*t11317*(2.0/3.0);
                v_rho_a_rho_a[Q] += scale * (EcP_1*t11432*-2.0+t11309*(t11415-t11421*t11418*(t11415-EcF_1*(t11332*t11335*t11362*(c*t11337*t11365*(4.0/9.0)-c*t11310*t11374*t11367+c*t11310*t11375*t11376*2.0-c*t11312*t11364*t11367*(2.0/3.0))+t11332*t11335*t11364*t11368-t11332*t11362*t11368*t11369*(1.0/3.0)+EcF_2*EcF_3*t11426*(-t11362*t11371*(c*t11337*t11365*(1.0/1.8E1)-t11372*t11374*t11367+t11372*t11375*t11376*2.0-c*t11370*t11317*t11337*t11365*(4.0/9.0)+t11350*t11351*t11370*t11365*t11349*(1.0/1.8E1)+c*t11312*t11370*t11317*t11364*t11367*(2.0/3.0))+t11371*t11390*t11364-c*t11380*t11337*t11391*t11387*(2.0/9.0)+c*t11317*t11380*t11381*t11337*t11387*(4.0/9.0)+c*t11380*t11337*t11392*t11384*t11393*(2.0/9.0)-t11350*t11351*t11380*t11381*t11349*t11387*(1.0/1.8E1)+c*t11312*1.0/(t11370*t11370*t11370)*t11317*t11362*t11390*(1.0/3.0))+EcF_3*c*t11337*t11391*t11387*(4.0/9.0)-EcF_3*c*t11317*t11381*t11337*t11387*(8.0/9.0)-EcF_3*c*t11337*t11392*t11384*t11393*(4.0/9.0)+EcF_3*t11350*t11351*t11381*t11349*t11387*(1.0/9.0))))+t11421*t11418*(EcP_1*t11432-EcF_1*(t11332*t11335*t11362*t11368+EcF_2*EcF_3*t11426*(t11362*t11371*t11390+c*t11312*t11317*t11380*t11381*t11387*(1.0/3.0))-EcF_3*c*t11312*t11317*t11381*t11387*(2.0/3.0)))*2.0);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t11434 = rho_a+rho_b;
                double t11435 = 1.0/pow(t11434,1.0/3.0);
                double t11436 = c*t11435;
                double t11437 = 1.0/pow(t11434,4.0/3.0);
                double t11438 = sqrt(t11436);
                double t11439 = EcP_3*t11438;
                double t11440 = EcP_4+t11436+t11439;
                double t11441 = c*t11437*(1.0/3.0);
                double t11442 = 1.0/sqrt(t11436);
                double t11443 = EcP_3*c*t11442*t11437*(1.0/6.0);
                double t11444 = t11441+t11443;
                double t11445 = EcP_2-t11438;
                double t11446 = 1.0/(t11440*t11440);
                double t11447 = 1.0/t11440;
                double t11448 = t11438*2.0;
                double t11449 = EcP_3+t11448;
                double t11450 = 1.0/(t11449*t11449);
                double t11451 = EcP_4*4.0;
                double t11452 = EcP_3*EcP_3;
                double t11453 = t11451-t11452;
                double t11454 = t11450*t11453;
                double t11455 = t11454+1.0;
                double t11456 = 1.0/t11455;
                double t11457 = 1.0/c;
                double t11458 = c*t11437*t11447*(1.0/3.0);
                double t11461 = c*t11435*t11444*t11446;
                double t11459 = -t11461+t11458;
                double t11460 = pow(t11434,1.0/3.0);
                double t11462 = 1.0/pow(t11434,7.0/3.0);
                double t11463 = EcP_2*EcP_2;
                double t11464 = EcP_2*EcP_3;
                double t11465 = EcP_4+t11463+t11464;
                double t11466 = 1.0/t11465;
                double t11467 = t11445*t11445;
                double t11468 = t11444*t11446*t11467;
                double t11469 = c*t11442*t11445*t11437*t11447*(1.0/3.0);
                double t11470 = t11468+t11469;
                double t11471 = 1.0/(t11445*t11445);
                double t11472 = c*t11462*(4.0/9.0);
                double t11473 = EcP_3*c*t11442*t11462*(2.0/9.0);
                double t11474 = c*c;
                double t11475 = 1.0/pow(t11434,8.0/3.0);
                double t11476 = 1.0/pow(t11436,3.0/2.0);
                double t11521 = EcP_3*t11474*t11475*t11476*(1.0/3.6E1);
                double t11477 = -t11521+t11472+t11473;
                double t11478 = t11444*t11444;
                double t11479 = 1.0/(t11440*t11440*t11440);
                double t11480 = EcP_2*4.0;
                double t11481 = EcP_3*2.0;
                double t11482 = t11480+t11481;
                double t11483 = 1.0/(t11449*t11449*t11449);
                double t11484 = 1.0/(t11449*t11449*t11449*t11449*t11449);
                double t11485 = 1.0/(t11455*t11455);
                double t11486 = EcF_3*t11438;
                double t11487 = EcF_4+t11436+t11486;
                double t11488 = EcF_3*c*t11442*t11437*(1.0/6.0);
                double t11489 = t11441+t11488;
                double t11490 = 1.0/t11487;
                double t11491 = c*t11490*t11437*(1.0/3.0);
                double t11492 = 1.0/(t11487*t11487);
                double t11547 = c*t11435*t11492*t11489;
                double t11493 = t11491-t11547;
                double t11494 = 1.0/pow(t11434,2.0/3.0);
                double t11495 = EcF_2-t11438;
                double t11496 = 1.0/(t11495*t11495);
                double t11497 = t11495*t11495;
                double t11498 = EcF_3*c*t11442*t11462*(2.0/9.0);
                double t11499 = t11472+t11498-EcF_3*t11474*t11475*t11476*(1.0/3.6E1);
                double t11500 = t11489*t11489;
                double t11501 = 1.0/(t11487*t11487*t11487);
                double t11502 = EcF_3+t11448;
                double t11503 = EcF_2*4.0;
                double t11504 = EcF_3*2.0;
                double t11505 = t11503+t11504;
                double t11506 = 1.0/(t11502*t11502);
                double t11507 = EcF_4*4.0;
                double t11508 = EcF_3*EcF_3;
                double t11509 = t11507-t11508;
                double t11510 = t11506*t11509;
                double t11511 = t11510+1.0;
                double t11512 = 1.0/t11511;
                double t11513 = t11492*t11497*t11489;
                double t11514 = c*t11442*t11490*t11437*t11495*(1.0/3.0);
                double t11515 = t11513+t11514;
                double t11516 = 1.0/(t11502*t11502*t11502);
                double t11517 = 1.0/(t11502*t11502*t11502*t11502*t11502);
                double t11518 = 1.0/(t11511*t11511);
                double t11519 = t11460*t11444*t11457*t11459;
                double t11520 = c*t11462*t11447*(4.0/9.0);
                double t11522 = c*t11435*t11478*t11479*2.0;
                double t11523 = t11520+t11522-c*t11444*t11437*t11446*(2.0/3.0)-c*t11435*t11446*t11477;
                double t11524 = t11440*t11460*t11523*t11457;
                double t11525 = t11470*t11444*t11471;
                double t11526 = c*t11462*t11447*(1.0/1.8E1);
                double t11527 = t11467*t11478*t11479*2.0;
                double t11528 = t11445*t11447*t11474*t11475*t11476*(1.0/1.8E1);
                double t11529 = c*t11442*t11444*t11445*t11437*t11446*(2.0/3.0);
                double t11530 = t11526+t11527+t11528+t11529-t11446*t11467*t11477-c*t11442*t11462*t11445*t11447*(4.0/9.0);
                double t11531 = c*t11450*t11442*t11462*t11482*t11456*(4.0/9.0);
                double t11532 = 1.0/(t11445*t11445*t11445);
                double t11533 = c*t11440*t11442*t11532*t11470*t11437*(1.0/3.0);
                double t11534 = c*t11453*t11462*t11482*t11484*t11485*(2.0/9.0);
                double t11535 = t11531+t11533+t11525+t11534-t11440*t11530*t11471-c*t11462*t11482*t11456*t11483*(2.0/9.0)-t11450*t11482*t11456*t11474*t11475*t11476*(1.0/1.8E1);
                double t11536 = EcP_2*EcP_3*t11535*t11466;
                double t11537 = EcP_3*c*t11462*t11456*t11483*(4.0/9.0);
                double t11538 = EcP_3*t11450*t11456*t11474*t11475*t11476*(1.0/9.0);
                double t11539 = t11524+t11536+t11519+t11537+t11538-t11440*t11457*t11494*t11459*(1.0/3.0)-EcP_3*c*t11450*t11442*t11462*t11456*(8.0/9.0)-EcP_3*c*t11453*t11462*t11484*t11485*(4.0/9.0);
                double t11540 = EcP_1*t11539;
                double t11541 = pow(2.0,1.0/3.0);
                double t11542 = t11541*2.0;
                double t11543 = t11542-2.0;
                double t11544 = two_13*2.0;
                double t11545 = t11544-2.0;
                double t11546 = 1.0/t11545;
                double t11548 = EcF_2*EcF_2;
                double t11549 = EcF_2*EcF_3;
                double t11550 = EcF_4+t11548+t11549;
                double t11551 = 1.0/t11550;
                double t11552 = t11440*t11460*t11457*t11459;
                double t11553 = t11440*t11470*t11471;
                double t11554 = c*t11450*t11442*t11437*t11482*t11456*(1.0/3.0);
                double t11555 = t11553+t11554;
                double t11556 = EcP_2*EcP_3*t11555*t11466;
                double t11557 = t11552+t11556-EcP_3*c*t11450*t11442*t11437*t11456*(2.0/3.0);
                v_rho_a_rho_b[Q] += scale * (EcP_1*t11557*-2.0+t11434*(t11540-t11543*t11546*(t11540-EcF_1*(t11460*t11457*t11487*(c*t11462*t11490*(4.0/9.0)+c*t11500*t11501*t11435*2.0-c*t11435*t11492*t11499-c*t11437*t11492*t11489*(2.0/3.0))+t11460*t11457*t11493*t11489-t11457*t11493*t11494*t11487*(1.0/3.0)+EcF_2*EcF_3*t11551*(-t11487*t11496*(c*t11462*t11490*(1.0/1.8E1)+t11500*t11501*t11497*2.0-t11492*t11497*t11499-c*t11442*t11462*t11490*t11495*(4.0/9.0)+t11490*t11474*t11475*t11476*t11495*(1.0/1.8E1)+c*t11442*t11437*t11492*t11495*t11489*(2.0/3.0))+t11515*t11496*t11489-c*t11512*t11505*t11462*t11516*(2.0/9.0)+c*t11512*t11442*t11505*t11506*t11462*(4.0/9.0)+c*t11505*t11462*t11517*t11509*t11518*(2.0/9.0)-t11512*t11505*t11506*t11474*t11475*t11476*(1.0/1.8E1)+c*t11442*t11515*t11437*1.0/(t11495*t11495*t11495)*t11487*(1.0/3.0))+EcF_3*c*t11512*t11462*t11516*(4.0/9.0)-EcF_3*c*t11512*t11442*t11506*t11462*(8.0/9.0)-EcF_3*c*t11462*t11517*t11509*t11518*(4.0/9.0)+EcF_3*t11512*t11506*t11474*t11475*t11476*(1.0/9.0))))+t11543*t11546*(EcP_1*t11557-EcF_1*(t11460*t11457*t11493*t11487+EcF_2*EcF_3*t11551*(t11515*t11487*t11496+c*t11512*t11442*t11505*t11506*t11437*(1.0/3.0))-EcF_3*c*t11512*t11442*t11506*t11437*(2.0/3.0)))*2.0);
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t11559 = rho_a+rho_b;
                double t11560 = 1.0/pow(t11559,1.0/3.0);
                double t11561 = c*t11560;
                double t11562 = 1.0/pow(t11559,4.0/3.0);
                double t11563 = sqrt(t11561);
                double t11564 = EcP_3*t11563;
                double t11565 = EcP_4+t11561+t11564;
                double t11566 = c*t11562*(1.0/3.0);
                double t11567 = 1.0/sqrt(t11561);
                double t11568 = EcP_3*c*t11562*t11567*(1.0/6.0);
                double t11569 = t11566+t11568;
                double t11570 = EcP_2-t11563;
                double t11571 = 1.0/(t11565*t11565);
                double t11572 = 1.0/t11565;
                double t11573 = t11563*2.0;
                double t11574 = EcP_3+t11573;
                double t11575 = 1.0/(t11574*t11574);
                double t11576 = EcP_4*4.0;
                double t11577 = EcP_3*EcP_3;
                double t11578 = t11576-t11577;
                double t11579 = t11575*t11578;
                double t11580 = t11579+1.0;
                double t11581 = 1.0/t11580;
                double t11582 = 1.0/c;
                double t11583 = c*t11562*t11572*(1.0/3.0);
                double t11586 = c*t11560*t11571*t11569;
                double t11584 = t11583-t11586;
                double t11585 = pow(t11559,1.0/3.0);
                double t11587 = 1.0/pow(t11559,7.0/3.0);
                double t11588 = EcP_2*EcP_2;
                double t11589 = EcP_2*EcP_3;
                double t11590 = EcP_4+t11588+t11589;
                double t11591 = 1.0/t11590;
                double t11592 = t11570*t11570;
                double t11593 = t11571*t11592*t11569;
                double t11594 = c*t11570*t11562*t11572*t11567*(1.0/3.0);
                double t11595 = t11593+t11594;
                double t11596 = 1.0/(t11570*t11570);
                double t11597 = c*t11587*(4.0/9.0);
                double t11598 = EcP_3*c*t11567*t11587*(2.0/9.0);
                double t11599 = c*c;
                double t11600 = 1.0/pow(t11559,8.0/3.0);
                double t11601 = 1.0/pow(t11561,3.0/2.0);
                double t11646 = EcP_3*t11600*t11601*t11599*(1.0/3.6E1);
                double t11602 = -t11646+t11597+t11598;
                double t11603 = t11569*t11569;
                double t11604 = 1.0/(t11565*t11565*t11565);
                double t11605 = EcP_2*4.0;
                double t11606 = EcP_3*2.0;
                double t11607 = t11605+t11606;
                double t11608 = 1.0/(t11574*t11574*t11574);
                double t11609 = 1.0/(t11574*t11574*t11574*t11574*t11574);
                double t11610 = 1.0/(t11580*t11580);
                double t11611 = EcF_3*t11563;
                double t11612 = EcF_4+t11611+t11561;
                double t11613 = EcF_3*c*t11562*t11567*(1.0/6.0);
                double t11614 = t11613+t11566;
                double t11615 = 1.0/t11612;
                double t11616 = c*t11615*t11562*(1.0/3.0);
                double t11617 = 1.0/(t11612*t11612);
                double t11672 = c*t11560*t11614*t11617;
                double t11618 = t11616-t11672;
                double t11619 = 1.0/pow(t11559,2.0/3.0);
                double t11620 = EcF_2-t11563;
                double t11621 = 1.0/(t11620*t11620);
                double t11622 = t11620*t11620;
                double t11623 = EcF_3*c*t11567*t11587*(2.0/9.0);
                double t11624 = t11623+t11597-EcF_3*t11600*t11601*t11599*(1.0/3.6E1);
                double t11625 = t11614*t11614;
                double t11626 = 1.0/(t11612*t11612*t11612);
                double t11627 = EcF_3+t11573;
                double t11628 = EcF_2*4.0;
                double t11629 = EcF_3*2.0;
                double t11630 = t11628+t11629;
                double t11631 = 1.0/(t11627*t11627);
                double t11632 = EcF_4*4.0;
                double t11633 = EcF_3*EcF_3;
                double t11634 = t11632-t11633;
                double t11635 = t11631*t11634;
                double t11636 = t11635+1.0;
                double t11637 = 1.0/t11636;
                double t11638 = t11622*t11614*t11617;
                double t11639 = c*t11620*t11615*t11562*t11567*(1.0/3.0);
                double t11640 = t11638+t11639;
                double t11641 = 1.0/(t11627*t11627*t11627);
                double t11642 = 1.0/(t11627*t11627*t11627*t11627*t11627);
                double t11643 = 1.0/(t11636*t11636);
                double t11644 = t11582*t11584*t11585*t11569;
                double t11645 = c*t11572*t11587*(4.0/9.0);
                double t11647 = c*t11603*t11604*t11560*2.0;
                double t11648 = t11645+t11647-c*t11602*t11560*t11571-c*t11562*t11571*t11569*(2.0/3.0);
                double t11649 = t11582*t11565*t11585*t11648;
                double t11650 = t11595*t11569*t11596;
                double t11651 = c*t11572*t11587*(1.0/1.8E1);
                double t11652 = t11603*t11604*t11592*2.0;
                double t11653 = t11600*t11601*t11570*t11572*t11599*(1.0/1.8E1);
                double t11654 = c*t11570*t11562*t11571*t11567*t11569*(2.0/3.0);
                double t11655 = t11651+t11652+t11653+t11654-t11602*t11571*t11592-c*t11570*t11572*t11567*t11587*(4.0/9.0);
                double t11656 = c*t11607*t11581*t11575*t11567*t11587*(4.0/9.0);
                double t11657 = 1.0/(t11570*t11570*t11570);
                double t11658 = c*t11562*t11565*t11567*t11657*t11595*(1.0/3.0);
                double t11659 = c*t11610*t11607*t11609*t11578*t11587*(2.0/9.0);
                double t11660 = t11650+t11656+t11658+t11659-t11565*t11655*t11596-c*t11607*t11581*t11608*t11587*(2.0/9.0)-t11600*t11601*t11607*t11581*t11575*t11599*(1.0/1.8E1);
                double t11661 = EcP_2*EcP_3*t11660*t11591;
                double t11662 = EcP_3*c*t11581*t11608*t11587*(4.0/9.0);
                double t11663 = EcP_3*t11600*t11601*t11581*t11575*t11599*(1.0/9.0);
                double t11664 = t11661+t11644+t11662+t11663+t11649-t11582*t11565*t11619*t11584*(1.0/3.0)-EcP_3*c*t11610*t11609*t11578*t11587*(4.0/9.0)-EcP_3*c*t11581*t11575*t11567*t11587*(8.0/9.0);
                double t11665 = EcP_1*t11664;
                double t11666 = pow(2.0,1.0/3.0);
                double t11667 = t11666*2.0;
                double t11668 = t11667-2.0;
                double t11669 = two_13*2.0;
                double t11670 = t11669-2.0;
                double t11671 = 1.0/t11670;
                double t11673 = EcF_2*EcF_2;
                double t11674 = EcF_2*EcF_3;
                double t11675 = EcF_4+t11673+t11674;
                double t11676 = 1.0/t11675;
                double t11677 = t11582*t11565*t11584*t11585;
                double t11678 = t11565*t11595*t11596;
                double t11679 = c*t11562*t11607*t11581*t11575*t11567*(1.0/3.0);
                double t11680 = t11678+t11679;
                double t11681 = EcP_2*EcP_3*t11680*t11591;
                double t11682 = t11681+t11677-EcP_3*c*t11562*t11581*t11575*t11567*(2.0/3.0);
                v_rho_b_rho_b[Q] += scale * (EcP_1*t11682*-2.0+t11559*(t11665-t11671*t11668*(t11665-EcF_1*(t11612*t11582*t11585*(c*t11615*t11587*(4.0/9.0)-c*t11560*t11624*t11617+c*t11560*t11625*t11626*2.0-c*t11614*t11562*t11617*(2.0/3.0))-t11612*t11582*t11618*t11619*(1.0/3.0)+t11614*t11582*t11618*t11585+EcF_2*EcF_3*t11676*(-t11612*t11621*(c*t11615*t11587*(1.0/1.8E1)-t11622*t11624*t11617+t11622*t11625*t11626*2.0-c*t11620*t11615*t11567*t11587*(4.0/9.0)+t11600*t11601*t11620*t11615*t11599*(1.0/1.8E1)+c*t11620*t11614*t11562*t11617*t11567*(2.0/3.0))+t11621*t11640*t11614-c*t11630*t11641*t11637*t11587*(2.0/9.0)+c*t11630*t11642*t11634*t11643*t11587*(2.0/9.0)+c*t11630*t11631*t11637*t11567*t11587*(4.0/9.0)-t11600*t11601*t11630*t11631*t11637*t11599*(1.0/1.8E1)+c*1.0/(t11620*t11620*t11620)*t11612*t11640*t11562*t11567*(1.0/3.0))+EcF_3*c*t11641*t11637*t11587*(4.0/9.0)-EcF_3*c*t11642*t11634*t11643*t11587*(4.0/9.0)-EcF_3*c*t11631*t11637*t11567*t11587*(8.0/9.0)+EcF_3*t11600*t11601*t11631*t11637*t11599*(1.0/9.0))))+t11671*t11668*(EcP_1*t11682-EcF_1*(t11612*t11582*t11618*t11585+EcF_2*EcF_3*t11676*(t11612*t11621*t11640+c*t11630*t11631*t11562*t11637*t11567*(1.0/3.0))-EcF_3*c*t11631*t11562*t11637*t11567*(2.0/3.0)))*2.0);
            }
            
        } else {
            // v
            if (deriv >= 0) {
                double t9113 = rho_a+rho_b;
                double t9114 = 1.0/pow(t9113,1.0/3.0);
                double t9115 = c*t9114;
                double t9116 = sqrt(t9115);
                double t9117 = EcP_4*4.0;
                double t9118 = EcP_3*EcP_3;
                double t9119 = t9117-t9118;
                double t9120 = EcP_2-t9116;
                double t9121 = EcP_3*t9116;
                double t9122 = EcP_4+t9121+t9115;
                double t9123 = 1.0/t9122;
                double t9124 = sqrt(t9119);
                double t9125 = t9116*2.0;
                double t9126 = EcP_3+t9125;
                double t9127 = 1.0/t9126;
                double t9128 = t9124*t9127;
                double t9129 = atan(t9128);
                double t9130 = 1.0/sqrt(t9119);
                double t9131 = rho_a-rho_b;
                double t9132 = t9131*t9131;
                double t9133 = EcF_4*4.0;
                double t9134 = EcF_3*EcF_3;
                double t9135 = t9133-t9134;
                double t9136 = EcF_2-t9116;
                double t9137 = EcF_3*t9116;
                double t9138 = EcF_4+t9115+t9137;
                double t9139 = 1.0/t9138;
                double t9140 = sqrt(t9135);
                double t9141 = EcF_3+t9125;
                double t9142 = 1.0/t9141;
                double t9143 = t9140*t9142;
                double t9144 = atan(t9143);
                double t9145 = 1.0/sqrt(t9135);
                double t9146 = c*t9114*t9123;
                double t9147 = log(t9146);
                double t9148 = EcP_3*t9130*t9129*2.0;
                double t9149 = t9120*t9120;
                double t9150 = t9123*t9149;
                double t9151 = log(t9150);
                double t9152 = EcP_2*4.0;
                double t9153 = EcP_3*2.0;
                double t9154 = t9152+t9153;
                double t9155 = t9130*t9154*t9129;
                double t9156 = t9151+t9155;
                double t9157 = EcP_2*EcP_2;
                double t9158 = EcP_2*EcP_3;
                double t9159 = EcP_4+t9157+t9158;
                double t9160 = 1.0/t9159;
                double t9161 = t9147+t9148-EcP_2*EcP_3*t9160*t9156;
                double t9162 = EcP_1*t9161;
                double t9163 = Ac_4*4.0;
                double t9164 = Ac_3*Ac_3;
                double t9165 = t9163-t9164;
                double t9166 = Ac_2-t9116;
                double t9167 = Ac_3*t9116;
                double t9168 = Ac_4+t9115+t9167;
                double t9169 = 1.0/t9168;
                double t9170 = sqrt(t9165);
                double t9171 = Ac_3+t9125;
                double t9172 = 1.0/t9171;
                double t9173 = t9170*t9172;
                double t9174 = atan(t9173);
                double t9175 = 1.0/sqrt(t9165);
                double t9176 = 1.0/t9113;
                double t9177 = t9131*t9176;
                double t9178 = c*t9114*t9169;
                double t9179 = log(t9178);
                double t9180 = Ac_3*t9174*t9175*2.0;
                double t9181 = t9166*t9166;
                double t9182 = t9181*t9169;
                double t9183 = log(t9182);
                double t9184 = Ac_2*4.0;
                double t9185 = Ac_3*2.0;
                double t9186 = t9184+t9185;
                double t9187 = t9174*t9175*t9186;
                double t9188 = t9183+t9187;
                double t9189 = Ac_2*Ac_2;
                double t9190 = Ac_2*Ac_3;
                double t9191 = Ac_4+t9190+t9189;
                double t9192 = 1.0/t9191;
                double t9193 = t9180+t9179-Ac_2*Ac_3*t9192*t9188;
                v[Q] += scale * (t9113*(t9162-(Ac_1*t9193*(1.0/(t9113*t9113*t9113*t9113)*(t9132*t9132)*((d2fz0*(t9162-EcF_1*(log(c*t9114*t9139)+EcF_3*t9144*t9145*2.0-(EcF_2*EcF_3*(log((t9136*t9136)*t9139)+t9144*t9145*(EcF_2*4.0+EcF_3*2.0)))/(EcF_4+EcF_2*EcF_2+EcF_2*EcF_3))))/(Ac_1*t9193)+1.0)-1.0)*(pow(t9177+1.0,4.0/3.0)+pow(-t9177+1.0,4.0/3.0)-2.0))/(d2fz0*(two_13*2.0-2.0))));
            }
            
            // v_rho_a
            if (deriv >= 1) {
                double t9195 = rho_a+rho_b;
                double t9196 = 1.0/pow(t9195,1.0/3.0);
                double t9197 = c*t9196;
                double t9198 = sqrt(t9197);
                double t9199 = EcP_4*4.0;
                double t9200 = EcP_3*EcP_3;
                double t9201 = t9199-t9200;
                double t9202 = EcP_2-t9198;
                double t9203 = EcP_3*t9198;
                double t9204 = EcP_4+t9197+t9203;
                double t9205 = 1.0/t9204;
                double t9206 = sqrt(t9201);
                double t9207 = t9198*2.0;
                double t9208 = EcP_3+t9207;
                double t9209 = 1.0/t9208;
                double t9210 = t9206*t9209;
                double t9211 = atan(t9210);
                double t9212 = 1.0/sqrt(t9201);
                double t9213 = 1.0/pow(t9195,4.0/3.0);
                double t9214 = c*t9213*(1.0/3.0);
                double t9215 = 1.0/sqrt(t9197);
                double t9216 = EcP_3*c*t9213*t9215*(1.0/6.0);
                double t9217 = t9214+t9216;
                double t9218 = t9202*t9202;
                double t9219 = 1.0/(t9204*t9204);
                double t9220 = EcP_2*4.0;
                double t9221 = EcP_3*2.0;
                double t9222 = t9220+t9221;
                double t9223 = 1.0/(t9208*t9208);
                double t9224 = EcP_2*EcP_2;
                double t9225 = EcP_2*EcP_3;
                double t9226 = EcP_4+t9224+t9225;
                double t9227 = 1.0/t9226;
                double t9228 = t9201*t9223;
                double t9229 = t9228+1.0;
                double t9230 = 1.0/t9229;
                double t9231 = 1.0/t9195;
                double t9232 = rho_a-rho_b;
                double t9233 = t9231*t9232;
                double t9234 = Ac_4*4.0;
                double t9235 = Ac_3*Ac_3;
                double t9236 = t9234-t9235;
                double t9237 = Ac_2-t9198;
                double t9238 = Ac_3*t9198;
                double t9239 = Ac_4+t9197+t9238;
                double t9240 = 1.0/t9239;
                double t9241 = sqrt(t9236);
                double t9242 = Ac_3+t9207;
                double t9243 = 1.0/t9242;
                double t9244 = t9241*t9243;
                double t9245 = atan(t9244);
                double t9246 = 1.0/sqrt(t9236);
                double t9247 = 1.0/c;
                double t9248 = EcF_3*t9198;
                double t9249 = EcF_4+t9197+t9248;
                double t9250 = pow(t9195,1.0/3.0);
                double t9251 = EcF_3*c*t9213*t9215*(1.0/6.0);
                double t9252 = t9214+t9251;
                double t9253 = EcF_2-t9198;
                double t9254 = 1.0/(t9249*t9249);
                double t9255 = 1.0/t9249;
                double t9256 = EcF_3+t9207;
                double t9257 = 1.0/(t9256*t9256);
                double t9258 = EcF_4*4.0;
                double t9259 = EcF_3*EcF_3;
                double t9260 = t9258-t9259;
                double t9261 = t9260*t9257;
                double t9262 = t9261+1.0;
                double t9263 = 1.0/t9262;
                double t9264 = c*t9213*t9205*(1.0/3.0);
                double t9265 = t9264-c*t9196*t9217*t9219;
                double t9266 = t9204*t9250*t9247*t9265;
                double t9267 = t9217*t9218*t9219;
                double t9268 = c*t9202*t9213*t9205*t9215*(1.0/3.0);
                double t9269 = t9267+t9268;
                double t9270 = 1.0/(t9202*t9202);
                double t9271 = t9204*t9270*t9269;
                double t9272 = c*t9230*t9213*t9222*t9223*t9215*(1.0/3.0);
                double t9273 = t9271+t9272;
                double t9274 = EcP_2*EcP_3*t9227*t9273;
                double t9275 = t9274+t9266-EcP_3*c*t9230*t9213*t9223*t9215*(2.0/3.0);
                double t9276 = EcP_1*t9275;
                double t9277 = c*t9196*t9240;
                double t9278 = log(t9277);
                double t9279 = Ac_3*t9245*t9246*2.0;
                double t9280 = t9237*t9237;
                double t9281 = t9240*t9280;
                double t9282 = log(t9281);
                double t9283 = Ac_2*4.0;
                double t9284 = Ac_3*2.0;
                double t9285 = t9283+t9284;
                double t9286 = t9245*t9246*t9285;
                double t9287 = t9282+t9286;
                double t9288 = Ac_2*Ac_2;
                double t9289 = Ac_2*Ac_3;
                double t9290 = Ac_4+t9288+t9289;
                double t9291 = 1.0/t9290;
                double t9316 = Ac_2*Ac_3*t9291*t9287;
                double t9292 = t9278+t9279-t9316;
                double t9293 = 1.0/Ac_1;
                double t9294 = t9253*t9253;
                double t9295 = sqrt(t9260);
                double t9296 = 1.0/t9256;
                double t9297 = t9295*t9296;
                double t9298 = atan(t9297);
                double t9299 = 1.0/sqrt(t9260);
                double t9300 = EcF_2*4.0;
                double t9301 = EcF_3*2.0;
                double t9302 = t9300+t9301;
                double t9303 = EcF_2*EcF_2;
                double t9304 = EcF_2*EcF_3;
                double t9305 = EcF_4+t9303+t9304;
                double t9306 = 1.0/t9305;
                double t9307 = c*t9196*t9205;
                double t9308 = log(t9307);
                double t9309 = EcP_3*t9211*t9212*2.0;
                double t9310 = t9205*t9218;
                double t9311 = log(t9310);
                double t9312 = t9211*t9212*t9222;
                double t9313 = t9311+t9312;
                double t9334 = EcP_2*EcP_3*t9227*t9313;
                double t9314 = -t9334+t9308+t9309;
                double t9315 = EcP_1*t9314;
                double t9317 = Ac_3*c*t9213*t9215*(1.0/6.0);
                double t9318 = t9214+t9317;
                double t9319 = 1.0/(t9239*t9239);
                double t9320 = 1.0/(t9242*t9242);
                double t9321 = t9236*t9320;
                double t9322 = t9321+1.0;
                double t9323 = 1.0/t9322;
                double t9324 = t9232*t9232;
                double t9325 = 1.0/(t9195*t9195*t9195*t9195);
                double t9326 = c*t9196*t9255;
                double t9327 = log(t9326);
                double t9328 = EcF_3*t9298*t9299*2.0;
                double t9329 = t9255*t9294;
                double t9330 = log(t9329);
                double t9331 = t9298*t9299*t9302;
                double t9332 = t9330+t9331;
                double t9338 = EcF_2*EcF_3*t9332*t9306;
                double t9333 = t9327+t9328-t9338;
                double t9335 = 1.0/t9292;
                double t9339 = EcF_1*t9333;
                double t9336 = t9315-t9339;
                double t9337 = t9324*t9324;
                double t9340 = d2fz0*t9293*t9335*t9336;
                double t9341 = t9340+1.0;
                double t9342 = 1.0/d2fz0;
                double t9343 = two_13*2.0;
                double t9344 = t9343-2.0;
                double t9345 = 1.0/t9344;
                double t9346 = t9233+1.0;
                double t9347 = pow(t9346,4.0/3.0);
                double t9348 = -t9233+1.0;
                double t9349 = pow(t9348,4.0/3.0);
                double t9350 = t9347+t9349-2.0;
                double t9351 = c*t9213*t9240*(1.0/3.0);
                double t9352 = t9351-c*t9196*t9318*t9319;
                double t9353 = t9250*t9247*t9239*t9352;
                double t9354 = t9280*t9318*t9319;
                double t9355 = c*t9213*t9240*t9215*t9237*(1.0/3.0);
                double t9356 = t9354+t9355;
                double t9357 = 1.0/(t9237*t9237);
                double t9358 = t9239*t9356*t9357;
                double t9359 = c*t9213*t9215*t9285*t9320*t9323*(1.0/3.0);
                double t9360 = t9358+t9359;
                double t9361 = Ac_2*Ac_3*t9291*t9360;
                double t9362 = t9361+t9353-Ac_3*c*t9213*t9215*t9320*t9323*(2.0/3.0);
                double t9363 = 1.0/(t9195*t9195);
                double t9364 = t9231-t9232*t9363;
                double t9365 = t9341*t9325*t9337;
                double t9366 = t9365-1.0;
                v_rho_a[Q] += scale * (t9315-t9195*(t9276-Ac_1*t9292*t9350*t9342*t9345*(1.0/(t9195*t9195*t9195*t9195*t9195)*t9341*t9337*4.0+t9325*t9337*(d2fz0*t9293*t9335*(t9276-EcF_1*(EcF_2*EcF_3*t9306*(1.0/(t9253*t9253)*t9249*(t9252*t9254*t9294+c*t9213*t9215*t9253*t9255*(1.0/3.0))+c*t9213*t9215*t9263*t9257*t9302*(1.0/3.0))+t9250*t9247*t9249*(c*t9213*t9255*(1.0/3.0)-c*t9196*t9252*t9254)-EcF_3*c*t9213*t9215*t9263*t9257*(2.0/3.0)))-d2fz0*1.0/(t9292*t9292)*t9293*t9362*t9336)-t9232*t9341*t9324*t9325*4.0)+Ac_1*t9292*t9342*t9345*t9366*(pow(t9346,1.0/3.0)*t9364*(4.0/3.0)-t9364*pow(t9348,1.0/3.0)*(4.0/3.0))-Ac_1*t9350*t9342*t9362*t9345*t9366)-Ac_1*t9292*t9350*t9342*t9345*t9366);
            }
            
            // v_rho_b
            if (deriv >= 1) {
                double t9368 = rho_a+rho_b;
                double t9369 = 1.0/pow(t9368,1.0/3.0);
                double t9370 = c*t9369;
                double t9371 = sqrt(t9370);
                double t9372 = EcP_4*4.0;
                double t9373 = EcP_3*EcP_3;
                double t9374 = t9372-t9373;
                double t9375 = EcP_2-t9371;
                double t9376 = EcP_3*t9371;
                double t9377 = EcP_4+t9370+t9376;
                double t9378 = 1.0/t9377;
                double t9379 = sqrt(t9374);
                double t9380 = t9371*2.0;
                double t9381 = EcP_3+t9380;
                double t9382 = 1.0/t9381;
                double t9383 = t9382*t9379;
                double t9384 = atan(t9383);
                double t9385 = 1.0/sqrt(t9374);
                double t9386 = 1.0/pow(t9368,4.0/3.0);
                double t9387 = c*t9386*(1.0/3.0);
                double t9388 = 1.0/sqrt(t9370);
                double t9389 = EcP_3*c*t9386*t9388*(1.0/6.0);
                double t9390 = t9387+t9389;
                double t9391 = t9375*t9375;
                double t9392 = 1.0/(t9377*t9377);
                double t9393 = EcP_2*4.0;
                double t9394 = EcP_3*2.0;
                double t9395 = t9393+t9394;
                double t9396 = 1.0/(t9381*t9381);
                double t9397 = EcP_2*EcP_2;
                double t9398 = EcP_2*EcP_3;
                double t9399 = EcP_4+t9397+t9398;
                double t9400 = 1.0/t9399;
                double t9401 = t9374*t9396;
                double t9402 = t9401+1.0;
                double t9403 = 1.0/t9402;
                double t9404 = 1.0/t9368;
                double t9405 = rho_a-rho_b;
                double t9406 = t9404*t9405;
                double t9407 = Ac_4*4.0;
                double t9408 = Ac_3*Ac_3;
                double t9409 = t9407-t9408;
                double t9410 = Ac_2-t9371;
                double t9411 = Ac_3*t9371;
                double t9412 = Ac_4+t9370+t9411;
                double t9413 = 1.0/t9412;
                double t9414 = sqrt(t9409);
                double t9415 = Ac_3+t9380;
                double t9416 = 1.0/t9415;
                double t9417 = t9414*t9416;
                double t9418 = atan(t9417);
                double t9419 = 1.0/sqrt(t9409);
                double t9420 = 1.0/c;
                double t9421 = EcF_3*t9371;
                double t9422 = EcF_4+t9370+t9421;
                double t9423 = pow(t9368,1.0/3.0);
                double t9424 = EcF_3*c*t9386*t9388*(1.0/6.0);
                double t9425 = t9387+t9424;
                double t9426 = EcF_2-t9371;
                double t9427 = 1.0/(t9422*t9422);
                double t9428 = 1.0/t9422;
                double t9429 = EcF_3+t9380;
                double t9430 = 1.0/(t9429*t9429);
                double t9431 = EcF_4*4.0;
                double t9432 = EcF_3*EcF_3;
                double t9433 = t9431-t9432;
                double t9434 = t9430*t9433;
                double t9435 = t9434+1.0;
                double t9436 = 1.0/t9435;
                double t9437 = c*t9386*t9378*(1.0/3.0);
                double t9438 = t9437-c*t9390*t9392*t9369;
                double t9439 = t9377*t9420*t9423*t9438;
                double t9440 = t9390*t9391*t9392;
                double t9441 = c*t9375*t9386*t9378*t9388*(1.0/3.0);
                double t9442 = t9440+t9441;
                double t9443 = 1.0/(t9375*t9375);
                double t9444 = t9377*t9442*t9443;
                double t9445 = c*t9386*t9395*t9396*t9388*t9403*(1.0/3.0);
                double t9446 = t9444+t9445;
                double t9447 = EcP_2*EcP_3*t9400*t9446;
                double t9448 = t9447+t9439-EcP_3*c*t9386*t9396*t9388*t9403*(2.0/3.0);
                double t9449 = EcP_1*t9448;
                double t9450 = c*t9369*t9413;
                double t9451 = log(t9450);
                double t9452 = Ac_3*t9418*t9419*2.0;
                double t9453 = t9410*t9410;
                double t9454 = t9413*t9453;
                double t9455 = log(t9454);
                double t9456 = Ac_2*4.0;
                double t9457 = Ac_3*2.0;
                double t9458 = t9456+t9457;
                double t9459 = t9418*t9419*t9458;
                double t9460 = t9455+t9459;
                double t9461 = Ac_2*Ac_2;
                double t9462 = Ac_2*Ac_3;
                double t9463 = Ac_4+t9461+t9462;
                double t9464 = 1.0/t9463;
                double t9489 = Ac_2*Ac_3*t9460*t9464;
                double t9465 = t9451+t9452-t9489;
                double t9466 = 1.0/Ac_1;
                double t9467 = t9426*t9426;
                double t9468 = sqrt(t9433);
                double t9469 = 1.0/t9429;
                double t9470 = t9468*t9469;
                double t9471 = atan(t9470);
                double t9472 = 1.0/sqrt(t9433);
                double t9473 = EcF_2*4.0;
                double t9474 = EcF_3*2.0;
                double t9475 = t9473+t9474;
                double t9476 = EcF_2*EcF_2;
                double t9477 = EcF_2*EcF_3;
                double t9478 = EcF_4+t9476+t9477;
                double t9479 = 1.0/t9478;
                double t9480 = c*t9369*t9378;
                double t9481 = log(t9480);
                double t9482 = EcP_3*t9384*t9385*2.0;
                double t9483 = t9391*t9378;
                double t9484 = log(t9483);
                double t9485 = t9384*t9385*t9395;
                double t9486 = t9484+t9485;
                double t9507 = EcP_2*EcP_3*t9400*t9486;
                double t9487 = t9481+t9482-t9507;
                double t9488 = EcP_1*t9487;
                double t9490 = Ac_3*c*t9386*t9388*(1.0/6.0);
                double t9491 = t9387+t9490;
                double t9492 = 1.0/(t9412*t9412);
                double t9493 = 1.0/(t9415*t9415);
                double t9494 = t9409*t9493;
                double t9495 = t9494+1.0;
                double t9496 = 1.0/t9495;
                double t9497 = t9405*t9405;
                double t9498 = 1.0/(t9368*t9368*t9368*t9368);
                double t9499 = c*t9369*t9428;
                double t9500 = log(t9499);
                double t9501 = EcF_3*t9471*t9472*2.0;
                double t9502 = t9428*t9467;
                double t9503 = log(t9502);
                double t9504 = t9471*t9472*t9475;
                double t9505 = t9503+t9504;
                double t9511 = EcF_2*EcF_3*t9479*t9505;
                double t9506 = t9500+t9501-t9511;
                double t9508 = 1.0/t9465;
                double t9512 = EcF_1*t9506;
                double t9509 = t9488-t9512;
                double t9510 = t9497*t9497;
                double t9513 = d2fz0*t9466*t9508*t9509;
                double t9514 = t9513+1.0;
                double t9515 = 1.0/d2fz0;
                double t9516 = two_13*2.0;
                double t9517 = t9516-2.0;
                double t9518 = 1.0/t9517;
                double t9519 = t9406+1.0;
                double t9520 = pow(t9519,4.0/3.0);
                double t9521 = -t9406+1.0;
                double t9522 = pow(t9521,4.0/3.0);
                double t9523 = t9520+t9522-2.0;
                double t9524 = c*t9386*t9413*(1.0/3.0);
                double t9525 = t9524-c*t9369*t9491*t9492;
                double t9526 = t9420*t9412*t9423*t9525;
                double t9527 = t9453*t9491*t9492;
                double t9528 = c*t9386*t9388*t9410*t9413*(1.0/3.0);
                double t9529 = t9527+t9528;
                double t9530 = 1.0/(t9410*t9410);
                double t9531 = t9412*t9530*t9529;
                double t9532 = c*t9386*t9388*t9493*t9458*t9496*(1.0/3.0);
                double t9533 = t9531+t9532;
                double t9534 = Ac_2*Ac_3*t9464*t9533;
                double t9535 = t9534+t9526-Ac_3*c*t9386*t9388*t9493*t9496*(2.0/3.0);
                double t9536 = 1.0/(t9368*t9368);
                double t9537 = t9405*t9536;
                double t9538 = t9404+t9537;
                double t9539 = t9498*t9510*t9514;
                double t9540 = t9539-1.0;
                v_rho_b[Q] += scale * (t9488-t9368*(t9449-Ac_1*t9465*t9523*t9515*t9518*(1.0/(t9368*t9368*t9368*t9368*t9368)*t9510*t9514*4.0+t9498*t9510*(d2fz0*t9466*t9508*(t9449-EcF_1*(EcF_2*EcF_3*t9479*(t9422*1.0/(t9426*t9426)*(t9425*t9427*t9467+c*t9386*t9388*t9426*t9428*(1.0/3.0))+c*t9386*t9388*t9430*t9436*t9475*(1.0/3.0))+t9420*t9422*t9423*(c*t9386*t9428*(1.0/3.0)-c*t9369*t9425*t9427)-EcF_3*c*t9386*t9388*t9430*t9436*(2.0/3.0)))-d2fz0*1.0/(t9465*t9465)*t9466*t9535*t9509)+t9405*t9497*t9498*t9514*4.0)+Ac_1*t9465*t9540*t9515*t9518*(pow(t9521,1.0/3.0)*t9538*(4.0/3.0)-pow(t9519,1.0/3.0)*t9538*(4.0/3.0))-Ac_1*t9540*t9523*t9515*t9535*t9518)-Ac_1*t9465*t9540*t9523*t9515*t9518);
            }
            
            // v_rho_a_rho_a
            if (deriv >= 2) {
                double t9547 = rho_a+rho_b;
                double t9548 = 1.0/pow(t9547,1.0/3.0);
                double t9549 = c*t9548;
                double t9550 = 1.0/pow(t9547,4.0/3.0);
                double t9551 = sqrt(t9549);
                double t9552 = EcP_3*t9551;
                double t9553 = EcP_4+t9552+t9549;
                double t9554 = c*t9550*(1.0/3.0);
                double t9555 = 1.0/sqrt(t9549);
                double t9556 = EcP_3*c*t9550*t9555*(1.0/6.0);
                double t9557 = t9554+t9556;
                double t9558 = 1.0/c;
                double t9559 = 1.0/t9553;
                double t9560 = c*t9550*t9559*(1.0/3.0);
                double t9561 = 1.0/(t9553*t9553);
                double t9695 = c*t9561*t9548*t9557;
                double t9562 = t9560-t9695;
                double t9563 = pow(t9547,1.0/3.0);
                double t9564 = 1.0/pow(t9547,7.0/3.0);
                double t9565 = EcP_2-t9551;
                double t9566 = 1.0/(t9565*t9565);
                double t9567 = t9565*t9565;
                double t9568 = c*t9564*(4.0/9.0);
                double t9569 = EcP_3*c*t9555*t9564*(2.0/9.0);
                double t9570 = c*c;
                double t9571 = 1.0/pow(t9547,8.0/3.0);
                double t9572 = 1.0/pow(t9549,3.0/2.0);
                double t9698 = EcP_3*t9570*t9571*t9572*(1.0/3.6E1);
                double t9573 = t9568+t9569-t9698;
                double t9574 = t9557*t9557;
                double t9575 = 1.0/(t9553*t9553*t9553);
                double t9576 = t9551*2.0;
                double t9577 = EcP_3+t9576;
                double t9578 = EcP_2*4.0;
                double t9579 = EcP_3*2.0;
                double t9580 = t9578+t9579;
                double t9581 = 1.0/(t9577*t9577);
                double t9582 = EcP_4*4.0;
                double t9583 = EcP_3*EcP_3;
                double t9584 = t9582-t9583;
                double t9585 = t9581*t9584;
                double t9586 = t9585+1.0;
                double t9587 = 1.0/t9586;
                double t9588 = t9561*t9557*t9567;
                double t9589 = c*t9550*t9555*t9565*t9559*(1.0/3.0);
                double t9590 = t9588+t9589;
                double t9591 = 1.0/(t9577*t9577*t9577);
                double t9592 = 1.0/(t9577*t9577*t9577*t9577*t9577);
                double t9593 = 1.0/(t9586*t9586);
                double t9594 = rho_a-rho_b;
                double t9595 = t9594*t9594;
                double t9596 = EcF_4*4.0;
                double t9597 = EcF_3*EcF_3;
                double t9598 = t9596-t9597;
                double t9599 = EcF_2-t9551;
                double t9600 = EcF_3*t9551;
                double t9601 = EcF_4+t9549+t9600;
                double t9602 = 1.0/t9601;
                double t9603 = sqrt(t9598);
                double t9604 = EcF_3+t9576;
                double t9605 = 1.0/t9604;
                double t9606 = t9603*t9605;
                double t9607 = atan(t9606);
                double t9608 = 1.0/sqrt(t9598);
                double t9609 = sqrt(t9584);
                double t9610 = 1.0/t9577;
                double t9611 = t9610*t9609;
                double t9612 = atan(t9611);
                double t9613 = 1.0/sqrt(t9584);
                double t9614 = EcP_2*EcP_2;
                double t9615 = EcP_2*EcP_3;
                double t9616 = EcP_4+t9614+t9615;
                double t9617 = 1.0/t9616;
                double t9618 = Ac_4*4.0;
                double t9619 = Ac_3*Ac_3;
                double t9620 = t9618-t9619;
                double t9621 = Ac_2-t9551;
                double t9622 = Ac_3*t9551;
                double t9623 = Ac_4+t9549+t9622;
                double t9624 = 1.0/t9623;
                double t9625 = sqrt(t9620);
                double t9626 = Ac_3+t9576;
                double t9627 = 1.0/t9626;
                double t9628 = t9625*t9627;
                double t9629 = atan(t9628);
                double t9630 = 1.0/sqrt(t9620);
                double t9631 = c*t9548*t9624;
                double t9632 = log(t9631);
                double t9633 = Ac_3*t9630*t9629*2.0;
                double t9634 = t9621*t9621;
                double t9635 = t9624*t9634;
                double t9636 = log(t9635);
                double t9637 = Ac_2*4.0;
                double t9638 = Ac_3*2.0;
                double t9639 = t9637+t9638;
                double t9640 = t9630*t9629*t9639;
                double t9641 = t9640+t9636;
                double t9642 = Ac_2*Ac_2;
                double t9643 = Ac_2*Ac_3;
                double t9644 = Ac_4+t9642+t9643;
                double t9645 = 1.0/t9644;
                double t9662 = Ac_2*Ac_3*t9641*t9645;
                double t9646 = t9632+t9633-t9662;
                double t9647 = 1.0/t9547;
                double t9648 = t9594*t9647;
                double t9649 = 1.0/(t9547*t9547);
                double t9650 = t9649*2.0;
                double t9651 = 1.0/(t9547*t9547*t9547);
                double t9652 = t9650-t9594*t9651*2.0;
                double t9653 = t9648+1.0;
                double t9656 = t9594*t9649;
                double t9654 = t9647-t9656;
                double t9655 = -t9648+1.0;
                double t9657 = t9654*t9654;
                double t9658 = 1.0/d2fz0;
                double t9659 = two_13*2.0;
                double t9660 = t9659-2.0;
                double t9661 = 1.0/t9660;
                double t9663 = 1.0/(t9547*t9547*t9547*t9547);
                double t9664 = t9595*t9595;
                double t9665 = 1.0/Ac_1;
                double t9666 = EcF_3*c*t9550*t9555*(1.0/6.0);
                double t9667 = t9554+t9666;
                double t9668 = c*t9550*t9602*(1.0/3.0);
                double t9669 = 1.0/(t9601*t9601);
                double t9719 = c*t9548*t9667*t9669;
                double t9670 = t9668-t9719;
                double t9671 = 1.0/pow(t9547,2.0/3.0);
                double t9672 = EcF_2*EcF_2;
                double t9673 = EcF_2*EcF_3;
                double t9674 = EcF_4+t9672+t9673;
                double t9675 = 1.0/t9674;
                double t9676 = t9599*t9599;
                double t9677 = 1.0/(t9599*t9599);
                double t9678 = EcF_3*c*t9555*t9564*(2.0/9.0);
                double t9679 = t9568+t9678-EcF_3*t9570*t9571*t9572*(1.0/3.6E1);
                double t9680 = t9667*t9667;
                double t9681 = 1.0/(t9601*t9601*t9601);
                double t9682 = EcF_2*4.0;
                double t9683 = EcF_3*2.0;
                double t9684 = t9682+t9683;
                double t9685 = 1.0/(t9604*t9604);
                double t9686 = t9598*t9685;
                double t9687 = t9686+1.0;
                double t9688 = 1.0/t9687;
                double t9689 = t9667*t9676*t9669;
                double t9690 = c*t9550*t9555*t9599*t9602*(1.0/3.0);
                double t9691 = t9690+t9689;
                double t9692 = 1.0/(t9604*t9604*t9604);
                double t9693 = 1.0/(t9604*t9604*t9604*t9604*t9604);
                double t9694 = 1.0/(t9687*t9687);
                double t9696 = t9562*t9563*t9557*t9558;
                double t9697 = c*t9564*t9559*(4.0/9.0);
                double t9699 = c*t9574*t9548*t9575*2.0;
                double t9700 = t9697+t9699-c*t9550*t9561*t9557*(2.0/3.0)-c*t9561*t9573*t9548;
                double t9701 = t9553*t9563*t9558*t9700;
                double t9702 = t9590*t9557*t9566;
                double t9703 = c*t9564*t9559*(1.0/1.8E1);
                double t9704 = t9574*t9575*t9567*2.0;
                double t9705 = t9570*t9571*t9572*t9565*t9559*(1.0/1.8E1);
                double t9706 = c*t9550*t9561*t9555*t9565*t9557*(2.0/3.0);
                double t9707 = t9703+t9704+t9705+t9706-t9561*t9573*t9567-c*t9555*t9564*t9565*t9559*(4.0/9.0);
                double t9708 = c*t9580*t9581*t9555*t9564*t9587*(4.0/9.0);
                double t9709 = 1.0/(t9565*t9565*t9565);
                double t9710 = c*t9550*t9553*t9590*t9555*t9709*(1.0/3.0);
                double t9711 = c*t9580*t9564*t9592*t9584*t9593*(2.0/9.0);
                double t9712 = t9710+t9702+t9711+t9708-t9553*t9566*t9707-c*t9580*t9564*t9591*t9587*(2.0/9.0)-t9570*t9571*t9580*t9572*t9581*t9587*(1.0/1.8E1);
                double t9713 = EcP_2*EcP_3*t9617*t9712;
                double t9714 = EcP_3*c*t9564*t9591*t9587*(4.0/9.0);
                double t9715 = EcP_3*t9570*t9571*t9572*t9581*t9587*(1.0/9.0);
                double t9716 = t9696+t9701+t9713+t9714+t9715-t9553*t9562*t9558*t9671*(1.0/3.0)-EcP_3*c*t9581*t9555*t9564*t9587*(8.0/9.0)-EcP_3*c*t9564*t9592*t9584*t9593*(4.0/9.0);
                double t9717 = EcP_1*t9716;
                double t9718 = 1.0/t9646;
                double t9720 = Ac_3*c*t9550*t9555*(1.0/6.0);
                double t9721 = t9554+t9720;
                double t9722 = 1.0/(t9623*t9623);
                double t9723 = 1.0/(t9626*t9626);
                double t9724 = t9620*t9723;
                double t9725 = t9724+1.0;
                double t9726 = 1.0/t9725;
                double t9727 = c*t9548*t9602;
                double t9728 = log(t9727);
                double t9729 = EcF_3*t9607*t9608*2.0;
                double t9730 = t9602*t9676;
                double t9731 = log(t9730);
                double t9732 = t9607*t9608*t9684;
                double t9733 = t9731+t9732;
                double t9757 = EcF_2*EcF_3*t9675*t9733;
                double t9734 = t9728+t9729-t9757;
                double t9735 = EcF_1*t9734;
                double t9736 = c*t9548*t9559;
                double t9737 = log(t9736);
                double t9738 = EcP_3*t9612*t9613*2.0;
                double t9739 = t9567*t9559;
                double t9740 = log(t9739);
                double t9741 = t9580*t9612*t9613;
                double t9742 = t9740+t9741;
                double t9758 = EcP_2*EcP_3*t9617*t9742;
                double t9743 = t9737+t9738-t9758;
                double t9759 = EcP_1*t9743;
                double t9744 = t9735-t9759;
                double t9745 = c*t9550*t9624*(1.0/3.0);
                double t9761 = c*t9548*t9721*t9722;
                double t9746 = -t9761+t9745;
                double t9747 = t9563*t9558*t9623*t9746;
                double t9748 = t9634*t9721*t9722;
                double t9749 = c*t9550*t9555*t9621*t9624*(1.0/3.0);
                double t9750 = t9748+t9749;
                double t9751 = 1.0/(t9621*t9621);
                double t9752 = t9623*t9750*t9751;
                double t9753 = c*t9550*t9555*t9639*t9723*t9726*(1.0/3.0);
                double t9754 = t9752+t9753;
                double t9755 = Ac_2*Ac_3*t9645*t9754;
                double t9783 = Ac_3*c*t9550*t9555*t9723*t9726*(2.0/3.0);
                double t9756 = t9755+t9747-t9783;
                double t9760 = 1.0/(t9646*t9646);
                double t9762 = Ac_3*c*t9555*t9564*(2.0/9.0);
                double t9806 = Ac_3*t9570*t9571*t9572*(1.0/3.6E1);
                double t9763 = t9568+t9762-t9806;
                double t9764 = t9721*t9721;
                double t9765 = 1.0/(t9623*t9623*t9623);
                double t9766 = 1.0/(t9626*t9626*t9626);
                double t9767 = 1.0/(t9626*t9626*t9626*t9626*t9626);
                double t9768 = 1.0/(t9725*t9725);
                double t9769 = t9563*t9558*t9601*t9670;
                double t9770 = t9601*t9691*t9677;
                double t9771 = c*t9550*t9555*t9684*t9685*t9688*(1.0/3.0);
                double t9772 = t9770+t9771;
                double t9773 = EcF_2*EcF_3*t9675*t9772;
                double t9784 = EcF_3*c*t9550*t9555*t9685*t9688*(2.0/3.0);
                double t9774 = t9773-t9784+t9769;
                double t9775 = EcF_1*t9774;
                double t9776 = t9553*t9562*t9563*t9558;
                double t9777 = t9553*t9590*t9566;
                double t9778 = c*t9550*t9580*t9581*t9555*t9587*(1.0/3.0);
                double t9779 = t9777+t9778;
                double t9780 = EcP_2*EcP_3*t9617*t9779;
                double t9785 = EcP_3*c*t9550*t9581*t9555*t9587*(2.0/3.0);
                double t9781 = t9780+t9776-t9785;
                double t9786 = EcP_1*t9781;
                double t9782 = t9775-t9786;
                double t9787 = d2fz0*t9665*t9718*t9782;
                double t9794 = d2fz0*t9665*t9760*t9744*t9756;
                double t9788 = -t9794+t9787;
                double t9789 = d2fz0*t9665*t9744*t9718;
                double t9790 = t9789-1.0;
                double t9791 = 1.0/(t9547*t9547*t9547*t9547*t9547);
                double t9792 = pow(t9653,1.0/3.0);
                double t9793 = pow(t9655,1.0/3.0);
                double t9795 = t9654*t9792*(4.0/3.0);
                double t9827 = t9654*t9793*(4.0/3.0);
                double t9796 = t9795-t9827;
                double t9797 = t9663*t9664*t9790;
                double t9798 = t9797+1.0;
                double t9799 = pow(t9653,4.0/3.0);
                double t9800 = pow(t9655,4.0/3.0);
                double t9801 = t9799+t9800-2.0;
                double t9802 = t9663*t9664*t9788;
                double t9803 = t9664*t9790*t9791*4.0;
                double t9826 = t9594*t9595*t9663*t9790*4.0;
                double t9804 = t9802+t9803-t9826;
                double t9805 = c*t9564*t9624*(4.0/9.0);
                double t9807 = c*t9548*t9764*t9765*2.0;
                double t9808 = t9805+t9807-c*t9550*t9721*t9722*(2.0/3.0)-c*t9548*t9722*t9763;
                double t9809 = t9563*t9558*t9623*t9808;
                double t9810 = t9721*t9750*t9751;
                double t9811 = c*t9564*t9624*(1.0/1.8E1);
                double t9812 = t9634*t9764*t9765*2.0;
                double t9813 = t9570*t9571*t9572*t9621*t9624*(1.0/1.8E1);
                double t9814 = c*t9550*t9555*t9621*t9721*t9722*(2.0/3.0);
                double t9815 = t9811+t9812+t9813+t9814-t9634*t9722*t9763-c*t9555*t9564*t9621*t9624*(4.0/9.0);
                double t9816 = c*t9555*t9564*t9639*t9723*t9726*(4.0/9.0);
                double t9817 = 1.0/(t9621*t9621*t9621);
                double t9818 = c*t9550*t9555*t9623*t9750*t9817*(1.0/3.0);
                double t9819 = c*t9564*t9620*t9639*t9767*t9768*(2.0/9.0);
                double t9820 = t9810+t9816+t9818+t9819-t9623*t9751*t9815-c*t9564*t9639*t9726*t9766*(2.0/9.0)-t9570*t9571*t9572*t9639*t9723*t9726*(1.0/1.8E1);
                double t9821 = Ac_2*Ac_3*t9645*t9820;
                double t9822 = t9563*t9558*t9721*t9746;
                double t9823 = Ac_3*c*t9564*t9726*t9766*(4.0/9.0);
                double t9824 = Ac_3*t9570*t9571*t9572*t9723*t9726*(1.0/9.0);
                double t9825 = t9821+t9822+t9823+t9824+t9809-t9558*t9623*t9671*t9746*(1.0/3.0)-Ac_3*c*t9555*t9564*t9723*t9726*(8.0/9.0)-Ac_3*c*t9564*t9620*t9767*t9768*(4.0/9.0);
                v_rho_a_rho_a[Q] += scale * (EcP_1*t9781*-2.0+t9547*(t9717+Ac_1*t9661*t9646*t9658*t9801*(1.0/(t9547*t9547*t9547*t9547*t9547*t9547)*t9664*t9790*2.0E1+t9595*t9663*t9790*1.2E1+t9664*t9791*t9788*8.0-t9663*t9664*(d2fz0*t9665*t9718*(t9717-EcF_1*(t9563*t9558*t9601*(c*t9564*t9602*(4.0/9.0)+c*t9548*t9680*t9681*2.0-c*t9550*t9667*t9669*(2.0/3.0)-c*t9548*t9669*t9679)+t9563*t9558*t9670*t9667-t9558*t9601*t9670*t9671*(1.0/3.0)+EcF_2*EcF_3*t9675*(-t9601*t9677*(c*t9564*t9602*(1.0/1.8E1)+t9680*t9681*t9676*2.0-t9676*t9669*t9679-c*t9555*t9564*t9599*t9602*(4.0/9.0)+t9570*t9571*t9572*t9599*t9602*(1.0/1.8E1)+c*t9550*t9555*t9599*t9667*t9669*(2.0/3.0))+t9691*t9667*t9677-c*t9564*t9692*t9684*t9688*(2.0/9.0)+c*t9555*t9564*t9684*t9685*t9688*(4.0/9.0)+c*t9564*t9598*t9684*t9693*t9694*(2.0/9.0)-t9570*t9571*t9572*t9684*t9685*t9688*(1.0/1.8E1)+c*t9550*t9555*1.0/(t9599*t9599*t9599)*t9601*t9691*(1.0/3.0))+EcF_3*c*t9564*t9692*t9688*(4.0/9.0)-EcF_3*c*t9555*t9564*t9685*t9688*(8.0/9.0)-EcF_3*c*t9564*t9598*t9693*t9694*(4.0/9.0)+EcF_3*t9570*t9571*t9572*t9685*t9688*(1.0/9.0)))-d2fz0*1.0/(t9646*t9646*t9646)*t9665*t9744*(t9756*t9756)*2.0+d2fz0*t9665*t9760*t9782*t9756*2.0+d2fz0*t9665*t9760*t9744*t9825)-t9594*t9595*t9663*t9788*8.0-t9594*t9595*t9790*t9791*3.2E1)+Ac_1*t9661*t9646*t9658*t9798*(t9652*t9792*(-4.0/3.0)+t9652*t9793*(4.0/3.0)+1.0/pow(t9653,2.0/3.0)*t9657*(4.0/9.0)+1.0/pow(t9655,2.0/3.0)*t9657*(4.0/9.0))-Ac_1*t9661*t9646*t9658*t9796*t9804*2.0-Ac_1*t9661*t9658*t9756*t9796*t9798*2.0+Ac_1*t9661*t9658*t9756*t9801*t9804*2.0+Ac_1*t9661*t9658*t9798*t9801*t9825)+Ac_1*t9661*t9646*t9658*t9796*t9798*2.0-Ac_1*t9661*t9646*t9658*t9801*t9804*2.0-Ac_1*t9661*t9658*t9756*t9798*t9801*2.0);
            }
            
            // v_rho_a_rho_b
            if (deriv >= 2) {
                double t9829 = rho_a+rho_b;
                double t9830 = 1.0/pow(t9829,1.0/3.0);
                double t9831 = c*t9830;
                double t9832 = 1.0/pow(t9829,4.0/3.0);
                double t9833 = sqrt(t9831);
                double t9834 = EcP_3*t9833;
                double t9835 = EcP_4+t9831+t9834;
                double t9836 = c*t9832*(1.0/3.0);
                double t9837 = 1.0/sqrt(t9831);
                double t9838 = EcP_3*c*t9832*t9837*(1.0/6.0);
                double t9839 = t9836+t9838;
                double t9840 = EcP_2-t9833;
                double t9841 = 1.0/(t9835*t9835);
                double t9842 = 1.0/t9835;
                double t9843 = t9833*2.0;
                double t9844 = EcP_3+t9843;
                double t9845 = 1.0/(t9844*t9844);
                double t9846 = EcP_4*4.0;
                double t9847 = EcP_3*EcP_3;
                double t9848 = t9846-t9847;
                double t9849 = t9845*t9848;
                double t9850 = t9849+1.0;
                double t9851 = 1.0/t9850;
                double t9852 = 1.0/c;
                double t9853 = c*t9832*t9842*(1.0/3.0);
                double t9856 = c*t9830*t9841*t9839;
                double t9854 = t9853-t9856;
                double t9855 = pow(t9829,1.0/3.0);
                double t9857 = 1.0/pow(t9829,7.0/3.0);
                double t9858 = EcP_2*EcP_2;
                double t9859 = EcP_2*EcP_3;
                double t9860 = EcP_4+t9858+t9859;
                double t9861 = 1.0/t9860;
                double t9862 = t9840*t9840;
                double t9863 = t9841*t9862*t9839;
                double t9864 = c*t9840*t9832*t9842*t9837*(1.0/3.0);
                double t9865 = t9863+t9864;
                double t9866 = 1.0/(t9840*t9840);
                double t9867 = c*t9857*(4.0/9.0);
                double t9868 = EcP_3*c*t9837*t9857*(2.0/9.0);
                double t9869 = c*c;
                double t9870 = 1.0/pow(t9829,8.0/3.0);
                double t9871 = 1.0/pow(t9831,3.0/2.0);
                double t10061 = EcP_3*t9870*t9871*t9869*(1.0/3.6E1);
                double t9872 = -t10061+t9867+t9868;
                double t9873 = t9839*t9839;
                double t9874 = 1.0/(t9835*t9835*t9835);
                double t9875 = EcP_2*4.0;
                double t9876 = EcP_3*2.0;
                double t9877 = t9875+t9876;
                double t9878 = 1.0/(t9844*t9844*t9844);
                double t9879 = 1.0/(t9844*t9844*t9844*t9844*t9844);
                double t9880 = 1.0/(t9850*t9850);
                double t9881 = rho_a-rho_b;
                double t9882 = t9881*t9881;
                double t9883 = EcF_4*4.0;
                double t9884 = EcF_3*EcF_3;
                double t9885 = t9883-t9884;
                double t9886 = EcF_2-t9833;
                double t9887 = EcF_3*t9833;
                double t9888 = EcF_4+t9831+t9887;
                double t9889 = 1.0/t9888;
                double t9890 = sqrt(t9885);
                double t9891 = EcF_3+t9843;
                double t9892 = 1.0/t9891;
                double t9893 = t9890*t9892;
                double t9894 = atan(t9893);
                double t9895 = 1.0/sqrt(t9885);
                double t9896 = sqrt(t9848);
                double t9897 = 1.0/t9844;
                double t9898 = t9896*t9897;
                double t9899 = atan(t9898);
                double t9900 = 1.0/sqrt(t9848);
                double t9901 = Ac_4*4.0;
                double t9902 = Ac_3*Ac_3;
                double t9903 = t9901-t9902;
                double t9904 = Ac_2-t9833;
                double t9905 = Ac_3*t9833;
                double t9906 = Ac_4+t9831+t9905;
                double t9907 = 1.0/t9906;
                double t9908 = sqrt(t9903);
                double t9909 = Ac_3+t9843;
                double t9910 = 1.0/t9909;
                double t9911 = t9910*t9908;
                double t9912 = atan(t9911);
                double t9913 = 1.0/sqrt(t9903);
                double t9914 = c*t9830*t9907;
                double t9915 = log(t9914);
                double t9916 = Ac_3*t9912*t9913*2.0;
                double t9917 = t9904*t9904;
                double t9918 = t9907*t9917;
                double t9919 = log(t9918);
                double t9920 = Ac_2*4.0;
                double t9921 = Ac_3*2.0;
                double t9922 = t9920+t9921;
                double t9923 = t9912*t9913*t9922;
                double t9924 = t9923+t9919;
                double t9925 = Ac_2*Ac_2;
                double t9926 = Ac_2*Ac_3;
                double t9927 = Ac_4+t9925+t9926;
                double t9928 = 1.0/t9927;
                double t9945 = Ac_2*Ac_3*t9924*t9928;
                double t9929 = t9915+t9916-t9945;
                double t9930 = 1.0/t9829;
                double t9931 = t9881*t9930;
                double t9932 = 1.0/(t9829*t9829*t9829);
                double t9933 = t9931+1.0;
                double t9934 = 1.0/(t9829*t9829);
                double t9935 = t9881*t9934;
                double t9936 = -t9931+1.0;
                double t9937 = t9930+t9935;
                double t9938 = t9930-t9935;
                double t9939 = 1.0/d2fz0;
                double t9940 = two_13*2.0;
                double t9941 = t9940-2.0;
                double t9942 = 1.0/t9941;
                double t9943 = pow(t9933,1.0/3.0);
                double t9944 = pow(t9936,1.0/3.0);
                double t9946 = 1.0/(t9829*t9829*t9829*t9829);
                double t9947 = 1.0/Ac_1;
                double t9948 = EcF_3*c*t9832*t9837*(1.0/6.0);
                double t9949 = t9836+t9948;
                double t9950 = t9886*t9886;
                double t9951 = 1.0/(t9888*t9888);
                double t9952 = EcF_2*4.0;
                double t9953 = EcF_3*2.0;
                double t9954 = t9952+t9953;
                double t9955 = 1.0/(t9891*t9891);
                double t9956 = EcF_2*EcF_2;
                double t9957 = EcF_2*EcF_3;
                double t9958 = EcF_4+t9956+t9957;
                double t9959 = 1.0/t9958;
                double t9960 = t9885*t9955;
                double t9961 = t9960+1.0;
                double t9962 = 1.0/t9961;
                double t9963 = t9852*t9835*t9854*t9855;
                double t9964 = t9835*t9865*t9866;
                double t9965 = c*t9832*t9851*t9845*t9837*t9877*(1.0/3.0);
                double t9966 = t9964+t9965;
                double t9967 = EcP_2*EcP_3*t9861*t9966;
                double t10014 = EcP_3*c*t9832*t9851*t9845*t9837*(2.0/3.0);
                double t9968 = -t10014+t9963+t9967;
                double t9969 = 1.0/t9929;
                double t9970 = c*t9830*t9889;
                double t9971 = log(t9970);
                double t9972 = EcF_3*t9894*t9895*2.0;
                double t9973 = t9889*t9950;
                double t9974 = log(t9973);
                double t9975 = t9894*t9895*t9954;
                double t9976 = t9974+t9975;
                double t9996 = EcF_2*EcF_3*t9976*t9959;
                double t9977 = t9971+t9972-t9996;
                double t9978 = EcF_1*t9977;
                double t9979 = c*t9830*t9842;
                double t9980 = log(t9979);
                double t9981 = EcP_3*t9899*t9900*2.0;
                double t9982 = t9842*t9862;
                double t9983 = log(t9982);
                double t9984 = t9877*t9899*t9900;
                double t9985 = t9983+t9984;
                double t9997 = EcP_2*EcP_3*t9861*t9985;
                double t9986 = t9980+t9981-t9997;
                double t9998 = EcP_1*t9986;
                double t9987 = t9978-t9998;
                double t9988 = Ac_3*c*t9832*t9837*(1.0/6.0);
                double t9989 = t9836+t9988;
                double t9990 = 1.0/(t9906*t9906);
                double t9991 = 1.0/(t9909*t9909);
                double t9992 = t9903*t9991;
                double t9993 = t9992+1.0;
                double t9994 = 1.0/t9993;
                double t9995 = t9882*t9882;
                double t9999 = d2fz0*t9947*t9969*t9987;
                double t10000 = t9999-1.0;
                double t10001 = c*t9832*t9889*(1.0/3.0);
                double t10042 = c*t9830*t9951*t9949;
                double t10002 = t10001-t10042;
                double t10003 = t10002*t9852*t9855*t9888;
                double t10004 = t9950*t9951*t9949;
                double t10005 = c*t9832*t9837*t9886*t9889*(1.0/3.0);
                double t10006 = t10004+t10005;
                double t10007 = 1.0/(t9886*t9886);
                double t10008 = t10006*t10007*t9888;
                double t10009 = c*t9832*t9837*t9962*t9954*t9955*(1.0/3.0);
                double t10010 = t10008+t10009;
                double t10011 = EcF_2*EcF_3*t10010*t9959;
                double t10043 = EcF_3*c*t9832*t9837*t9962*t9955*(2.0/3.0);
                double t10012 = t10011+t10003-t10043;
                double t10013 = EcF_1*t10012;
                double t10044 = EcP_1*t9968;
                double t10015 = t10013-t10044;
                double t10016 = d2fz0*t10015*t9947*t9969;
                double t10017 = 1.0/(t9929*t9929);
                double t10018 = c*t9832*t9907*(1.0/3.0);
                double t10038 = c*t9830*t9990*t9989;
                double t10019 = t10018-t10038;
                double t10020 = t10019*t9852*t9855*t9906;
                double t10021 = t9917*t9990*t9989;
                double t10022 = c*t9832*t9837*t9904*t9907*(1.0/3.0);
                double t10023 = t10021+t10022;
                double t10024 = 1.0/(t9904*t9904);
                double t10025 = t10023*t10024*t9906;
                double t10026 = c*t9832*t9837*t9922*t9991*t9994*(1.0/3.0);
                double t10027 = t10025+t10026;
                double t10028 = Ac_2*Ac_3*t10027*t9928;
                double t10039 = Ac_3*c*t9832*t9837*t9991*t9994*(2.0/3.0);
                double t10029 = t10020+t10028-t10039;
                double t10045 = d2fz0*t10017*t10029*t9947*t9987;
                double t10030 = t10016-t10045;
                double t10031 = t10030*t9946*t9995;
                double t10032 = 1.0/(t9829*t9829*t9829*t9829*t9829);
                double t10033 = t10000*t10032*t9995*4.0;
                double t10034 = t9943*t9937*(4.0/3.0);
                double t10110 = t9944*t9937*(4.0/3.0);
                double t10035 = -t10110+t10034;
                double t10036 = t10000*t9946*t9995;
                double t10037 = t10036+1.0;
                double t10040 = t9943*t9938*(4.0/3.0);
                double t10111 = t9944*t9938*(4.0/3.0);
                double t10041 = -t10111+t10040;
                double t10046 = t10000*t9881*t9882*t9946*4.0;
                double t10047 = pow(t9933,4.0/3.0);
                double t10048 = pow(t9936,4.0/3.0);
                double t10049 = t10047+t10048-2.0;
                double t10050 = t10031+t10033+t10046;
                double t10051 = 1.0/pow(t9829,2.0/3.0);
                double t10052 = EcF_3*c*t9837*t9857*(2.0/9.0);
                double t10053 = t10052+t9867-EcF_3*t9870*t9871*t9869*(1.0/3.6E1);
                double t10054 = t9949*t9949;
                double t10055 = 1.0/(t9888*t9888*t9888);
                double t10056 = 1.0/(t9891*t9891*t9891);
                double t10057 = 1.0/(t9891*t9891*t9891*t9891*t9891);
                double t10058 = 1.0/(t9961*t9961);
                double t10059 = t9852*t9854*t9855*t9839;
                double t10060 = c*t9842*t9857*(4.0/9.0);
                double t10062 = c*t9830*t9873*t9874*2.0;
                double t10063 = t10060+t10062-c*t9830*t9841*t9872-c*t9832*t9841*t9839*(2.0/3.0);
                double t10064 = t10063*t9852*t9835*t9855;
                double t10065 = t9865*t9839*t9866;
                double t10066 = c*t9842*t9857*(1.0/1.8E1);
                double t10067 = t9862*t9873*t9874*2.0;
                double t10068 = t9840*t9842*t9870*t9871*t9869*(1.0/1.8E1);
                double t10069 = c*t9840*t9832*t9841*t9837*t9839*(2.0/3.0);
                double t10070 = t10066+t10067+t10068+t10069-t9841*t9862*t9872-c*t9840*t9842*t9837*t9857*(4.0/9.0);
                double t10071 = c*t9851*t9845*t9837*t9857*t9877*(4.0/9.0);
                double t10072 = 1.0/(t9840*t9840*t9840);
                double t10073 = c*t10072*t9832*t9835*t9837*t9865*(1.0/3.0);
                double t10074 = c*t9880*t9848*t9857*t9877*t9879*(2.0/9.0);
                double t10075 = t10071+t10073+t10065+t10074-t10070*t9835*t9866-c*t9851*t9857*t9877*t9878*(2.0/9.0)-t9851*t9870*t9871*t9845*t9877*t9869*(1.0/1.8E1);
                double t10076 = EcP_2*EcP_3*t10075*t9861;
                double t10077 = EcP_3*c*t9851*t9857*t9878*(4.0/9.0);
                double t10078 = EcP_3*t9851*t9870*t9871*t9845*t9869*(1.0/9.0);
                double t10079 = t10064+t10076+t10059+t10077+t10078-t10051*t9852*t9835*t9854*(1.0/3.0)-EcP_3*c*t9851*t9845*t9837*t9857*(8.0/9.0)-EcP_3*c*t9880*t9848*t9857*t9879*(4.0/9.0);
                double t10080 = EcP_1*t10079;
                double t10081 = Ac_3*c*t9837*t9857*(2.0/9.0);
                double t10089 = Ac_3*t9870*t9871*t9869*(1.0/3.6E1);
                double t10082 = t10081-t10089+t9867;
                double t10083 = t9989*t9989;
                double t10084 = 1.0/(t9906*t9906*t9906);
                double t10085 = 1.0/(t9909*t9909*t9909);
                double t10086 = 1.0/(t9909*t9909*t9909*t9909*t9909);
                double t10087 = 1.0/(t9993*t9993);
                double t10088 = c*t9857*t9907*(4.0/9.0);
                double t10090 = c*t10083*t10084*t9830*2.0;
                double t10091 = t10090+t10088-c*t10082*t9830*t9990-c*t9832*t9990*t9989*(2.0/3.0);
                double t10092 = t10091*t9852*t9855*t9906;
                double t10093 = t10023*t10024*t9989;
                double t10094 = c*t9857*t9907*(1.0/1.8E1);
                double t10095 = t10083*t10084*t9917*2.0;
                double t10096 = t9870*t9871*t9869*t9904*t9907*(1.0/1.8E1);
                double t10097 = c*t9832*t9837*t9904*t9990*t9989*(2.0/3.0);
                double t10098 = t10094+t10095+t10096+t10097-t10082*t9917*t9990-c*t9837*t9857*t9904*t9907*(4.0/9.0);
                double t10099 = c*t9837*t9857*t9922*t9991*t9994*(4.0/9.0);
                double t10100 = 1.0/(t9904*t9904*t9904);
                double t10101 = c*t10100*t10023*t9832*t9837*t9906*(1.0/3.0);
                double t10102 = c*t10086*t10087*t9857*t9903*t9922*(2.0/9.0);
                double t10103 = t10101+t10102+t10093+t10099-t10024*t10098*t9906-c*t10085*t9857*t9922*t9994*(2.0/9.0)-t9870*t9871*t9869*t9922*t9991*t9994*(1.0/1.8E1);
                double t10104 = Ac_2*Ac_3*t10103*t9928;
                double t10105 = t10019*t9852*t9855*t9989;
                double t10106 = Ac_3*c*t10085*t9857*t9994*(4.0/9.0);
                double t10107 = Ac_3*t9870*t9871*t9869*t9991*t9994*(1.0/9.0);
                double t10108 = t10104+t10105+t10106+t10107+t10092-t10051*t10019*t9852*t9906*(1.0/3.0)-Ac_3*c*t10086*t10087*t9857*t9903*(4.0/9.0)-Ac_3*c*t9837*t9857*t9991*t9994*(8.0/9.0);
                double t10109 = t10031+t10033-t10046;
                v_rho_a_rho_b[Q] += scale * (EcP_1*t9968*-2.0+t9829*(t10080+Ac_1*t10049*t9942*t9929*t9939*(t10000*1.0/(t9829*t9829*t9829*t9829*t9829*t9829)*t9995*2.0E1+t10030*t10032*t9995*8.0-t10000*t9882*t9946*1.2E1-t9946*t9995*(d2fz0*t9947*t9969*(t10080-EcF_1*(t9852*t9855*t9888*(c*t9857*t9889*(4.0/9.0)+c*t10054*t10055*t9830*2.0-c*t10053*t9830*t9951-c*t9832*t9951*t9949*(2.0/3.0))-t10002*t10051*t9852*t9888*(1.0/3.0)+t10002*t9852*t9855*t9949+EcF_2*EcF_3*t9959*(-t10007*t9888*(c*t9857*t9889*(1.0/1.8E1)+t10054*t10055*t9950*2.0-t10053*t9950*t9951-c*t9837*t9857*t9886*t9889*(4.0/9.0)+t9870*t9871*t9886*t9869*t9889*(1.0/1.8E1)+c*t9832*t9837*t9886*t9951*t9949*(2.0/3.0))+t10006*t10007*t9949-c*t10056*t9857*t9962*t9954*(2.0/9.0)+c*t10057*t10058*t9857*t9885*t9954*(2.0/9.0)+c*t9837*t9857*t9962*t9954*t9955*(4.0/9.0)-t9870*t9871*t9869*t9962*t9954*t9955*(1.0/1.8E1)+c*t10006*t9832*t9837*1.0/(t9886*t9886*t9886)*t9888*(1.0/3.0))+EcF_3*c*t10056*t9857*t9962*(4.0/9.0)-EcF_3*c*t10057*t10058*t9857*t9885*(4.0/9.0)-EcF_3*c*t9837*t9857*t9962*t9955*(8.0/9.0)+EcF_3*t9870*t9871*t9869*t9962*t9955*(1.0/9.0)))-d2fz0*(t10029*t10029)*1.0/(t9929*t9929*t9929)*t9947*t9987*2.0+d2fz0*t10015*t10017*t10029*t9947*2.0+d2fz0*t10017*t10108*t9947*t9987))-Ac_1*t10037*t9942*t9929*t9939*(1.0/pow(t9933,2.0/3.0)*t9937*t9938*(4.0/9.0)+1.0/pow(t9936,2.0/3.0)*t9937*t9938*(4.0/9.0)-t9881*t9932*t9943*(8.0/3.0)+t9881*t9932*t9944*(8.0/3.0))-Ac_1*t10041*t10037*t10029*t9942*t9939+Ac_1*t10050*t10029*t10049*t9942*t9939+Ac_1*t10035*t10037*t10029*t9942*t9939+Ac_1*t10108*t10037*t10049*t9942*t9939+Ac_1*t10109*t10029*t10049*t9942*t9939-Ac_1*t10041*t10050*t9942*t9929*t9939+Ac_1*t10035*t9942*t9929*t9939*(t10031+t10033-t10000*t9881*t9882*t9946*4.0))-Ac_1*t10037*t10029*t10049*t9942*t9939*2.0+Ac_1*t10041*t10037*t9942*t9929*t9939-Ac_1*t10050*t10049*t9942*t9929*t9939-Ac_1*t10035*t10037*t9942*t9929*t9939-Ac_1*t10109*t10049*t9942*t9929*t9939);
            }
            
            // v_rho_b_rho_b
            if (deriv >= 2) {
                double t10113 = rho_a+rho_b;
                double t10114 = 1.0/pow(t10113,1.0/3.0);
                double t10115 = c*t10114;
                double t10116 = 1.0/pow(t10113,4.0/3.0);
                double t10117 = sqrt(t10115);
                double t10118 = EcP_3*t10117;
                double t10119 = EcP_4+t10115+t10118;
                double t10120 = c*t10116*(1.0/3.0);
                double t10121 = 1.0/sqrt(t10115);
                double t10122 = EcP_3*c*t10121*t10116*(1.0/6.0);
                double t10123 = t10120+t10122;
                double t10124 = 1.0/c;
                double t10125 = 1.0/t10119;
                double t10126 = c*t10116*t10125*(1.0/3.0);
                double t10127 = 1.0/(t10119*t10119);
                double t10262 = c*t10114*t10123*t10127;
                double t10128 = t10126-t10262;
                double t10129 = pow(t10113,1.0/3.0);
                double t10130 = 1.0/pow(t10113,7.0/3.0);
                double t10131 = EcP_2-t10117;
                double t10132 = 1.0/(t10131*t10131);
                double t10133 = t10131*t10131;
                double t10134 = c*t10130*(4.0/9.0);
                double t10135 = EcP_3*c*t10121*t10130*(2.0/9.0);
                double t10136 = c*c;
                double t10137 = 1.0/pow(t10113,8.0/3.0);
                double t10138 = 1.0/pow(t10115,3.0/2.0);
                double t10265 = EcP_3*t10136*t10137*t10138*(1.0/3.6E1);
                double t10139 = t10134+t10135-t10265;
                double t10140 = t10123*t10123;
                double t10141 = 1.0/(t10119*t10119*t10119);
                double t10142 = t10117*2.0;
                double t10143 = EcP_3+t10142;
                double t10144 = EcP_2*4.0;
                double t10145 = EcP_3*2.0;
                double t10146 = t10144+t10145;
                double t10147 = 1.0/(t10143*t10143);
                double t10148 = EcP_4*4.0;
                double t10149 = EcP_3*EcP_3;
                double t10150 = t10148-t10149;
                double t10151 = t10150*t10147;
                double t10152 = t10151+1.0;
                double t10153 = 1.0/t10152;
                double t10154 = t10123*t10133*t10127;
                double t10155 = c*t10121*t10131*t10116*t10125*(1.0/3.0);
                double t10156 = t10154+t10155;
                double t10157 = 1.0/(t10143*t10143*t10143);
                double t10158 = 1.0/(t10143*t10143*t10143*t10143*t10143);
                double t10159 = 1.0/(t10152*t10152);
                double t10160 = rho_a-rho_b;
                double t10161 = t10160*t10160;
                double t10162 = EcF_4*4.0;
                double t10163 = EcF_3*EcF_3;
                double t10164 = t10162-t10163;
                double t10165 = EcF_2-t10117;
                double t10166 = EcF_3*t10117;
                double t10167 = EcF_4+t10115+t10166;
                double t10168 = 1.0/t10167;
                double t10169 = sqrt(t10164);
                double t10170 = EcF_3+t10142;
                double t10171 = 1.0/t10170;
                double t10172 = t10171*t10169;
                double t10173 = atan(t10172);
                double t10174 = 1.0/sqrt(t10164);
                double t10175 = sqrt(t10150);
                double t10176 = 1.0/t10143;
                double t10177 = t10175*t10176;
                double t10178 = atan(t10177);
                double t10179 = 1.0/sqrt(t10150);
                double t10180 = EcP_2*EcP_2;
                double t10181 = EcP_2*EcP_3;
                double t10182 = EcP_4+t10180+t10181;
                double t10183 = 1.0/t10182;
                double t10184 = Ac_4*4.0;
                double t10185 = Ac_3*Ac_3;
                double t10186 = t10184-t10185;
                double t10187 = Ac_2-t10117;
                double t10188 = Ac_3*t10117;
                double t10189 = Ac_4+t10115+t10188;
                double t10190 = 1.0/t10189;
                double t10191 = sqrt(t10186);
                double t10192 = Ac_3+t10142;
                double t10193 = 1.0/t10192;
                double t10194 = t10191*t10193;
                double t10195 = atan(t10194);
                double t10196 = 1.0/sqrt(t10186);
                double t10197 = c*t10114*t10190;
                double t10198 = log(t10197);
                double t10199 = Ac_3*t10195*t10196*2.0;
                double t10200 = t10187*t10187;
                double t10201 = t10200*t10190;
                double t10202 = log(t10201);
                double t10203 = Ac_2*4.0;
                double t10204 = Ac_3*2.0;
                double t10205 = t10203+t10204;
                double t10206 = t10205*t10195*t10196;
                double t10207 = t10202+t10206;
                double t10208 = Ac_2*Ac_2;
                double t10209 = Ac_2*Ac_3;
                double t10210 = Ac_4+t10208+t10209;
                double t10211 = 1.0/t10210;
                double t10229 = Ac_2*Ac_3*t10211*t10207;
                double t10212 = -t10229+t10198+t10199;
                double t10213 = 1.0/t10113;
                double t10214 = t10213*t10160;
                double t10215 = 1.0/(t10113*t10113);
                double t10216 = t10215*2.0;
                double t10217 = 1.0/(t10113*t10113*t10113);
                double t10218 = t10160*t10217*2.0;
                double t10219 = t10216+t10218;
                double t10220 = t10214+1.0;
                double t10223 = t10160*t10215;
                double t10221 = t10213+t10223;
                double t10222 = -t10214+1.0;
                double t10224 = t10221*t10221;
                double t10225 = 1.0/d2fz0;
                double t10226 = two_13*2.0;
                double t10227 = t10226-2.0;
                double t10228 = 1.0/t10227;
                double t10230 = 1.0/(t10113*t10113*t10113*t10113);
                double t10231 = t10161*t10161;
                double t10232 = 1.0/Ac_1;
                double t10233 = EcF_3*c*t10121*t10116*(1.0/6.0);
                double t10234 = t10120+t10233;
                double t10235 = c*t10116*t10168*(1.0/3.0);
                double t10236 = 1.0/(t10167*t10167);
                double t10286 = c*t10114*t10234*t10236;
                double t10237 = t10235-t10286;
                double t10238 = 1.0/pow(t10113,2.0/3.0);
                double t10239 = EcF_2*EcF_2;
                double t10240 = EcF_2*EcF_3;
                double t10241 = EcF_4+t10240+t10239;
                double t10242 = 1.0/t10241;
                double t10243 = t10165*t10165;
                double t10244 = 1.0/(t10165*t10165);
                double t10245 = EcF_3*c*t10121*t10130*(2.0/9.0);
                double t10246 = t10134+t10245-EcF_3*t10136*t10137*t10138*(1.0/3.6E1);
                double t10247 = t10234*t10234;
                double t10248 = 1.0/(t10167*t10167*t10167);
                double t10249 = EcF_2*4.0;
                double t10250 = EcF_3*2.0;
                double t10251 = t10250+t10249;
                double t10252 = 1.0/(t10170*t10170);
                double t10253 = t10252*t10164;
                double t10254 = t10253+1.0;
                double t10255 = 1.0/t10254;
                double t10256 = t10234*t10243*t10236;
                double t10257 = c*t10121*t10116*t10165*t10168*(1.0/3.0);
                double t10258 = t10256+t10257;
                double t10259 = 1.0/(t10170*t10170*t10170);
                double t10260 = 1.0/(t10170*t10170*t10170*t10170*t10170);
                double t10261 = 1.0/(t10254*t10254);
                double t10263 = t10123*t10124*t10128*t10129;
                double t10264 = c*t10130*t10125*(4.0/9.0);
                double t10266 = c*t10140*t10114*t10141*2.0;
                double t10267 = t10264+t10266-c*t10123*t10116*t10127*(2.0/3.0)-c*t10114*t10127*t10139;
                double t10268 = t10124*t10119*t10129*t10267;
                double t10269 = t10123*t10132*t10156;
                double t10270 = c*t10130*t10125*(1.0/1.8E1);
                double t10271 = t10140*t10141*t10133*2.0;
                double t10272 = t10131*t10125*t10136*t10137*t10138*(1.0/1.8E1);
                double t10273 = c*t10121*t10131*t10123*t10116*t10127*(2.0/3.0);
                double t10274 = t10270+t10271+t10272+t10273-t10133*t10127*t10139-c*t10121*t10130*t10131*t10125*(4.0/9.0);
                double t10275 = c*t10121*t10130*t10153*t10146*t10147*(4.0/9.0);
                double t10276 = 1.0/(t10131*t10131*t10131);
                double t10277 = c*t10121*t10116*t10119*t10156*t10276*(1.0/3.0);
                double t10278 = c*t10130*t10150*t10146*t10158*t10159*(2.0/9.0);
                double t10279 = t10275+t10277+t10269+t10278-t10132*t10119*t10274-c*t10130*t10153*t10146*t10157*(2.0/9.0)-t10153*t10136*t10137*t10146*t10138*t10147*(1.0/1.8E1);
                double t10280 = EcP_2*EcP_3*t10183*t10279;
                double t10281 = EcP_3*c*t10130*t10153*t10157*(4.0/9.0);
                double t10282 = EcP_3*t10153*t10136*t10137*t10138*t10147*(1.0/9.0);
                double t10283 = t10280+t10263+t10281+t10282+t10268-t10124*t10119*t10128*t10238*(1.0/3.0)-EcP_3*c*t10121*t10130*t10153*t10147*(8.0/9.0)-EcP_3*c*t10130*t10150*t10158*t10159*(4.0/9.0);
                double t10284 = EcP_1*t10283;
                double t10285 = 1.0/t10212;
                double t10287 = Ac_3*c*t10121*t10116*(1.0/6.0);
                double t10288 = t10120+t10287;
                double t10289 = 1.0/(t10189*t10189);
                double t10290 = 1.0/(t10192*t10192);
                double t10291 = t10290*t10186;
                double t10292 = t10291+1.0;
                double t10293 = 1.0/t10292;
                double t10294 = c*t10114*t10168;
                double t10295 = log(t10294);
                double t10296 = EcF_3*t10173*t10174*2.0;
                double t10297 = t10243*t10168;
                double t10298 = log(t10297);
                double t10299 = t10251*t10173*t10174;
                double t10300 = t10298+t10299;
                double t10324 = EcF_2*EcF_3*t10300*t10242;
                double t10301 = -t10324+t10295+t10296;
                double t10302 = EcF_1*t10301;
                double t10303 = c*t10114*t10125;
                double t10304 = log(t10303);
                double t10305 = EcP_3*t10178*t10179*2.0;
                double t10306 = t10133*t10125;
                double t10307 = log(t10306);
                double t10308 = t10146*t10178*t10179;
                double t10309 = t10307+t10308;
                double t10325 = EcP_2*EcP_3*t10183*t10309;
                double t10310 = t10304+t10305-t10325;
                double t10326 = EcP_1*t10310;
                double t10311 = t10302-t10326;
                double t10312 = c*t10116*t10190*(1.0/3.0);
                double t10328 = c*t10114*t10288*t10289;
                double t10313 = t10312-t10328;
                double t10314 = t10124*t10313*t10129*t10189;
                double t10315 = t10200*t10288*t10289;
                double t10316 = c*t10121*t10116*t10190*t10187*(1.0/3.0);
                double t10317 = t10315+t10316;
                double t10318 = 1.0/(t10187*t10187);
                double t10319 = t10317*t10318*t10189;
                double t10320 = c*t10121*t10205*t10116*t10290*t10293*(1.0/3.0);
                double t10321 = t10320+t10319;
                double t10322 = Ac_2*Ac_3*t10211*t10321;
                double t10350 = Ac_3*c*t10121*t10116*t10290*t10293*(2.0/3.0);
                double t10323 = t10322+t10314-t10350;
                double t10327 = 1.0/(t10212*t10212);
                double t10329 = Ac_3*c*t10121*t10130*(2.0/9.0);
                double t10374 = Ac_3*t10136*t10137*t10138*(1.0/3.6E1);
                double t10330 = t10134+t10329-t10374;
                double t10331 = t10288*t10288;
                double t10332 = 1.0/(t10189*t10189*t10189);
                double t10333 = 1.0/(t10192*t10192*t10192);
                double t10334 = 1.0/(t10192*t10192*t10192*t10192*t10192);
                double t10335 = 1.0/(t10292*t10292);
                double t10336 = t10124*t10129*t10237*t10167;
                double t10337 = t10244*t10167*t10258;
                double t10338 = c*t10121*t10116*t10251*t10252*t10255*(1.0/3.0);
                double t10339 = t10337+t10338;
                double t10340 = EcF_2*EcF_3*t10242*t10339;
                double t10351 = EcF_3*c*t10121*t10116*t10252*t10255*(2.0/3.0);
                double t10341 = t10340-t10351+t10336;
                double t10342 = EcF_1*t10341;
                double t10343 = t10124*t10119*t10128*t10129;
                double t10344 = t10132*t10119*t10156;
                double t10345 = c*t10121*t10116*t10153*t10146*t10147*(1.0/3.0);
                double t10346 = t10344+t10345;
                double t10347 = EcP_2*EcP_3*t10183*t10346;
                double t10352 = EcP_3*c*t10121*t10116*t10153*t10147*(2.0/3.0);
                double t10348 = t10343-t10352+t10347;
                double t10353 = EcP_1*t10348;
                double t10349 = t10342-t10353;
                double t10354 = d2fz0*t10232*t10285*t10349;
                double t10361 = d2fz0*t10311*t10232*t10323*t10327;
                double t10355 = -t10361+t10354;
                double t10356 = d2fz0*t10311*t10232*t10285;
                double t10357 = t10356-1.0;
                double t10358 = 1.0/(t10113*t10113*t10113*t10113*t10113);
                double t10359 = pow(t10220,1.0/3.0);
                double t10360 = pow(t10222,1.0/3.0);
                double t10362 = t10221*t10360*(4.0/3.0);
                double t10363 = t10230*t10231*t10357;
                double t10364 = t10363+1.0;
                double t10394 = t10221*t10359*(4.0/3.0);
                double t10365 = t10362-t10394;
                double t10366 = pow(t10220,4.0/3.0);
                double t10367 = pow(t10222,4.0/3.0);
                double t10368 = t10366+t10367-2.0;
                double t10369 = t10230*t10231*t10355;
                double t10370 = t10230*t10160*t10161*t10357*4.0;
                double t10371 = t10231*t10357*t10358*4.0;
                double t10372 = t10370+t10371+t10369;
                double t10373 = c*t10130*t10190*(4.0/9.0);
                double t10375 = c*t10114*t10331*t10332*2.0;
                double t10376 = t10373+t10375-c*t10114*t10330*t10289-c*t10116*t10288*t10289*(2.0/3.0);
                double t10377 = t10124*t10129*t10376*t10189;
                double t10378 = t10317*t10318*t10288;
                double t10379 = c*t10130*t10190*(1.0/1.8E1);
                double t10380 = t10200*t10331*t10332*2.0;
                double t10381 = t10136*t10190*t10137*t10138*t10187*(1.0/1.8E1);
                double t10382 = c*t10121*t10116*t10187*t10288*t10289*(2.0/3.0);
                double t10383 = t10380+t10381+t10382+t10379-t10200*t10330*t10289-c*t10121*t10130*t10190*t10187*(4.0/9.0);
                double t10384 = c*t10121*t10130*t10205*t10290*t10293*(4.0/9.0);
                double t10385 = 1.0/(t10187*t10187*t10187);
                double t10386 = c*t10121*t10116*t10317*t10385*t10189*(1.0/3.0);
                double t10387 = c*t10130*t10205*t10334*t10335*t10186*(2.0/9.0);
                double t10388 = t10384+t10386+t10378+t10387-t10318*t10383*t10189-c*t10130*t10205*t10333*t10293*(2.0/9.0)-t10205*t10136*t10137*t10290*t10138*t10293*(1.0/1.8E1);
                double t10389 = Ac_2*Ac_3*t10211*t10388;
                double t10390 = t10124*t10313*t10129*t10288;
                double t10391 = Ac_3*c*t10130*t10333*t10293*(4.0/9.0);
                double t10392 = Ac_3*t10136*t10137*t10290*t10138*t10293*(1.0/9.0);
                double t10393 = t10390+t10391+t10392+t10377+t10389-t10124*t10313*t10238*t10189*(1.0/3.0)-Ac_3*c*t10121*t10130*t10290*t10293*(8.0/9.0)-Ac_3*c*t10130*t10334*t10335*t10186*(4.0/9.0);
                v_rho_b_rho_b[Q] += scale * (EcP_1*t10348*-2.0+t10113*(t10284+Ac_1*t10212*t10225*t10228*t10368*(1.0/(t10113*t10113*t10113*t10113*t10113*t10113)*t10231*t10357*2.0E1+t10230*t10161*t10357*1.2E1+t10231*t10355*t10358*8.0-t10230*t10231*(d2fz0*t10232*t10285*(t10284-EcF_1*(t10124*t10129*t10167*(c*t10130*t10168*(4.0/9.0)-c*t10116*t10234*t10236*(2.0/3.0)-c*t10114*t10236*t10246+c*t10114*t10247*t10248*2.0)+t10124*t10234*t10129*t10237-t10124*t10237*t10238*t10167*(1.0/3.0)+EcF_2*EcF_3*t10242*(-t10244*t10167*(c*t10130*t10168*(1.0/1.8E1)-t10243*t10236*t10246+t10243*t10247*t10248*2.0-c*t10121*t10130*t10165*t10168*(4.0/9.0)+t10136*t10137*t10138*t10165*t10168*(1.0/1.8E1)+c*t10121*t10116*t10234*t10236*t10165*(2.0/3.0))+t10234*t10244*t10258-c*t10130*t10251*t10255*t10259*(2.0/9.0)+c*t10121*t10130*t10251*t10252*t10255*(4.0/9.0)+c*t10130*t10251*t10260*t10261*t10164*(2.0/9.0)-t10251*t10252*t10136*t10137*t10138*t10255*(1.0/1.8E1)+c*t10121*t10116*1.0/(t10165*t10165*t10165)*t10167*t10258*(1.0/3.0))+EcF_3*c*t10130*t10255*t10259*(4.0/9.0)-EcF_3*c*t10121*t10130*t10252*t10255*(8.0/9.0)-EcF_3*c*t10130*t10260*t10261*t10164*(4.0/9.0)+EcF_3*t10252*t10136*t10137*t10138*t10255*(1.0/9.0)))-d2fz0*1.0/(t10212*t10212*t10212)*t10311*t10232*(t10323*t10323)*2.0+d2fz0*t10311*t10232*t10327*t10393+d2fz0*t10232*t10323*t10327*t10349*2.0)+t10230*t10160*t10161*t10355*8.0+t10160*t10161*t10357*t10358*3.2E1)+Ac_1*t10212*t10225*t10228*t10364*(t10360*t10219*(-4.0/3.0)+t10219*t10359*(4.0/3.0)+1.0/pow(t10220,2.0/3.0)*t10224*(4.0/9.0)+1.0/pow(t10222,2.0/3.0)*t10224*(4.0/9.0))-Ac_1*t10212*t10225*t10228*t10372*t10365*2.0-Ac_1*t10323*t10225*t10228*t10364*t10365*2.0+Ac_1*t10323*t10225*t10228*t10372*t10368*2.0+Ac_1*t10225*t10228*t10364*t10393*t10368)+Ac_1*t10212*t10225*t10228*t10364*(t10362-t10394)*2.0-Ac_1*t10212*t10225*t10228*t10372*t10368*2.0-Ac_1*t10323*t10225*t10228*t10364*t10368*2.0);
            }
            
        }
    }
}

}