/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.tools.ant.types;

import org.apache.tools.ant.BuildException;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

/**
 * JUnit testcases for org.apache.tools.ant.EnumeratedAttribute.
 */
public class EnumeratedAttributeTest {

    private static String[] expected = {"a", "b", "c"};

    @Test
    public void testContains() {
        EnumeratedAttribute t1 = new TestNormal();
        for (String value : expected) {
            assertTrue(value + " is in TestNormal",
                    t1.containsValue(value));
            assertTrue(value.toUpperCase() + " is in TestNormal",
                    !t1.containsValue(value.toUpperCase()));
        }
        assertTrue("TestNormal doesn\'t have \"d\" attribute",
               !t1.containsValue("d"));
        assertTrue("TestNull doesn\'t have \"d\" attribute and doesn\'t die",
               !(new TestNull()).containsValue("d"));
    }

    @Test
    public void testFactory() {
        Factory ea = (Factory) EnumeratedAttribute.getInstance(Factory.class, "one");
        assertEquals("Factory did not set the right value.", ea.getValue(), "one");
        try {
            EnumeratedAttribute.getInstance(Factory.class, "illegal");
            fail("Factory should fail when trying to set an illegal value.");
        } catch (BuildException be) {
            // was expected
            //TODO assert exception message
        }
    }

    @Test
    public void testExceptions() {
        EnumeratedAttribute t1 = new TestNormal();
        for (String value : expected) {
            try {
                t1.setValue(value);
            } catch (BuildException be) {
                fail("unexpected exception for value " + value);
            }
        }
        try {
            t1.setValue("d");
            fail("expected exception for value \"d\"");
        } catch (BuildException be) {
         //TODO assert build exception
        }
        try {
            (new TestNull()).setValue("d");
            fail("expected exception for value \"d\" in TestNull");
        } catch (BuildException be) {
            //TODO assert exception message
        }
    }

    public static class TestNormal extends EnumeratedAttribute {
        public String[] getValues() {
            return expected;
        }
    }

    public static class TestNull extends EnumeratedAttribute {
        public String[] getValues() {
            return null;
        }
    }

    public static class Factory extends EnumeratedAttribute {
        public String[] getValues() {
            return new String[] {"one", "two", "three"};
        }
    }

}
