// ---------------------------------------------------------------------------
// - TlsSocket.cpp                                                           -
// - afnix:tls service - tls socket class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2016 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "TlsSocket.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a tls socket by socket and state

  TlsSocket::TlsSocket (TcpSocket* s, TlsState* sta) {
    // detach the stream id
    d_sid = (s == nilp) ? -1 : s->detach ();
    // save the tls state
    Object::iref (p_tlss = sta);
    // bind the associated protocol
    Object::iref (p_tlsp = TlsProto::create (sta));
  }

  // destroy this tls socket

  TlsSocket::~TlsSocket (void) {
    Object::dref (p_tlss);
    Object::dref (p_tlsp);
  }

  // return the class name
  
  String TlsSocket::repr (void) const {
    return "TlsSocket";
  }

  // get the tls state

  TlsState* TlsSocket::getstate (void) const {
    rdlock ();
    try {
      TlsState* result = p_tlss;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the tls protocol

  TlsProto* TlsSocket::getproto (void) const {
    rdlock ();
    try {
      TlsProto* result = p_tlsp;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GETSTATE = zone.intern ("get-state");
  static const long QUARK_GETPROTO = zone.intern ("get-protocol");
 
  // create a new object in a generic way

  Object* TlsSocket::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 2 arguments
    if (argc == 2) {
      Object* obj = argv->get (0);
      // check for a socket
      TcpSocket* s = dynamic_cast<TcpSocket*> (obj);
      if (s == nilp) {
	throw Exception ("type-error", "invalid object as tcp socket",
			 Object::repr (obj));
      }
      // check for a state
      obj = argv->get (1);
      TlsState* sta =  dynamic_cast<TlsState*> (obj);
      if (sta == nilp) {
	throw Exception ("type-error", "invalid object as tls state",
			 Object::repr (obj));
      }
      return new TlsSocket (s, sta);
    }
    // too many arguments
    throw Exception ("argument-error", 
                     "too many argument with tls socket constructor");
  }

  // return true if the given quark is defined

  bool TlsSocket::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? TcpSocket::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark
  
  Object* TlsSocket::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETSTATE) {
	rdlock ();
	try {
	  TlsState* result = getstate ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_GETPROTO) {
	rdlock ();
	try {
	  TlsProto* result = getproto ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }
    // call the tcp socket method
    return TcpSocket::apply (robj, nset, quark, argv);
  }
}
