extern crate clap;

use clap::{Arg, App};

use std::fs::File;
use std::fs::OpenOptions;
use std::fs;
use std::io::ErrorKind;
use std::io::prelude::*;
use std::io::{BufReader, BufWriter};
use std::path::{Path, PathBuf};
use std::slice::Iter;

// NOTE: code below is garbage and should be rewritten

// TODO: header/footer support
// TODO: add notes for examples
// TODO: FAQ
// TODO: support options without examples

#[derive(Clone,PartialEq)]
enum DocMode {
    Cli,
    Gui,
}

struct Data {
    workdir: String,
    mode: DocMode,
    img_before_path: String,
    img_after_path: String,
}

fn main() {
    let m = App::new("docgen")
        .version("0.1.0")
        .arg(Arg::with_name("docdir")
            .long("docdir")
            .help("Sets path to documentation directory")
            .value_name("DIR")
            .required(true))
        .arg(Arg::with_name("for-gui")
            .long("for-gui")
            .help("Generate documentation for GUI")
            .requires_all(&["outdir"]))
        .arg(Arg::with_name("outdir")
            .long("outdir")
            .help("Sets path to working directory")
            .value_name("DIR"))
        .get_matches();

    let workdir = m.value_of("docdir").unwrap();
    let srcdir = Path::new(workdir).join("src").to_str().unwrap().to_owned();

    let mode;
    if m.is_present("for-gui") {
        mode = DocMode::Gui;
    } else {
        mode = DocMode::Cli;
    }

    let outdir;
    if m.is_present("outdir") {
        outdir = m.value_of("outdir").unwrap().to_owned();
    } else {
        outdir = String::new();
    }

    // init folders:
    // doc/images/before
    // doc/images/after
    let imgs_path = Path::new(workdir).join("images");
    let before_dir = Path::new(&imgs_path).join("before");
    let after_dir = Path::new(&imgs_path).join("after");
    if mode == DocMode::Cli {
        create_dir(imgs_path.to_str().unwrap());
        create_dir(before_dir.to_str().unwrap());
        create_dir(after_dir.to_str().unwrap());
    }

    let out_path = Path::new(workdir).join("svgcleaner.adoc");

    if mode == DocMode::Cli {
        let mut out_buf = BufWriter::new(File::create(&out_path).unwrap());
        out_buf.write(b"// This file is autogenerated. Do not edit it!\n\n").unwrap();

        if mode == DocMode::Cli {
            writeln!(out_buf, ":toc:").unwrap();
            writeln!(out_buf, ":toc-title:\n").unwrap();
            writeln!(out_buf, "= List of cleaning options\n").unwrap();
        }
    }

    // process rst files
    let lines = load_lines(Path::new(&srcdir).join("order.txt").to_str().unwrap());
    for line in lines {
        if line.starts_with("-- ") {
            if mode == DocMode::Cli {
                let file = OpenOptions::new().append(true).open(&out_path).unwrap();
                let mut out_buf = BufWriter::new(file);

                let title = line.replace("-- ", "");
                writeln!(out_buf, "== {}\n", title).unwrap();
            }

            continue;
        }

        let path = Path::new(&srcdir).join(&line);
        let path_str = path.to_str().unwrap();

        println!("{:?}", path_str);

        let doc_basename = basename(path_str);
        let data = Data {
            workdir: workdir.to_owned(),
            mode: mode.clone(),
            img_before_path: gen_svg_path(&before_dir, doc_basename),
            img_after_path: gen_svg_path(&after_dir, doc_basename),
        };

        if mode == DocMode::Cli {
            let file = OpenOptions::new().append(true).open(&out_path).unwrap();
            let mut out_buf = BufWriter::new(file);
            prepare_page(path_str, &data, &mut out_buf);
        } else {
            let path = Path::new(&outdir).join(&line);
            let mut out_buf = BufWriter::new(File::create(path).unwrap());

            prepare_page(path_str, &data, &mut out_buf);
        }
    }
}

fn gen_svg_path(dir: &PathBuf, basename: &str) -> String {
    let filename = String::from(basename) + ".svg";
    Path::new(&dir).join(filename).to_str().unwrap().to_owned()
}

fn prepare_page(page_path: &str, data: &Data, out_buf: &mut BufWriter<File>) {
    let lines = load_lines(page_path);
    let mut lines_iter = lines.iter();

    if data.mode == DocMode::Gui {
        // skip title
        lines_iter.next();
        lines_iter.next();
    }

    while let Some(line) = lines_iter.next() {
        if line == "////" {
            if data.mode == DocMode::Cli {
                // write CLI arg before table
                writeln!(out_buf, "CLI argument: `--{}`\n", basename(page_path)).unwrap();
            } else {
                writeln!(out_buf, "{{empty}} +\n").unwrap();
            }

            let table = gen_table(&mut lines_iter, &data);
            out_buf.write(table.as_bytes()).unwrap();
            out_buf.write(b"\n").unwrap();

            lines_iter.next();
        } else {
            out_buf.write(line.as_bytes()).unwrap();
            out_buf.write(b"\n").unwrap();
        }
    }

    out_buf.write(b"\n").unwrap();
}

fn load_lines(path: &str) -> Vec<String> {
    let f = File::open(path).unwrap();
    let f = BufReader::new(f);

    let mut lines = Vec::new();
    for line in f.lines() {
        lines.push(line.unwrap());
    }

    lines
}

#[derive(PartialEq)]
enum Column {
    Before,
    After,
}

fn gen_table(lines: &mut Iter<String>, data: &Data) -> String {
    let mut svg1 = String::new();
    let mut svg2 = String::new();

    let mut column = Column::Before;
    let mut insert_xmlns_xlink = true;

    while let Some(line) = lines.next() {
        if line == "NO_XMLNS_XLINK" {
            insert_xmlns_xlink = false;
            continue;
        } else if line == "SPLIT" {
            column = Column::After;
            continue;
        } else if line == "////" {
            break;
        }

        if column == Column::Before {
            svg1.push_str(line.as_str());
            svg1.push('\n');
        } else {
            svg2.push_str(line.as_str());
            svg2.push('\n');
        }
    }

    svg1.pop();
    svg2.pop();

    let svg1_bytes = gen_svg_file(&svg1, insert_xmlns_xlink, Column::Before, data);
    let svg2_bytes = gen_svg_file(&svg2, insert_xmlns_xlink, Column::After, data);

    fn gen_img_link(path: &str, workdir: &str) -> String {
        let relative = Path::new(path).strip_prefix(workdir).unwrap().to_str().unwrap();
        relative.into()
    }

    let img_link_before = gen_img_link(&data.img_before_path, &data.workdir);
    let img_link_after = gen_img_link(&data.img_after_path, &data.workdir);

    let mut table = String::new();
    table.push_str("|===\n");
    table.push_str(&format!("|Before ({}B) |After ({}B)\n", svg1_bytes, svg2_bytes));
    table.push('\n');
    table.push_str("a|\n");
    table.push_str("[source,xml]\n");
    table.push_str("----\n");
    table.push_str(&svg1);
    table.push('\n');
    table.push_str("----\n");
    table.push('\n');
    table.push_str("a|\n");
    table.push_str("[source,xml]\n");
    table.push_str("----\n");
    table.push_str(&svg2);
    table.push('\n');
    table.push_str("----\n");
    table.push('\n');
    if data.mode == DocMode::Cli {
        table.push_str(&format!("a|image::{}[]\n", img_link_before));
        table.push_str(&format!("a|image::{}[]\n", img_link_after));
    }
    table.push_str("|===");

    table
}

fn basename<'a>(path: &'a str) -> &str {
    Path::new(path).file_stem().unwrap().to_str().unwrap()
}

fn gen_svg_file(content: &str, insert_xmlns_xlink: bool, col: Column, data: &Data) -> usize {
    let svg_attrs = if insert_xmlns_xlink {
        "<svg xmlns=\"http://www.w3.org/2000/svg\" \
                    xmlns:xlink=\"http://www.w3.org/1999/xlink\" \
                    width=\"200\" height=\"100\""
    } else {
        "<svg xmlns=\"http://www.w3.org/2000/svg\" \
                    width=\"200\" height=\"100\""
    };

    let new = content.replace("<svg", svg_attrs);

    let mut f = match col {
        Column::Before => File::create(&data.img_before_path).unwrap(),
        Column::After => File::create(&data.img_after_path).unwrap(),
    };

    f.write(new.as_bytes()).unwrap();

    new.len()
}

fn create_dir(path: &str) {
    match fs::create_dir(path) {
        Err(e) => {
            match e.kind() {
                ErrorKind::AlreadyExists => {},
                _ => {
                    println!("{:?}", e.kind());
                    return;
                },
            }
        },
        Ok(_) => {},
    }
}
