/* -*- mode: C; c-basic-offset: 2; indent-tabs-mode: nil; -*- */
/*
 * Copyright (C) 2009-2011  Tiger Soldier <tigersoldier@gmail.com>
 *
 * This file is part of OSD Lyrics.
 * 
 * OSD Lyrics is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OSD Lyrics is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OSD Lyrics.  If not, see <http://www.gnu.org/licenses/>. 
 */
#ifndef _OL_MUSIC_INFO_H_
#define _OL_MUSIC_INFO_H_
#include <stdlib.h>

/**
 * defines a music's infomation structure
 */
typedef struct _OlMusicInfo OlMusicInfo;
struct _OlMusicInfo
{
  char *title;                 /* The title of the music */
  char *artist;                /* The artist of the music */
  char *album;                 /* The album name of the music */
  int track_number;            /* The track number of the music */
  char *uri;                   /* URI of the music */
};

OlMusicInfo *ol_music_info_new ();

/** 
 * @brief Initialize an OlMusicInfo
 * All the fields of music_info will set to empty without memory free
 * 
 * @param music_info 
 */
void ol_music_info_init (OlMusicInfo *music_info);

/** 
 * @brief Clears an OlMusicInfo
 * All fields will be freed if they are not empty
 * 
 * @param music_info 
 */
void ol_music_info_clear (OlMusicInfo *music_info);
void ol_music_info_copy (OlMusicInfo *dest, const OlMusicInfo *src);

/** 
 * @brief Sets the value of music title
 * 
 * @param music_info An OlMusicInfo
 * @param title The value of title. If not NULL, it will be copied
 *              inside the music_info
 */
void ol_music_info_set_title (OlMusicInfo *music_info,
                              const char *title);
const char *ol_music_info_get_title (const OlMusicInfo *music_info);

/** 
 * @brief Sets the artist of music 
 * 
 * @param music_info An OlMusicInfo
 * @param artist The value of artist. If not NULL, it will be copied
 *               inside the music_info
 */
void ol_music_info_set_artist (OlMusicInfo *music_info,
                               const char *artist);
const char *ol_music_info_get_artist (const OlMusicInfo *music_info);

/** 
 * @brief Sets the name of music album
 * 
 * @param music_info An OlMusicInfo
 * @param album The name of album. If not NULL, it will be copied
 *              inside the music_info
 */
void ol_music_info_set_album (OlMusicInfo *music_info,
                              const char *album);
const char *ol_music_info_get_album (const OlMusicInfo *music_info);

void ol_music_info_set_track_number (OlMusicInfo *music_info,
                                     int track_number);
int ol_music_info_get_track_number (const OlMusicInfo *music_info);

/** 
 * @brief Sets the location of music file
 * 
 * @param music_info An OlMusicInfo
 * @param uri The value of uri. If not NULL, it will be copied
 *            inside the music_info
 */
void ol_music_info_set_uri (OlMusicInfo *music_info,
                            const char *uri);
const char *ol_music_info_get_uri (const OlMusicInfo *music_info);

/** 
 * @brief Check whether two MusicInfos are equal
 * Two MusicInfos are equal if and only if all their fields are equal
 *
 * @param lhs An OlMusicInfo, or NULL
 * @param rhs An OlMusicInfo, or NULL
 * 
 * @return If lhs is equal to rhs, return 1. Otherwise return 0
 */
int ol_music_info_equal (const OlMusicInfo *lhs,
                         const OlMusicInfo *rhs);

/** 
 * @ Free music_info, including its members.
 * 
 * @param music_info 
 */
void ol_music_info_destroy (OlMusicInfo *music_info);

/** 
 * @brief Converts a music info to a string
 * The returned buffer is NUL-terminated
 * @param music_info A MusicInfo
 * @param buffer Buffer of serialzed string, or NULL.
 *               If not NULL, the serialzed string is terminated with NUL.
 * @param count The size of the buffer. 
 * 
 * @return The length of the serialized string, regardless of the size of buffer.
 */
int ol_music_info_serialize (OlMusicInfo *music_info,
                             char *buffer,
                             size_t count);

/** 
 * @brief Converts a string to an OlMusicInfo
 * 
 * @param music_info A MusicInfo
 * @param data The serialized string from an OlMusicInfo
 * 
 * @return 1 if succeeded, or 0 if failed
 */
int ol_music_info_deserialize (OlMusicInfo *music_info,
                               const char *data);
#endif /* _OL_MUSIC_INFO_H_ */
