/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { HookRegistry } from './hookRegistry.js';
import { HookRunner } from './hookRunner.js';
import { HookAggregator } from './hookAggregator.js';
import { HookPlanner } from './hookPlanner.js';
import { HookEventHandler } from './hookEventHandler.js';
import { logs } from '@opentelemetry/api-logs';
import { SERVICE_NAME } from '../telemetry/constants.js';
import { debugLogger } from '../utils/debugLogger.js';
/**
 * Main hook system that coordinates all hook-related functionality
 */
export class HookSystem {
    hookRegistry;
    hookRunner;
    hookAggregator;
    hookPlanner;
    hookEventHandler;
    initialized = false;
    constructor(config) {
        const logger = logs.getLogger(SERVICE_NAME);
        const messageBus = config.getMessageBus();
        // Initialize components
        this.hookRegistry = new HookRegistry(config);
        this.hookRunner = new HookRunner();
        this.hookAggregator = new HookAggregator();
        this.hookPlanner = new HookPlanner(this.hookRegistry);
        this.hookEventHandler = new HookEventHandler(config, logger, this.hookPlanner, this.hookRunner, this.hookAggregator, messageBus);
    }
    /**
     * Initialize the hook system
     */
    async initialize() {
        if (this.initialized) {
            return;
        }
        await this.hookRegistry.initialize();
        this.initialized = true;
        debugLogger.debug('Hook system initialized successfully');
    }
    /**
     * Get the hook event bus for firing events
     */
    getEventHandler() {
        if (!this.initialized) {
            throw new Error('Hook system not initialized');
        }
        return this.hookEventHandler;
    }
    /**
     * Get hook registry for management operations
     */
    getRegistry() {
        return this.hookRegistry;
    }
    /**
     * Enable or disable a hook
     */
    setHookEnabled(hookName, enabled) {
        this.hookRegistry.setHookEnabled(hookName, enabled);
    }
    /**
     * Get all registered hooks for display/management
     */
    getAllHooks() {
        return this.hookRegistry.getAllHooks();
    }
    /**
     * Get hook system status for debugging
     */
    getStatus() {
        const allHooks = this.initialized ? this.hookRegistry.getAllHooks() : [];
        return {
            initialized: this.initialized,
            totalHooks: allHooks.length,
        };
    }
}
//# sourceMappingURL=hookSystem.js.map