"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.keyPair = exports.verify = exports.sign = void 0;
const crypto = __importStar(require("crypto"));
const fs = __importStar(require("node:fs"));
const constants_1 = require("./constants");
const errors_1 = require("./errors");
const extension_metadata_1 = require("./extension-metadata");
const keys_1 = require("./keys");
const sign_1 = require("./sign");
const signature_manifest_1 = require("./signature-manifest");
const verify_1 = require("./verify");
const zip_1 = require("./zip");
/**
 * Sign an extension package. The signature is saved to `extension.sigzip`
 * @param vsixFilePath the path to the `.vsix` file of the extension
 * @param privateKeyFilePath the path to the private key used to sign the extension
 */
const sign = async (vsixFilePath, privateKeyFilePath, options) => {
    const extensionFile = await fs.promises.readFile(vsixFilePath);
    const privateKey = await (0, keys_1.loadPrivateKey)(privateKeyFilePath);
    const outputPath = options?.output ?? `./${constants_1.SIGNED_ARCHIVE_NAME}`;
    const signature = await (0, sign_1.signFile)(extensionFile, privateKey);
    const signatureManifest = await (0, signature_manifest_1.generateManifest)(vsixFilePath);
    const files = [
        { filename: constants_1.SIGNATURE_FILE_NAME, buffer: signature },
        { filename: constants_1.SIGNATURE_MANIFEST_FILE_NAME, buffer: Buffer.from(JSON.stringify(signatureManifest)) },
        // We leave the p7s file empty because VS Code expects it to be present
        // https://github.com/microsoft/vscode/blob/0ead1f80c9e0d6ea0732c40faea3095c6f7f165a/src/vs/platform/extensionManagement/node/extensionDownloader.ts#L157
        { filename: ".signature.p7s", buffer: Buffer.alloc(0) },
    ];
    const zippedSignature = await (0, zip_1.zipBuffers)(files);
    await fs.promises.writeFile(outputPath, zippedSignature);
    console.info(`Signature file created at ${outputPath}`);
};
exports.sign = sign;
/**
 * Verify an extension package against a signature archive
 * @param vsixFilePath The extension file path.
 * @param signatureArchiveFilePath The signature archive file path (`.sigzip`).
 * @param verbose A flag indicating whether or not to capture verbose detail in the event of an error.
 * @throws { ExtensionSignatureVerificationError } An error with a code indicating the validity, integrity, or trust issue
 * found during verification or a more fundamental issue (e.g. a required dependency was not found).
 */
const verify = async (vsixFilePath, signatureArchiveFilePath, verbose = false, options) => {
    if (!fs.existsSync(vsixFilePath)) {
        throw new errors_1.ExtensionSignatureVerificationError("PackageIsInvalidZip", false, "The extension package is not a valid zip file");
    }
    if (!fs.existsSync(signatureArchiveFilePath)) {
        throw new errors_1.ExtensionSignatureVerificationError("SignatureArchiveIsInvalidZip", false, "The signature archive is not a valid zip file");
    }
    verbose && console.info("Reading extension file");
    const extensionFile = await fs.promises.readFile(vsixFilePath);
    verbose && console.info("Getting extension id from extension manifest");
    const extensionMetaFromManifest = await (0, extension_metadata_1.getExtensionMeta)(vsixFilePath);
    if (!extensionMetaFromManifest.id) {
        throw new errors_1.ExtensionSignatureVerificationError("ExtensionManifestIsInvalid", false, "The extension manifest is not valid");
    }
    verbose && console.info(`Got extension metadata for ${extensionMetaFromManifest.id}`);
    verbose && console.info("Loading public key");
    const publicKey = await (0, keys_1.loadPublicKey)(options?.publicKey || (await (0, keys_1.downloadPublicKey)(extensionMetaFromManifest)));
    verbose && console.info("Reading signature archive");
    const signature = await (0, zip_1.extractFileAsBufferUsingStreams)(signatureArchiveFilePath, constants_1.SIGNATURE_FILE_NAME).catch(() => {
        throw new errors_1.ExtensionSignatureVerificationError("SignatureIsMissing", false, "The signature is missing from the signature archive");
    });
    if (options?.verifySignatureManifest) {
        const manifest = JSON.parse((await (0, zip_1.extractFileAsBufferUsingStreams)(signatureArchiveFilePath, constants_1.SIGNATURE_MANIFEST_FILE_NAME).catch(() => {
            throw new errors_1.ExtensionSignatureVerificationError("SignatureManifestIsMissing", false, "The signature manifest is missing from the signature archive");
        })).toString("utf-8"));
        verbose && console.info("Verifying signature manifest");
        const signatureManifestValid = await (0, signature_manifest_1.verifyManifest)(manifest, vsixFilePath);
        if (!signatureManifestValid) {
            throw new errors_1.ExtensionSignatureVerificationError("SignatureManifestIsInvalid", true, "The signature manifest is not valid");
        }
        console.info("Signature manifest is valid");
    }
    verbose && console.info("Verifying signature");
    const signatureValid = await (0, verify_1.verifySignature)(extensionFile, publicKey, signature);
    if (!signatureValid) {
        console.error("Signature is not valid");
        throw new errors_1.ExtensionSignatureVerificationError("SignatureManifestIsInvalid", true, "The signature is not valid");
    }
    console.info("Signature is valid");
    return true;
};
exports.verify = verify;
const keyPair = async (options) => {
    const keyPairOptions = {
        publicKeyEncoding: {
            type: "spki",
            format: "pem",
        },
        privateKeyEncoding: {
            type: "pkcs8",
            format: "pem",
        },
    };
    const pair = crypto.generateKeyPairSync("ed25519", keyPairOptions);
    const pairObj = pair;
    const outputDir = options?.outputDir ?? ".";
    const flag = options?.overwrite ? "w" : "wx";
    const publicKeyPath = `${outputDir}/public.pem`;
    const privateKeyPath = `${outputDir}/private.pem`;
    await fs.promises.writeFile(publicKeyPath, pairObj.publicKey, { flag });
    await fs.promises.writeFile(privateKeyPath, pairObj.privateKey, { flag });
    return { publicKeyPath, privateKeyPath };
};
exports.keyPair = keyPair;//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/bf9252a2fb45be6893dd8870c0bf37e2e1766d61/node_modules/node-ovsx-sign/lib/utils/commands.js.map