﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DeletePublicIpv4PoolRequest : public EC2Request {
 public:
  AWS_EC2_API DeletePublicIpv4PoolRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeletePublicIpv4Pool"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A check for whether you have the required permissions for the action without
   * actually making the request and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DeletePublicIpv4PoolRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the public IPv4 pool you want to delete.</p>
   */
  inline const Aws::String& GetPoolId() const { return m_poolId; }
  inline bool PoolIdHasBeenSet() const { return m_poolIdHasBeenSet; }
  template <typename PoolIdT = Aws::String>
  void SetPoolId(PoolIdT&& value) {
    m_poolIdHasBeenSet = true;
    m_poolId = std::forward<PoolIdT>(value);
  }
  template <typename PoolIdT = Aws::String>
  DeletePublicIpv4PoolRequest& WithPoolId(PoolIdT&& value) {
    SetPoolId(std::forward<PoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Availability Zone (AZ) or Local Zone (LZ) network border group that the
   * resource that the IP address is assigned to is in. Defaults to an AZ network
   * border group. For more information on available Local Zones, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">Local
   * Zone availability</a> in the <i>Amazon EC2 User Guide</i>.</p>
   */
  inline const Aws::String& GetNetworkBorderGroup() const { return m_networkBorderGroup; }
  inline bool NetworkBorderGroupHasBeenSet() const { return m_networkBorderGroupHasBeenSet; }
  template <typename NetworkBorderGroupT = Aws::String>
  void SetNetworkBorderGroup(NetworkBorderGroupT&& value) {
    m_networkBorderGroupHasBeenSet = true;
    m_networkBorderGroup = std::forward<NetworkBorderGroupT>(value);
  }
  template <typename NetworkBorderGroupT = Aws::String>
  DeletePublicIpv4PoolRequest& WithNetworkBorderGroup(NetworkBorderGroupT&& value) {
    SetNetworkBorderGroup(std::forward<NetworkBorderGroupT>(value));
    return *this;
  }
  ///@}
 private:
  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;

  Aws::String m_poolId;
  bool m_poolIdHasBeenSet = false;

  Aws::String m_networkBorderGroup;
  bool m_networkBorderGroupHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
