﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/resiliencehub/ResilienceHubRequest.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>
#include <aws/resiliencehub/model/EksSourceClusterNamespace.h>
#include <aws/resiliencehub/model/TerraformSource.h>

#include <utility>

namespace Aws {
namespace ResilienceHub {
namespace Model {

/**
 */
class DeleteAppInputSourceRequest : public ResilienceHubRequest {
 public:
  AWS_RESILIENCEHUB_API DeleteAppInputSourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteAppInputSource"; }

  AWS_RESILIENCEHUB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the Resilience Hub application. The format for
   * this ARN is:
   * arn:<code>partition</code>:resiliencehub:<code>region</code>:<code>account</code>:app/<code>app-id</code>.
   * For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetAppArn() const { return m_appArn; }
  inline bool AppArnHasBeenSet() const { return m_appArnHasBeenSet; }
  template <typename AppArnT = Aws::String>
  void SetAppArn(AppArnT&& value) {
    m_appArnHasBeenSet = true;
    m_appArn = std::forward<AppArnT>(value);
  }
  template <typename AppArnT = Aws::String>
  DeleteAppInputSourceRequest& WithAppArn(AppArnT&& value) {
    SetAppArn(std::forward<AppArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Used for an idempotency token. A client token is a unique, case-sensitive
   * string of up to 64 ASCII characters. You should not reuse the same client token
   * for other API requests.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  DeleteAppInputSourceRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace on your Amazon Elastic Kubernetes Service cluster that you want
   * to delete from the Resilience Hub application.</p>
   */
  inline const EksSourceClusterNamespace& GetEksSourceClusterNamespace() const { return m_eksSourceClusterNamespace; }
  inline bool EksSourceClusterNamespaceHasBeenSet() const { return m_eksSourceClusterNamespaceHasBeenSet; }
  template <typename EksSourceClusterNamespaceT = EksSourceClusterNamespace>
  void SetEksSourceClusterNamespace(EksSourceClusterNamespaceT&& value) {
    m_eksSourceClusterNamespaceHasBeenSet = true;
    m_eksSourceClusterNamespace = std::forward<EksSourceClusterNamespaceT>(value);
  }
  template <typename EksSourceClusterNamespaceT = EksSourceClusterNamespace>
  DeleteAppInputSourceRequest& WithEksSourceClusterNamespace(EksSourceClusterNamespaceT&& value) {
    SetEksSourceClusterNamespace(std::forward<EksSourceClusterNamespaceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the imported resource you want to remove
   * from the Resilience Hub application. For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetSourceArn() const { return m_sourceArn; }
  inline bool SourceArnHasBeenSet() const { return m_sourceArnHasBeenSet; }
  template <typename SourceArnT = Aws::String>
  void SetSourceArn(SourceArnT&& value) {
    m_sourceArnHasBeenSet = true;
    m_sourceArn = std::forward<SourceArnT>(value);
  }
  template <typename SourceArnT = Aws::String>
  DeleteAppInputSourceRequest& WithSourceArn(SourceArnT&& value) {
    SetSourceArn(std::forward<SourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The imported Terraform s3 state ﬁle you want to remove from the Resilience
   * Hub application.</p>
   */
  inline const TerraformSource& GetTerraformSource() const { return m_terraformSource; }
  inline bool TerraformSourceHasBeenSet() const { return m_terraformSourceHasBeenSet; }
  template <typename TerraformSourceT = TerraformSource>
  void SetTerraformSource(TerraformSourceT&& value) {
    m_terraformSourceHasBeenSet = true;
    m_terraformSource = std::forward<TerraformSourceT>(value);
  }
  template <typename TerraformSourceT = TerraformSource>
  DeleteAppInputSourceRequest& WithTerraformSource(TerraformSourceT&& value) {
    SetTerraformSource(std::forward<TerraformSourceT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_appArn;
  bool m_appArnHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  EksSourceClusterNamespace m_eksSourceClusterNamespace;
  bool m_eksSourceClusterNamespaceHasBeenSet = false;

  Aws::String m_sourceArn;
  bool m_sourceArnHasBeenSet = false;

  TerraformSource m_terraformSource;
  bool m_terraformSourceHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
