﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/neptune-graph/NeptuneGraphRequest.h>
#include <aws/neptune-graph/NeptuneGraph_EXPORTS.h>

#include <utility>

namespace Aws {
namespace NeptuneGraph {
namespace Model {

/**
 */
class ResetGraphRequest : public NeptuneGraphRequest {
 public:
  AWS_NEPTUNEGRAPH_API ResetGraphRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ResetGraph"; }

  AWS_NEPTUNEGRAPH_API Aws::String SerializePayload() const override;

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_NEPTUNEGRAPH_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>ID of the graph to reset.</p>
   */
  inline const Aws::String& GetGraphIdentifier() const { return m_graphIdentifier; }
  inline bool GraphIdentifierHasBeenSet() const { return m_graphIdentifierHasBeenSet; }
  template <typename GraphIdentifierT = Aws::String>
  void SetGraphIdentifier(GraphIdentifierT&& value) {
    m_graphIdentifierHasBeenSet = true;
    m_graphIdentifier = std::forward<GraphIdentifierT>(value);
  }
  template <typename GraphIdentifierT = Aws::String>
  ResetGraphRequest& WithGraphIdentifier(GraphIdentifierT&& value) {
    SetGraphIdentifier(std::forward<GraphIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Determines whether a final graph snapshot is created before the graph data is
   * deleted. If set to <code>true</code>, no graph snapshot is created. If set to
   * <code>false</code>, a graph snapshot is created before the data is deleted.</p>
   */
  inline bool GetSkipSnapshot() const { return m_skipSnapshot; }
  inline bool SkipSnapshotHasBeenSet() const { return m_skipSnapshotHasBeenSet; }
  inline void SetSkipSnapshot(bool value) {
    m_skipSnapshotHasBeenSet = true;
    m_skipSnapshot = value;
  }
  inline ResetGraphRequest& WithSkipSnapshot(bool value) {
    SetSkipSnapshot(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_graphIdentifier;
  bool m_graphIdentifierHasBeenSet = false;

  bool m_skipSnapshot{false};
  bool m_skipSnapshotHasBeenSet = false;
};

}  // namespace Model
}  // namespace NeptuneGraph
}  // namespace Aws
