﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/mediaconvert/MediaConvert_EXPORTS.h>
#include <aws/mediaconvert/model/FrameMetricType.h>
#include <aws/mediaconvert/model/ProresChromaSampling.h>
#include <aws/mediaconvert/model/ProresCodecProfile.h>
#include <aws/mediaconvert/model/ProresFramerateControl.h>
#include <aws/mediaconvert/model/ProresFramerateConversionAlgorithm.h>
#include <aws/mediaconvert/model/ProresInterlaceMode.h>
#include <aws/mediaconvert/model/ProresParControl.h>
#include <aws/mediaconvert/model/ProresScanTypeConversionMode.h>
#include <aws/mediaconvert/model/ProresSlowPal.h>
#include <aws/mediaconvert/model/ProresTelecine.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace MediaConvert {
namespace Model {

/**
 * Required when you set Codec to the value PRORES.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ProresSettings">AWS
 * API Reference</a></p>
 */
class ProresSettings {
 public:
  AWS_MEDIACONVERT_API ProresSettings() = default;
  AWS_MEDIACONVERT_API ProresSettings(Aws::Utils::Json::JsonView jsonValue);
  AWS_MEDIACONVERT_API ProresSettings& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MEDIACONVERT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * This setting applies only to ProRes 4444 and ProRes 4444 XQ outputs that you
   * create from inputs that use 4:4:4 chroma sampling. Set Preserve 4:4:4 sampling
   * to allow outputs to also use 4:4:4 chroma sampling. You must specify a value for
   * this setting when your output codec profile supports 4:4:4 chroma sampling.
   * Related Settings: For Apple ProRes outputs with 4:4:4 chroma sampling: Choose
   * Preserve 4:4:4 sampling. Use when your input has 4:4:4 chroma sampling and your
   * output codec Profile is Apple ProRes 4444 or 4444 XQ. Note that when you choose
   * Preserve 4:4:4 sampling, you cannot include any of the following Preprocessors:
   * Dolby Vision, HDR10+, or Noise reducer.
   */
  inline ProresChromaSampling GetChromaSampling() const { return m_chromaSampling; }
  inline bool ChromaSamplingHasBeenSet() const { return m_chromaSamplingHasBeenSet; }
  inline void SetChromaSampling(ProresChromaSampling value) {
    m_chromaSamplingHasBeenSet = true;
    m_chromaSampling = value;
  }
  inline ProresSettings& WithChromaSampling(ProresChromaSampling value) {
    SetChromaSampling(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Use Profile to specify the type of Apple ProRes codec to use for this output.
   */
  inline ProresCodecProfile GetCodecProfile() const { return m_codecProfile; }
  inline bool CodecProfileHasBeenSet() const { return m_codecProfileHasBeenSet; }
  inline void SetCodecProfile(ProresCodecProfile value) {
    m_codecProfileHasBeenSet = true;
    m_codecProfile = value;
  }
  inline ProresSettings& WithCodecProfile(ProresCodecProfile value) {
    SetCodecProfile(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * If you are using the console, use the Framerate setting to specify the frame
   * rate for this output. If you want to keep the same frame rate as the input
   * video, choose Follow source. If you want to do frame rate conversion, choose a
   * frame rate from the dropdown list or choose Custom. The framerates shown in the
   * dropdown list are decimal approximations of fractions. If you choose Custom,
   * specify your frame rate as a fraction.
   */
  inline ProresFramerateControl GetFramerateControl() const { return m_framerateControl; }
  inline bool FramerateControlHasBeenSet() const { return m_framerateControlHasBeenSet; }
  inline void SetFramerateControl(ProresFramerateControl value) {
    m_framerateControlHasBeenSet = true;
    m_framerateControl = value;
  }
  inline ProresSettings& WithFramerateControl(ProresFramerateControl value) {
    SetFramerateControl(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Choose the method that you want MediaConvert to use when increasing or
   * decreasing your video's frame rate. For numerically simple conversions, such as
   * 60 fps to 30 fps: We recommend that you keep the default value, Drop duplicate.
   * For numerically complex conversions, to avoid stutter: Choose Interpolate. This
   * results in a smooth picture, but might introduce undesirable video artifacts.
   * For complex frame rate conversions, especially if your source video has already
   * been converted from its original cadence: Choose FrameFormer to do
   * motion-compensated interpolation. FrameFormer uses the best conversion method
   * frame by frame. Note that using FrameFormer increases the transcoding time and
   * incurs a significant add-on cost. When you choose FrameFormer, your input video
   * resolution must be at least 128x96. To create an output with the same number of
   * frames as your input: Choose Maintain frame count. When you do, MediaConvert
   * will not drop, interpolate, add, or otherwise change the frame count from your
   * input to your output. Note that since the frame count is maintained, the
   * duration of your output will become shorter at higher frame rates and longer at
   * lower frame rates.
   */
  inline ProresFramerateConversionAlgorithm GetFramerateConversionAlgorithm() const { return m_framerateConversionAlgorithm; }
  inline bool FramerateConversionAlgorithmHasBeenSet() const { return m_framerateConversionAlgorithmHasBeenSet; }
  inline void SetFramerateConversionAlgorithm(ProresFramerateConversionAlgorithm value) {
    m_framerateConversionAlgorithmHasBeenSet = true;
    m_framerateConversionAlgorithm = value;
  }
  inline ProresSettings& WithFramerateConversionAlgorithm(ProresFramerateConversionAlgorithm value) {
    SetFramerateConversionAlgorithm(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * When you use the API for transcode jobs that use frame rate conversion, specify
   * the frame rate as a fraction. For example, 24000 / 1001 = 23.976 fps. Use
   * FramerateDenominator to specify the denominator of this fraction. In this
   * example, use 1001 for the value of FramerateDenominator. When you use the
   * console for transcode jobs that use frame rate conversion, provide the value as
   * a decimal number for Framerate. In this example, specify 23.976.
   */
  inline int GetFramerateDenominator() const { return m_framerateDenominator; }
  inline bool FramerateDenominatorHasBeenSet() const { return m_framerateDenominatorHasBeenSet; }
  inline void SetFramerateDenominator(int value) {
    m_framerateDenominatorHasBeenSet = true;
    m_framerateDenominator = value;
  }
  inline ProresSettings& WithFramerateDenominator(int value) {
    SetFramerateDenominator(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * When you use the API for transcode jobs that use frame rate conversion, specify
   * the frame rate as a fraction. For example, 24000 / 1001 = 23.976 fps. Use
   * FramerateNumerator to specify the numerator of this fraction. In this example,
   * use 24000 for the value of FramerateNumerator. When you use the console for
   * transcode jobs that use frame rate conversion, provide the value as a decimal
   * number for Framerate. In this example, specify 23.976.
   */
  inline int GetFramerateNumerator() const { return m_framerateNumerator; }
  inline bool FramerateNumeratorHasBeenSet() const { return m_framerateNumeratorHasBeenSet; }
  inline void SetFramerateNumerator(int value) {
    m_framerateNumeratorHasBeenSet = true;
    m_framerateNumerator = value;
  }
  inline ProresSettings& WithFramerateNumerator(int value) {
    SetFramerateNumerator(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Choose the scan line type for the output. Keep the default value, Progressive to
   * create a progressive output, regardless of the scan type of your input. Use Top
   * field first or Bottom field first to create an output that's interlaced with the
   * same field polarity throughout. Use Follow, default top or Follow, default
   * bottom to produce outputs with the same field polarity as the source. For jobs
   * that have multiple inputs, the output field polarity might change over the
   * course of the output. Follow behavior depends on the input scan type. If the
   * source is interlaced, the output will be interlaced with the same polarity as
   * the source. If the source is progressive, the output will be interlaced with top
   * field bottom field first, depending on which of the Follow options you choose.
   */
  inline ProresInterlaceMode GetInterlaceMode() const { return m_interlaceMode; }
  inline bool InterlaceModeHasBeenSet() const { return m_interlaceModeHasBeenSet; }
  inline void SetInterlaceMode(ProresInterlaceMode value) {
    m_interlaceModeHasBeenSet = true;
    m_interlaceMode = value;
  }
  inline ProresSettings& WithInterlaceMode(ProresInterlaceMode value) {
    SetInterlaceMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Optional. Specify how the service determines the pixel aspect ratio (PAR) for
   * this output. The default behavior, Follow source, uses the PAR from your input
   * video for your output. To specify a different PAR, choose any value other than
   * Follow source. When you choose SPECIFIED for this setting, you must also specify
   * values for the parNumerator and parDenominator settings.
   */
  inline ProresParControl GetParControl() const { return m_parControl; }
  inline bool ParControlHasBeenSet() const { return m_parControlHasBeenSet; }
  inline void SetParControl(ProresParControl value) {
    m_parControlHasBeenSet = true;
    m_parControl = value;
  }
  inline ProresSettings& WithParControl(ProresParControl value) {
    SetParControl(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Required when you set Pixel aspect ratio to SPECIFIED. On the console, this
   * corresponds to any value other than Follow source. When you specify an output
   * pixel aspect ratio (PAR) that is different from your input video PAR, provide
   * your output PAR as a ratio. For example, for D1/DV NTSC widescreen, you would
   * specify the ratio 40:33. In this example, the value for parDenominator is 33.
   */
  inline int GetParDenominator() const { return m_parDenominator; }
  inline bool ParDenominatorHasBeenSet() const { return m_parDenominatorHasBeenSet; }
  inline void SetParDenominator(int value) {
    m_parDenominatorHasBeenSet = true;
    m_parDenominator = value;
  }
  inline ProresSettings& WithParDenominator(int value) {
    SetParDenominator(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Required when you set Pixel aspect ratio to SPECIFIED. On the console, this
   * corresponds to any value other than Follow source. When you specify an output
   * pixel aspect ratio (PAR) that is different from your input video PAR, provide
   * your output PAR as a ratio. For example, for D1/DV NTSC widescreen, you would
   * specify the ratio 40:33. In this example, the value for parNumerator is 40.
   */
  inline int GetParNumerator() const { return m_parNumerator; }
  inline bool ParNumeratorHasBeenSet() const { return m_parNumeratorHasBeenSet; }
  inline void SetParNumerator(int value) {
    m_parNumeratorHasBeenSet = true;
    m_parNumerator = value;
  }
  inline ProresSettings& WithParNumerator(int value) {
    SetParNumerator(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Optionally choose one or more per frame metric reports to generate along with
   * your output. You can use these metrics to analyze your video output according to
   * one or more commonly used image quality metrics. You can specify per frame
   * metrics for output groups or for individual outputs. When you do, MediaConvert
   * writes a CSV (Comma-Separated Values) file to your S3 output destination, named
   * after the output name and metric type. For example: videofile_PSNR.csv Jobs that
   * generate per frame metrics will take longer to complete, depending on the
   * resolution and complexity of your output. For example, some 4K jobs might take
   * up to twice as long to complete. Note that when analyzing the video quality of
   * your output, or when comparing the video quality of multiple different outputs,
   * we generally also recommend a detailed visual review in a controlled
   * environment. You can choose from the following per frame metrics: * PSNR: Peak
   * Signal-to-Noise Ratio * SSIM: Structural Similarity Index Measure * MS_SSIM:
   * Multi-Scale Similarity Index Measure * PSNR_HVS: Peak Signal-to-Noise Ratio,
   * Human Visual System * VMAF: Video Multi-Method Assessment Fusion * QVBR:
   * Quality-Defined Variable Bitrate. This option is only available when your output
   * uses the QVBR rate control mode. * SHOT_CHANGE: Shot Changes
   */
  inline const Aws::Vector<FrameMetricType>& GetPerFrameMetrics() const { return m_perFrameMetrics; }
  inline bool PerFrameMetricsHasBeenSet() const { return m_perFrameMetricsHasBeenSet; }
  template <typename PerFrameMetricsT = Aws::Vector<FrameMetricType>>
  void SetPerFrameMetrics(PerFrameMetricsT&& value) {
    m_perFrameMetricsHasBeenSet = true;
    m_perFrameMetrics = std::forward<PerFrameMetricsT>(value);
  }
  template <typename PerFrameMetricsT = Aws::Vector<FrameMetricType>>
  ProresSettings& WithPerFrameMetrics(PerFrameMetricsT&& value) {
    SetPerFrameMetrics(std::forward<PerFrameMetricsT>(value));
    return *this;
  }
  inline ProresSettings& AddPerFrameMetrics(FrameMetricType value) {
    m_perFrameMetricsHasBeenSet = true;
    m_perFrameMetrics.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Use this setting for interlaced outputs, when your output frame rate is half of
   * your input frame rate. In this situation, choose Optimized interlacing to create
   * a better quality interlaced output. In this case, each progressive frame from
   * the input corresponds to an interlaced field in the output. Keep the default
   * value, Basic interlacing, for all other output frame rates. With basic
   * interlacing, MediaConvert performs any frame rate conversion first and then
   * interlaces the frames. When you choose Optimized interlacing and you set your
   * output frame rate to a value that isn't suitable for optimized interlacing,
   * MediaConvert automatically falls back to basic interlacing. Required settings:
   * To use optimized interlacing, you must set Telecine to None or Soft. You can't
   * use optimized interlacing for hard telecine outputs. You must also set Interlace
   * mode to a value other than Progressive.
   */
  inline ProresScanTypeConversionMode GetScanTypeConversionMode() const { return m_scanTypeConversionMode; }
  inline bool ScanTypeConversionModeHasBeenSet() const { return m_scanTypeConversionModeHasBeenSet; }
  inline void SetScanTypeConversionMode(ProresScanTypeConversionMode value) {
    m_scanTypeConversionModeHasBeenSet = true;
    m_scanTypeConversionMode = value;
  }
  inline ProresSettings& WithScanTypeConversionMode(ProresScanTypeConversionMode value) {
    SetScanTypeConversionMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * Ignore this setting unless your input frame rate is 23.976 or 24 frames per
   * second (fps). Enable slow PAL to create a 25 fps output. When you enable slow
   * PAL, MediaConvert relabels the video frames to 25 fps and resamples your audio
   * to keep it synchronized with the video. Note that enabling this setting will
   * slightly reduce the duration of your video. Required settings: You must also set
   * Framerate to 25.
   */
  inline ProresSlowPal GetSlowPal() const { return m_slowPal; }
  inline bool SlowPalHasBeenSet() const { return m_slowPalHasBeenSet; }
  inline void SetSlowPal(ProresSlowPal value) {
    m_slowPalHasBeenSet = true;
    m_slowPal = value;
  }
  inline ProresSettings& WithSlowPal(ProresSlowPal value) {
    SetSlowPal(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * When you do frame rate conversion from 23.976 frames per second (fps) to 29.97
   * fps, and your output scan type is interlaced, you can optionally enable hard
   * telecine to create a smoother picture. When you keep the default value, None,
   * MediaConvert does a standard frame rate conversion to 29.97 without doing
   * anything with the field polarity to create a smoother picture.
   */
  inline ProresTelecine GetTelecine() const { return m_telecine; }
  inline bool TelecineHasBeenSet() const { return m_telecineHasBeenSet; }
  inline void SetTelecine(ProresTelecine value) {
    m_telecineHasBeenSet = true;
    m_telecine = value;
  }
  inline ProresSettings& WithTelecine(ProresTelecine value) {
    SetTelecine(value);
    return *this;
  }
  ///@}
 private:
  ProresChromaSampling m_chromaSampling{ProresChromaSampling::NOT_SET};
  bool m_chromaSamplingHasBeenSet = false;

  ProresCodecProfile m_codecProfile{ProresCodecProfile::NOT_SET};
  bool m_codecProfileHasBeenSet = false;

  ProresFramerateControl m_framerateControl{ProresFramerateControl::NOT_SET};
  bool m_framerateControlHasBeenSet = false;

  ProresFramerateConversionAlgorithm m_framerateConversionAlgorithm{ProresFramerateConversionAlgorithm::NOT_SET};
  bool m_framerateConversionAlgorithmHasBeenSet = false;

  int m_framerateDenominator{0};
  bool m_framerateDenominatorHasBeenSet = false;

  int m_framerateNumerator{0};
  bool m_framerateNumeratorHasBeenSet = false;

  ProresInterlaceMode m_interlaceMode{ProresInterlaceMode::NOT_SET};
  bool m_interlaceModeHasBeenSet = false;

  ProresParControl m_parControl{ProresParControl::NOT_SET};
  bool m_parControlHasBeenSet = false;

  int m_parDenominator{0};
  bool m_parDenominatorHasBeenSet = false;

  int m_parNumerator{0};
  bool m_parNumeratorHasBeenSet = false;

  Aws::Vector<FrameMetricType> m_perFrameMetrics;
  bool m_perFrameMetricsHasBeenSet = false;

  ProresScanTypeConversionMode m_scanTypeConversionMode{ProresScanTypeConversionMode::NOT_SET};
  bool m_scanTypeConversionModeHasBeenSet = false;

  ProresSlowPal m_slowPal{ProresSlowPal::NOT_SET};
  bool m_slowPalHasBeenSet = false;

  ProresTelecine m_telecine{ProresTelecine::NOT_SET};
  bool m_telecineHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaConvert
}  // namespace Aws
