﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/logs/CloudWatchLogs_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CloudWatchLogs {
namespace Model {

/**
 * <p>The entity associated with the log events in a <code>PutLogEvents</code>
 * call.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/Entity">AWS API
 * Reference</a></p>
 */
class Entity {
 public:
  AWS_CLOUDWATCHLOGS_API Entity() = default;
  AWS_CLOUDWATCHLOGS_API Entity(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API Entity& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The attributes of the entity which identify the specific entity, as a list of
   * key-value pairs. Entities with the same <code>keyAttributes</code> are
   * considered to be the same entity.</p> <p>There are five allowed attributes (key
   * names): <code>Type</code>, <code>ResourceType</code>, <code>Identifier</code>
   * <code>Name</code>, and <code>Environment</code>.</p> <p>For details about how to
   * use the key attributes, see <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/adding-your-own-related-telemetry.html">How
   * to add related information to telemetry</a> in the <i>CloudWatch User
   * Guide</i>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetKeyAttributes() const { return m_keyAttributes; }
  inline bool KeyAttributesHasBeenSet() const { return m_keyAttributesHasBeenSet; }
  template <typename KeyAttributesT = Aws::Map<Aws::String, Aws::String>>
  void SetKeyAttributes(KeyAttributesT&& value) {
    m_keyAttributesHasBeenSet = true;
    m_keyAttributes = std::forward<KeyAttributesT>(value);
  }
  template <typename KeyAttributesT = Aws::Map<Aws::String, Aws::String>>
  Entity& WithKeyAttributes(KeyAttributesT&& value) {
    SetKeyAttributes(std::forward<KeyAttributesT>(value));
    return *this;
  }
  template <typename KeyAttributesKeyT = Aws::String, typename KeyAttributesValueT = Aws::String>
  Entity& AddKeyAttributes(KeyAttributesKeyT&& key, KeyAttributesValueT&& value) {
    m_keyAttributesHasBeenSet = true;
    m_keyAttributes.emplace(std::forward<KeyAttributesKeyT>(key), std::forward<KeyAttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Additional attributes of the entity that are not used to specify the identity
   * of the entity. A list of key-value pairs.</p> <p>For details about how to use
   * the attributes, see <a
   * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/adding-your-own-related-telemetry.html">How
   * to add related information to telemetry</a> in the <i>CloudWatch User
   * Guide</i>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAttributes() const { return m_attributes; }
  inline bool AttributesHasBeenSet() const { return m_attributesHasBeenSet; }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  void SetAttributes(AttributesT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes = std::forward<AttributesT>(value);
  }
  template <typename AttributesT = Aws::Map<Aws::String, Aws::String>>
  Entity& WithAttributes(AttributesT&& value) {
    SetAttributes(std::forward<AttributesT>(value));
    return *this;
  }
  template <typename AttributesKeyT = Aws::String, typename AttributesValueT = Aws::String>
  Entity& AddAttributes(AttributesKeyT&& key, AttributesValueT&& value) {
    m_attributesHasBeenSet = true;
    m_attributes.emplace(std::forward<AttributesKeyT>(key), std::forward<AttributesValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Map<Aws::String, Aws::String> m_keyAttributes;
  bool m_keyAttributesHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_attributes;
  bool m_attributesHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchLogs
}  // namespace Aws
