﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/kendra/KendraRequest.h>
#include <aws/kendra/Kendra_EXPORTS.h>
#include <aws/kendra/model/FaqFileFormat.h>
#include <aws/kendra/model/S3Path.h>
#include <aws/kendra/model/Tag.h>

#include <utility>

namespace Aws {
namespace kendra {
namespace Model {

/**
 */
class CreateFaqRequest : public KendraRequest {
 public:
  AWS_KENDRA_API CreateFaqRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateFaq"; }

  AWS_KENDRA_API Aws::String SerializePayload() const override;

  AWS_KENDRA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The identifier of the index for the FAQ.</p>
   */
  inline const Aws::String& GetIndexId() const { return m_indexId; }
  inline bool IndexIdHasBeenSet() const { return m_indexIdHasBeenSet; }
  template <typename IndexIdT = Aws::String>
  void SetIndexId(IndexIdT&& value) {
    m_indexIdHasBeenSet = true;
    m_indexId = std::forward<IndexIdT>(value);
  }
  template <typename IndexIdT = Aws::String>
  CreateFaqRequest& WithIndexId(IndexIdT&& value) {
    SetIndexId(std::forward<IndexIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A name for the FAQ.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateFaqRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the FAQ.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateFaqRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The path to the FAQ file in S3.</p>
   */
  inline const S3Path& GetS3Path() const { return m_s3Path; }
  inline bool S3PathHasBeenSet() const { return m_s3PathHasBeenSet; }
  template <typename S3PathT = S3Path>
  void SetS3Path(S3PathT&& value) {
    m_s3PathHasBeenSet = true;
    m_s3Path = std::forward<S3PathT>(value);
  }
  template <typename S3PathT = S3Path>
  CreateFaqRequest& WithS3Path(S3PathT&& value) {
    SetS3Path(std::forward<S3PathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an IAM role with permission to access the
   * S3 bucket that contains the FAQ file. For more information, see <a
   * href="https://docs.aws.amazon.com/kendra/latest/dg/iam-roles.html">IAM access
   * roles for Amazon Kendra</a>.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateFaqRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of key-value pairs that identify the FAQ. You can use the tags to
   * identify and organize your resources and to control access to resources.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateFaqRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateFaqRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The format of the FAQ input file. You can choose between a basic CSV format,
   * a CSV format that includes customs attributes in a header, and a JSON format
   * that includes custom attributes.</p> <p>The default format is CSV.</p> <p>The
   * format must match the format of the file stored in the S3 bucket identified in
   * the <code>S3Path</code> parameter.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/kendra/latest/dg/in-creating-faq.html">Adding
   * questions and answers</a>.</p>
   */
  inline FaqFileFormat GetFileFormat() const { return m_fileFormat; }
  inline bool FileFormatHasBeenSet() const { return m_fileFormatHasBeenSet; }
  inline void SetFileFormat(FaqFileFormat value) {
    m_fileFormatHasBeenSet = true;
    m_fileFormat = value;
  }
  inline CreateFaqRequest& WithFileFormat(FaqFileFormat value) {
    SetFileFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that you provide to identify the request to create a FAQ. Multiple
   * calls to the <code>CreateFaqRequest</code> API with the same client token will
   * create only one FAQ. </p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateFaqRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The code for a language. This allows you to support a language for the FAQ
   * document. English is supported by default. For more information on supported
   * languages, including their codes, see <a
   * href="https://docs.aws.amazon.com/kendra/latest/dg/in-adding-languages.html">Adding
   * documents in languages other than English</a>.</p>
   */
  inline const Aws::String& GetLanguageCode() const { return m_languageCode; }
  inline bool LanguageCodeHasBeenSet() const { return m_languageCodeHasBeenSet; }
  template <typename LanguageCodeT = Aws::String>
  void SetLanguageCode(LanguageCodeT&& value) {
    m_languageCodeHasBeenSet = true;
    m_languageCode = std::forward<LanguageCodeT>(value);
  }
  template <typename LanguageCodeT = Aws::String>
  CreateFaqRequest& WithLanguageCode(LanguageCodeT&& value) {
    SetLanguageCode(std::forward<LanguageCodeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_indexId;
  bool m_indexIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  S3Path m_s3Path;
  bool m_s3PathHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  FaqFileFormat m_fileFormat{FaqFileFormat::NOT_SET};
  bool m_fileFormatHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_languageCode;
  bool m_languageCodeHasBeenSet = false;
};

}  // namespace Model
}  // namespace kendra
}  // namespace Aws
